﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

namespace nn {
namespace socket {

/* BSD style release date */
const uint32_t Bpf_Release = 199606;

/*
* Alignment macros. Bpf_WordAlign rounds up to the next
* even multiple of Bpf_Alignment.
*/
const size_t Bpf_Alignment = sizeof(long);

inline uint32_t Bpf_WordAlign(uint32_t x) NN_NOEXCEPT
{
    return (((x)+(Bpf_Alignment - 1))&~(Bpf_Alignment - 1));
}

typedef int32_t  BpfInt32;
typedef uint32_t BpfUInt32;
typedef int64_t  BpfInt64;
typedef uint64_t BpfUInt64;

const uint32_t Bpf_MaxInsns = 512;
const uint32_t Bpf_MaxBufSize = 0x80000;
const uint32_t Bpf_MinBufSize = 32;

enum class BpfCode : uint16_t
{
    Bpf_EthCook     = 0x00,
    Bpf_WholePacket = 0xff,
    Bpf_Ld          = 0x00,
    Bpf_Ldx         = 0x01,
    Bpf_St          = 0x02,
    Bpf_Stx         = 0x03,
    Bpf_Alu         = 0x04,
    Bpf_Jmp         = 0x05,
    Bpf_Ret         = 0x06,
    Bpf_Misc        = 0x07,
    Bpf_W           = 0x00,
    Bpf_H           = 0x08,
    Bpf_B           = 0x10,
    Bpf_Imm         = 0x00,
    Bpf_Abs         = 0x20,
    Bpf_Ind         = 0x40,
    Bpf_Mem         = 0x60,
    Bpf_Len         = 0x80,
    Bpf_Msh         = 0xa0,
    Bpf_Add         = 0x00,
    Bpf_Sub         = 0x10,
    Bpf_Mul         = 0x20,
    Bpf_Div         = 0x30,
    Bpf_Or          = 0x40,
    Bpf_And         = 0x50,
    Bpf_Lsh         = 0x60,
    Bpf_Rsh         = 0x70,
    Bpf_Neg         = 0x80,
    Bpf_Ja          = 0x00,
    Bpf_Jeq         = 0x10,
    Bpf_Jgt         = 0x20,
    Bpf_Jge         = 0x30,
    Bpf_Jset        = 0x40,
    Bpf_K           = 0x00,
    Bpf_X           = 0x08,
    Bpf_A           = 0x10,
    Bpf_Tax         = 0x00,
    Bpf_Txa         = 0x80,
};

/*
* The instruction data structure.
*/
struct BpfInsn
{
    BpfCode     code;
    uint8_t     jt;
    uint8_t     jf;
    BpfUInt32   k;
};

/*
*  Structure for BiocSetF.
*/
struct BpfProgram
{
    uint32_t    bf_len;
    BpfInsn     bf_insns[Bpf_MaxInsns]; /* bf_len number of elements */
};

/*
* Struct returned by BiocGStats.
*/
struct BpfStat
{
    uint32_t bs_recv;        /* number of packets received */
    uint32_t bs_drop;        /* number of packets dropped */
};

/*
* Struct return by BiocVersion.  This represents the version number of
* the filter language described by the instruction encodings below.
* bpf understands a program iff kernel_major == filter_major &&
* kernel_minor >= filter_minor, that is, if the value returned by the
* running kernel has the same major number and a minor number equal
* equal to or less than the filter being downloaded.  Otherwise, the
* results are undefined, meaning an error may be returned or packets
* may be accepted haphazardly.
* It has nothing to do with the source code version.
*/
struct BpfVersion
{
    uint16_t bv_major;
    uint16_t bv_minor;
};

/* Current version number of filter architecture. */
const uint16_t Bpf_Major_Version = 1;
const uint16_t Bpf_Minor_Version = 1;

/*
* Historically, BPF has supported a single buffering model, first using mbuf
* clusters in kernel, and later using malloc(9) buffers in kernel.  We now
* support multiple buffering modes, which may be queried and set using
* BiocGetBufMode and BiocSetBufMode.  So as to avoid handling the complexity
* of changing modes while sniffing packets, the mode becomes fixed once an
* interface has been attached to the BPF descriptor.
*/
enum class BpfBufMode : uint32_t
{
    Bpf_BufMode_Buffer = 1,     /* Kernel buffers with read(). */
    Bpf_BufMode_ZBuf   = 2      /* Zero-copy buffers. */
};

/*-
* Struct used by BiocSetZBuf, BiocRotZuf: describes up to two zero-copy
* buffer as used by BPF.
*/
struct BpfZBuf
{
    void        *bz_bufa;       /* Location of 'a' zero-copy buffer. */
    void        *bz_bufb;       /* Location of 'b' zero-copy buffer. */
    size_t       bz_buflen;     /* Size of zero-copy buffers. */
};

/* Packet directions */
enum class BpfDirection : uint32_t
{
    Bpf_D_In,       /* See incoming packets */
    Bpf_D_InOut,    /* See incoming and outgoing packets */
    Bpf_D_Out       /* See outgoing packets */
};

/* Time stamping functions */
enum class BpfTimestamp : uint16_t
{
    Bpf_T_MicroTime                 = 0x0000,
    Bpf_T_NanoTime                  = 0x0001,
    Bpf_T_BinTime                   = 0x0002,
    Bpf_T_None                      = 0x0003,
    Bpf_T_Format_Mask               = 0x0003,
    Bpf_T_Normal                    = 0x0000,
    Bpf_T_Fast                      = 0x0100,
    Bpf_T_Monotonic                 = 0x0200,
    Bpf_T_Monotonic_Fast            = (Bpf_T_Fast | Bpf_T_Monotonic),
    Bpf_T_Flag_Mask                 = 0x0300,

    Bpf_T_MicroTime_Fast            = (Bpf_T_MicroTime | Bpf_T_Fast),
    Bpf_T_NanoTime_Fast             = (Bpf_T_NanoTime | Bpf_T_Fast),
    Bpf_T_BinTime_Fast              = (Bpf_T_BinTime | Bpf_T_Fast),
    Bpf_T_MicroTime_Monotonic       = (Bpf_T_MicroTime | Bpf_T_Monotonic),
    Bpf_T_NanoTime_Monotonic        = (Bpf_T_NanoTime | Bpf_T_Monotonic),
    Bpf_T_BinTime_Monotonic         = (Bpf_T_BinTime | Bpf_T_Monotonic),
    Bpf_T_MicroTime_Monotonic_Fast  = (Bpf_T_MicroTime | Bpf_T_Monotonic_Fast),
    Bpf_T_NanoTime_Monotonic_Fast   = (Bpf_T_NanoTime | Bpf_T_Monotonic_Fast),
    Bpf_T_BinTime_Monotonic_Fast    = (Bpf_T_BinTime | Bpf_T_Monotonic_Fast)
};

inline BpfTimestamp operator | (BpfTimestamp a, BpfTimestamp b) NN_NOEXCEPT
{
    return static_cast<BpfTimestamp>(static_cast<int>(a) | static_cast<int>(b));
}

inline BpfTimestamp operator |= (BpfTimestamp &a, BpfTimestamp b) NN_NOEXCEPT
{
    return a = a | b;
}

inline BpfTimestamp operator & (BpfTimestamp a, BpfTimestamp b) NN_NOEXCEPT
{
    return static_cast<BpfTimestamp>(static_cast<int>(a) & static_cast<int>(b));
}

inline BpfTimestamp operator &= (BpfTimestamp &a, BpfTimestamp b) NN_NOEXCEPT
{
    return a = a & b;
}

inline BpfTimestamp operator ^ (BpfTimestamp a, BpfTimestamp b) NN_NOEXCEPT
{
    return static_cast<BpfTimestamp>(static_cast<int>(a) ^ static_cast<int>(b));
}

inline BpfTimestamp operator ^= (BpfTimestamp &a, BpfTimestamp b) NN_NOEXCEPT
{
    return a = a ^ b;
}

inline BpfTimestamp operator ~ (BpfTimestamp a) NN_NOEXCEPT
{
    return static_cast<BpfTimestamp>(~static_cast<int>(a));
}

inline BpfTimestamp Bpf_T_Format(BpfTimestamp t)
{
    return ((t) & BpfTimestamp::Bpf_T_Format_Mask);
}

inline BpfTimestamp Bpf_T_Flag(BpfTimestamp t) {
    return ((t) & BpfTimestamp::Bpf_T_Flag_Mask);
}

inline bool Bpf_T_Valid(BpfTimestamp t)
{
    return ((t) == BpfTimestamp::Bpf_T_None || (Bpf_T_Format(t) != BpfTimestamp::Bpf_T_None &&
           ((t) & ~(BpfTimestamp::Bpf_T_Format_Mask | BpfTimestamp::Bpf_T_Flag_Mask)) == static_cast<BpfTimestamp>(0)));
}

/*
* Structure prepended to each packet.
*/
struct BpfTs
{
    BpfInt64    bt_sec;         /* seconds */
    BpfUInt64   bt_frac;        /* fraction */
};
struct BpfXHdr
{
    BpfTs       bh_tstamp;      /* time stamp */
    BpfUInt32 bh_caplen;      /* length of captured portion */
    BpfUInt32 bh_datalen;     /* original length of packet */
    uint16_t    bh_hdrlen;      /* length of bpf header (this struct
                                   plus alignment padding) */
};
/* Obsolete */
struct BpfHdr
{
    TimeVal     bh_tstamp;      /* time stamp */
    BpfUInt32   bh_caplen;      /* length of captured portion */
    BpfUInt32   bh_datalen;     /* original length of packet */
    uint16_t    bh_hdrlen;      /* length of bpf header (this struct
                                   plus alignment padding) */
};

/*
* When using zero-copy BPF buffers, a shared memory header is present
* allowing the kernel BPF implementation and user process to synchronize
* without using system calls.  This structure defines that header.  When
* accessing these fields, appropriate atomic operation and memory barriers
* are required in order not to see stale or out-of-order data; see bpf(4)
* for reference code to access these fields from userspace.
*
* The layout of this structure is critical, and must not be changed; if must
* fit in a single page on all architectures.
*/
struct BpfZBufHeader
{
    volatile uint32_t    bzh_kernel_gen;    /* Kernel generation number. */
    volatile uint32_t    bzh_kernel_len;    /* Length of data in the buffer. */
    volatile uint32_t    bzh_user_gen;      /* User generation number. */
    uint32_t _bzh_pad[5];
};

/*
* Data-link level type codes.
*/
enum class DataLinkType : uint32_t
{
    Dlt_Null                    = 0,    /* BSD loopback encapsulation */
    Dlt_En10Mb                  = 1,    /* Ethernet (10Mb) */
    Dlt_En3Mb                   = 2,    /* Experimental Ethernet (3Mb) */
    Dlt_Ax25                    = 3,    /* Amateur Radio AX.25 */
    Dlt_ProNet                  = 4,    /* Proteon ProNET Token Ring */
    Dlt_Chaos                   = 5,    /* Chaos */
    Dlt_Ieee802                 = 6,    /* Ieee 802 Networks */
    Dlt_ArcNet                  = 7,    /* ARCNET */
    Dlt_Slip                    = 8,    /* Serial Line IP */
    Dlt_Ppp                     = 9,    /* Point-to-point Protocol */
    Dlt_Fddi                    = 10,   /* FDDI */
    Dlt_Atm_Rfc1483             = 11,   /* LLC/SNAP encapsulated atm */
    Dlt_Raw                     = 12,   /* raw IP */

/*
 * These are values from BSD/OS's "bpf.h".
 * These are not the same as the values from the traditional libpcap
 * "bpf.h"; however, these values shouldn't be generated by any
 * OS other than BSD/OS, so the correct values to use here are the
 * BSD/OS values.
 *
 * Platforms that have already assigned these values to other
 * Dlt_ codes, however, should give these codes the values
 * from that platform, so that programs that use these codes will
 * continue to compile - even though they won't correctly read
 * files of these types.
 */
    Dlt_Slip_BsDos              = 15,   /* BSD/OS Serial Line IP */
    Dlt_Ppp_BsDos               = 16,   /* BSD/OS Point-to-point Protocol */

    Dlt_Atm_Clip                = 19,   /* Linux Classical-IP over Atm */

/*
 * These values are defined by NetBSD; other platforms should refrain from
 * using them for other purposes, so that NetBSD savefiles with link
 * types of 50 or 51 can be read as this type on all platforms.
 */
    Dlt_Ppp_Serial              = 50,    /* Ppp over serial with HDLC encapsulation */
    Dlt_Ppp_Ether               = 51,    /* Ppp over Ethernet */

/*
 * Reserved for the Symantec Enterprise Firewall.
 */
    Dlt_Symantec_Firewall       = 99,

/*
 * Values between 100 and 103 are used in capture file headers as
 * link-layer header type LINKTYPE_ values corresponding to Dlt_ types
 * that differ between platforms; don't use those values for new Dlt_
 * new types.
 */

/*
 * Values starting with 104 are used for newly-assigned link-layer
 * header type values; for those link-layer header types, the Dlt_
 * value returned by pcap_datalink() and passed to pcap_open_dead(),
 * and the LINKTYPE_ value that appears in capture files, are the
 * same.
 *
 * Dlt_Matching_MIN is the lowest such value; Dlt_Matching_MAX is
 * the highest such value.
 */
    Dlt_Matching_MIN            = 104,

/*
 * This value was defined by libpcap 0.5; platforms that have defined
 * it with a different value should define it here with that value -
 * a link type of 104 in a save file will be mapped to Dlt_C_HDLC,
 * whatever value that happens to be, so programs will correctly
 * handle files with that link type regardless of the value of
 * Dlt_C_Hdlc.
 *
 * The name Dlt_C_Hdlc was used by BSD/OS; we use that name for source
 * compatibility with programs written for BSD/OS.
 *
 * libpcap 0.5 defined it as Dlt_CHdlc; we define Dlt_CHdlc as well,
 * for source compatibility with programs written for libpcap 0.5.
 */
    Dlt_C_Hdlc                  = 104,    /* Cisco HDLC */
    Dlt_CHdlc                   = Dlt_C_Hdlc,

    Dlt_Ieee802_11              = 105,    /* Ieee 802.11 wireless */

/*
 * Values between 106 and 107 are used in capture file headers as
 * link-layer types corresponding to Dlt_ types that might differ
 * between platforms; don't use those values for new Dlt_ new types.
 */

/*
 * Frame Relay; BSD/OS has a Dlt_FR with a value of 11, but that collides
 * with other values.
 * Dlt_FR and Dlt_FRELAY packets start with the Q.922 Frame Relay header
 * (DLCI, etc.).
 */
    Dlt_FRelay                  = 107,

/*
 * OpenBSD Dlt_LOOP, for loopback devices; it's like Dlt_NULL, except
 * that the AF_ type in the link-layer header is in network byte order.
 *
 * OpenBSD defines it as 12, but that collides with Dlt_RAW, so we
 * define it as 108 here.  If OpenBSD picks up this file, it should
 * define Dlt_LOOP as 12 in its version, as per the comment above -
 * and should not use 108 as a Dlt_ value.
 */
    Dlt_Loop                    = 108,

/*
 * Values between 109 and 112 are used in capture file headers as
 * link-layer types corresponding to Dlt_ types that might differ
 * between platforms; don't use those values for new Dlt_ new types.
 */

/*
 * Encapsulated packets for IPsec; Dlt_ENC is 13 in OpenBSD, but that's
 * Dlt_Slip_BsDos in NetBSD, so we don't use 13 for it in OSes other
 * than OpenBSD.
 */
    Dlt_Enc                     = 109,

/*
 * This is for Linux cooked sockets.
 */
    Dlt_Linux_Sll               = 113,

/*
 * Apple LocalTalk hardware.
 */
    Dlt_LTalk                   = 114,

/*
 * Acorn Econet.
 */
    Dlt_EcoNet                  = 115,

/*
 * Reserved for use with OpenBSD ipfilter.
 */
    Dlt_IpFilter                = 116,

/*
 * Reserved for use in capture-file headers as a link-layer type
 * corresponding to OpenBSD DLT_PFLOG; DLT_PFLOG is 17 in OpenBSD,
 * but that's DLT_LANE8023 in SuSE 6.3, so we can't use 17 for it
 * in capture-file headers.
 */
    Dlt_PfLog                   = 117,

/*
 * Registered for Cisco-internal use.
 */
    Dlt_Cisco_Ios               = 118,

/*
 * Reserved for 802.11 cards using the Prism II chips, with a link-layer
 * header including Prism monitor mode information plus an 802.11
 * header.
 */
    Dlt_Prism_Header            = 119,

/*
 * Reserved for Aironet 802.11 cards, with an Aironet link-layer header
 * (see Doug Ambrisko's FreeBSD patches).
 */
    Dlt_AiroNet_Header          = 120,

/*
 * Reserved for use by OpenBSD's pfsync device.
 */
    Dlt_PpSync                  = 121,

/*
 * Reserved for Siemens HiPath HDLC. XXX
 */
    Dlt_HHdlc                   = 121,

/*
 * Reserved for RFC 2625 IP-over-Fibre Channel.
 */
    Dlt_Ip_Over_Fc              = 122,

/*
 * Reserved for Full Frontal Atm on Solaris.
 */
    Dlt_SunAtm                  = 123,

/*
 * Reserved as per request from Kent Dahlgren <kent@praesum.com>
 * for private use.
 */
    Dlt_Rio                     = 124,    /* RapidIO */
    Dlt_Pci_Exp                 = 125,    /* PCI Express */
    Dlt_Aurora                  = 126,    /* Xilinx Aurora link layer */

/*
 * BSD header for 802.11 plus a number of bits of link-layer information
 * including radio information.
 */
#ifndef Dlt_Ieee802_11_Radio
    Dlt_Ieee802_11_Radio        = 127,
#endif

/*
 * Reserved for TZSP encapsulation.
 */
    Dlt_Tzsp                    = 128,    /* Tazmen Sniffer Protocol */

/*
 * Reserved for Linux ARCNET.
 */
    Dlt_ArcNet_Linux            = 129,

/*
 * Juniper-private data link types.
 */
    Dlt_Juniper_MlPpp           = 130,
    Dlt_Juniper_Mlfr            = 131,
    Dlt_Juniper_Es              = 132,
    Dlt_Juniper_Ggsn            = 133,
    Dlt_Juniper_MFR             = 134,
    Dlt_Juniper_Atm2            = 135,
    Dlt_Juniper_Services        = 136,
    Dlt_Juniper_Atm1            = 137,

/*
 * Apple IP-over-Ieee 1394, as per a request from Dieter Siegmund
 * <dieter@apple.com>.  The header that's presented is an Ethernet-like
 * header:
 *
 *    #define FIREWIRE_EUI64_LEN    8
 *    struct firewire_header {
 *        u_char  firewire_dhost[FIREWIRE_EUI64_LEN];
 *        u_char  firewire_shost[FIREWIRE_EUI64_LEN];
 *        u_short firewire_type;
 *    };
 *
 * with "firewire_type" being an Ethernet type value, rather than,
 * for example, raw GASP frames being handed up.
 */
    Dlt_Apple_Ip_Over_Ieee1394  = 138,

/*
 * Various SS7 encapsulations, as per a request from Jeff Morriss
 * <jeff.morriss[AT]ulticom.com> and subsequent discussions.
 */
    Dlt_Mtp2_With_PHdr          = 139,    /* pseudo-header with various info, followed by Mtp2 */
    Dlt_Mtp2                    = 140,    /* Mtp2, without pseudo-header */
    Dlt_Mtp3                    = 141,    /* Mtp3, without pseudo-header or Mtp2 */
    Dlt_Sccp                    = 142,    /* SCCP, without pseudo-header or Mtp2 or Mtp3 */

/*
 * Reserved for DOCSIS.
 */
    Dlt_Docsis                  = 143,

/*
 * Reserved for Linux IrDA.
 */
    Dlt_Linux_Irda              = 144,

/*
 * Reserved for Ibm SP switch and Ibm Next Federation switch.
 */
    Dlt_Ibm_Sp                  = 145,
    Dlt_Ibm_Sn                  = 146,

/*
 * Reserved for private use.  If you have some link-layer header type
 * that you want to use within your organization, with the capture files
 * using that link-layer header type not ever be sent outside your
 * organization, you can use these values.
 *
 * No libpcap release will use these for any purpose, nor will any
 * tcpdump release use them, either.
 *
 * Do *NOT* use these in capture files that you expect anybody not using
 * your private versions of capture-file-reading tools to read; in
 * particular, do *NOT* use them in products, otherwise you may find that
 * people won't be able to use tcpdump, or snort, or Ethereal, or... to
 * read capture files from your firewall/intrusion detection/traffic
 * monitoring/etc. appliance, or whatever product uses that Dlt_ value,
 * and you may also find that the developers of those applications will
 * not accept patches to let them read those files.
 *
 * Also, do not use them if somebody might send you a capture using them
 * for *their* private type and tools using them for *your* private type
 * would have to read them.
 *
 * Instead, ask "tcpdump-workers@tcpdump.org" for a new Dlt_ value,
 * as per the comment above, and use the type you're given.
 */
    Dlt_User0                   = 147,
    Dlt_User1                   = 148,
    Dlt_User2                   = 149,
    Dlt_User3                   = 150,
    Dlt_User4                   = 151,
    Dlt_User5                   = 152,
    Dlt_User6                   = 153,
    Dlt_User7                   = 154,
    Dlt_User8                   = 155,
    Dlt_User9                   = 156,
    Dlt_User10                  = 157,
    Dlt_User11                  = 158,
    Dlt_User12                  = 159,
    Dlt_User13                  = 160,
    Dlt_User14                  = 161,
    Dlt_User15                  = 162,

/*
 * For future use with 802.11 captures - defined by AbsoluteValue
 * Systems to store a number of bits of link-layer information
 * including radio information:
 *
 *    http://www.shaftnet.org/~pizza/software/capturefrm.txt
 *
 * but it might be used by some non-AVS drivers now or in the
 * future.
 */
    Dlt_Ieee802_11_Radio_Avs    = 163,    /* 802.11 plus AVS radio header */

/*
 * Juniper-private data link type, as per request from
 * Hannes Gredler <hannes@juniper.net>.  The Dlt_s are used
 * for passing on chassis-internal metainformation such as
 * QOS profiles, etc..
 */
    Dlt_Juniper_Monitor         = 164,

/*
 * Reserved for BACnet MS/TP.
 */
    Dlt_BacNet_Ms_Tp            = 165,

/*
 * Another Ppp variant as per request from Karsten Keil <kkeil@suse.de>.
 *
 * This is used in some OSes to allow a kernel socket filter to distinguish
 * between incoming and outgoing packets, on a socket intended to
 * supply pppd with outgoing packets so it can do dial-on-demand and
 * hangup-on-lack-of-demand; incoming packets are filtered out so they
 * don't cause pppd to hold the connection up (you don't want random
 * input packets such as port scans, packets from old lost connections,
 * etc. to force the connection to stay up).
 *
 * The first byte of the Ppp header (0xff03) is modified to accomodate
 * the direction - 0x00 = IN, 0x01 = OUT.
 */
    Dlt_Ppp_PppD                = 166,

/*
 * Names for backwards compatibility with older versions of some Ppp
 * software; new software should use Dlt_Ppp_PppD.
 */
    Dlt_Ppp_With_Direction      = Dlt_Ppp_PppD,
    Dlt_Linux_Ppp_WithDirection = Dlt_Ppp_PppD,

/*
 * Juniper-private data link type, as per request from
 * Hannes Gredler <hannes@juniper.net>.  The Dlt_s are used
 * for passing on chassis-internal metainformation such as
 * QOS profiles, cookies, etc..
 */
    Dlt_Juniper_Pppoe           = 167,
    Dlt_Juniper_Pppoe_Atm       = 168,

    Dlt_Gprs_Llc                = 169,    /* GPRS LLC */
    Dlt_Gpf_T                   = 170,    /* GPF-T (ITU-T G.7041/Y.1303) */
    Dlt_Gpf_F                   = 171,    /* GPF-F (ITU-T G.7041/Y.1303) */

/*
 * Requested by Oolan Zimmer <oz@gcom.com> for use in Gcom's T1/E1 line
 * monitoring equipment.
 */
    Dlt_Gcom_T1E1               = 172,
    Dlt_Gcom_Serial             = 173,

/*
 * Juniper-private data link type, as per request from
 * Hannes Gredler <hannes@juniper.net>.  The Dlt_ is used
 * for internal communication to Physical Interface Cards (PIC)
 */
    Dlt_Juniper_Pic_Peer        = 174,

/*
 * Link types requested by Gregor Maier <gregor@endace.com> of Endace
 * Measurement Systems.  They add an ERF header (see
 * http://www.endace.com/support/EndaceRecordFormat.pdf) in front of
 * the link-layer header.
 */
    Dlt_Erf_Eth                 = 175,    /* Ethernet */
    Dlt_Erf_Pos                 = 176,    /* Packet-over-SONET */

/*
 * Requested by Daniele Orlandi <daniele@orlandi.com> for raw LAPD
 * for vISDN (http://www.orlandi.com/visdn/).  Its link-layer header
 * includes additional information before the LAPD header, so it's
 * not necessarily a generic LAPD header.
 */
    Dlt_Linux_Lapd              = 177,

/*
 * Juniper-private data link type, as per request from
 * Hannes Gredler <hannes@juniper.net>.
 * The Dlt_ are used for prepending meta-information
 * like interface index, interface name
 * before standard Ethernet, Ppp, Frelay & C-HDLC Frames
 */
    Dlt_Juniper_Ether           = 178,
    Dlt_Juniper_Ppp             = 179,
    Dlt_Juniper_FRelay          = 180,
    Dlt_Juniper_CHdlc           = 181,

/*
 * Multi Link Frame Relay (FRF.16)
 */
    Dlt_Mfr                     = 182,

/*
 * Juniper-private data link type, as per request from
 * Hannes Gredler <hannes@juniper.net>.
 * The Dlt_ is used for internal communication with a
 * voice Adapter Card (PIC)
 */
    Dlt_Juniper_Vp              = 183,

/*
 * Arinc 429 frames.
 * Dlt_ requested by Gianluca Varenni <gianluca.varenni@cacetech.com>.
 * Every frame contains a 32bit A429 label.
 * More documentation on Arinc 429 can be found at
 * http://www.condoreng.com/support/downloads/tutorials/ARINCTutorial.pdf
 */
    Dlt_A429                    = 184,

/*
 * Arinc 653 Interpartition Communication messages.
 * Dlt_ requested by Gianluca Varenni <gianluca.varenni@cacetech.com>.
 * Please refer to the A653-1 standard for more information.
 */
    Dlt_A653_Icm                = 185,

/*
 * USB packets, beginning with a USB setup header; requested by
 * Paolo Abeni <paolo.abeni@email.it>.
 */
    Dlt_Usb                     = 186,

/*
 * Bluetooth HCI UART transport layer (part H:4); requested by
 * Paolo Abeni.
 */
    Dlt_Bluetooth_Hci_H4        = 187,

/*
 * Ieee 802.16 MAC Common Part Sublayer; requested by Maria Cruz
 * <cruz_petagay@bah.com>.
 */
    Dlt_Ieee802_16_Mac_Cps      = 188,

/*
 * USB packets, beginning with a Linux USB header; requested by
 * Paolo Abeni <paolo.abeni@email.it>.
 */
    Dlt_Usb_Linux               = 189,

/*
 * Controller Area Network (CAN) v. 2.0B packets.
 * Dlt_ requested by Gianluca Varenni <gianluca.varenni@cacetech.com>.
 * Used to dump CAN packets coming from a CAN Vector board.
 * More documentation on the CAN v2.0B frames can be found at
 * http://www.can-cia.org/downloads/?269
 */
    Dlt_Can20B                  = 190,

/*
 * Ieee 802.15.4, with address fields padded, as is done by Linux
 * drivers; requested by Juergen Schimmer.
 */
    Dlt_Ieee802_15_4_Linux      = 191,

/*
 * Per Packet Information encapsulated packets.
 * Dlt_ requested by Gianluca Varenni <gianluca.varenni@cacetech.com>.
 */
    Dlt_Ppi                     = 192,

/*
 * Header for 802.16 MAC Common Part Sublayer plus a radiotap radio header;
 * requested by Charles Clancy.
 */
    Dlt_Ieee802_16_Mac_Cps_Radio = 193,

/*
 * Juniper-private data link type, as per request from
 * Hannes Gredler <hannes@juniper.net>.
 * The Dlt_ is used for internal communication with a
 * integrated service module (ISM).
 */
    Dlt_Juniper_Ism             = 194,

/*
 * Ieee 802.15.4, exactly as it appears in the spec (no padding, no
 * nothing); requested by Mikko Saarnivala <mikko.saarnivala@sensinode.com>.
 */
    Dlt_Ieee802_15_4            = 195,

/*
 * Various link-layer types, with a pseudo-header, for SITA
 * (http://www.sita.aero/); requested by Fulko Hew (fulko.hew@gmail.com).
 */
    Dlt_Sita                    = 196,

/*
 * Various link-layer types, with a pseudo-header, for Endace DAG cards;
 * encapsulates Endace ERF records.  Requested by Stephen Donnelly
 * <stephen@endace.com>.
 */
    Dlt_Erf                     = 197,

/*
 * Special header prepended to Ethernet packets when capturing from a
 * u10 Networks board.  Requested by Phil Mulholland
 * <phil@u10networks.com>.
 */
    Dlt_Raif1                   = 198,

/*
 * IPMB packet for IPMI, beginning with the I2C slave address, followed
 * by the netFn and LUN, etc..  Requested by Chanthy Toeung
 * <chanthy.toeung@ca.kontron.com>.
 */
    Dlt_Ipmb                    = 199,

/*
 * Juniper-private data link type, as per request from
 * Hannes Gredler <hannes@juniper.net>.
 * The Dlt_ is used for capturing data on a secure tunnel interface.
 */
    Dlt_Juniper_St              = 200,

/*
 * Bluetooth HCI UART transport layer (part H:4), with pseudo-header
 * that includes direction information; requested by Paolo Abeni.
 */
    Dlt_Bluetooth_Hci_H4_With_PHdr = 201,

/*
 * AX.25 packet with a 1-byte KISS header; see
 *
 *      http://www.ax25.net/kiss.htm
 *
 * as per Richard Stearn <richard@rns-stearn.demon.co.uk>.
 */
    Dlt_AX25_Kiss               = 202,

/*
 * LAPD packets from an ISDN channel, starting with the address field,
 * with no pseudo-header.
 * Requested by Varuna De Silva <varunax@gmail.com>.
 */
    Dlt_Lapd                    = 203,

/*
 * Variants of various link-layer headers, with a one-byte direction
 * pseudo-header prepended - zero means "received by this host",
 * non-zero (any non-zero value) means "sent by this host" - as per
 * Will Barker <w.barker@zen.co.uk>.
 */
    Dlt_Ppp_With_Dir            = 204,     /* Ppp - don't confuse with Dlt_Ppp_WITH_DIRECTION */
    Dlt_C_Hdlc_With_Dir         = 205,     /* Cisco HDLC */
    Dlt_FRelay_With_Dir         = 206,     /* Frame Relay */
    Dlt_Lapb_With_Dir           = 207,     /* LAPB */

/*
 * 208 is reserved for an as-yet-unspecified proprietary link-layer
 * type, as requested by Will Barker.
 */

/*
 * IPMB with a Linux-specific pseudo-header; as requested by Alexey Neyman
 * <avn@pigeonpoint.com>.
 */
    Dlt_Ipmb_Linux              = 209,

/*
 * FlexRay automotive bus - http://www.flexray.com/ - as requested
 * by Hannes Kaelber <hannes.kaelber@x2e.de>.
 */
    Dlt_FlexRay                 = 210,

/*
 * Media Oriented Systems Transport (MOST) bus for multimedia
 * transport - http://www.mostcooperation.com/ - as requested
 * by Hannes Kaelber <hannes.kaelber@x2e.de>.
 */
    Dlt_Most                    = 211,

/*
 * Local Interconnect Network (LIN) bus for vehicle networks -
 * http://www.lin-subbus.org/ - as requested by Hannes Kaelber
 * <hannes.kaelber@x2e.de>.
 */
    Dlt_Lin                     = 212,

/*
 * X2E-private data link type used for serial line capture,
 * as requested by Hannes Kaelber <hannes.kaelber@x2e.de>.
 */
    Dlt_X2E_Serial              = 213,

/*
 * X2E-private data link type used for the Xoraya data logger
 * family, as requested by Hannes Kaelber <hannes.kaelber@x2e.de>.
 */
    Dlt_X2E_Xoraya              = 214,

/*
 * Ieee 802.15.4, exactly as it appears in the spec (no padding, no
 * nothing), but with the PHY-level data for non-ASK PHYs (4 octets
 * of 0 as preamble, one octet of SFD, one octet of frame length+
 * reserved bit, and then the MAC-layer data, starting with the
 * frame control field).
 *
 * Requested by Max Filippov <jcmvbkbc@gmail.com>.
 */
    Dlt_Ieee802_15_4_NonAsk_Phy = 215,

/*
 * David Gibson <david@gibson.dropbear.id.au> requested this for
 * captures from the Linux kernel /dev/input/eventN devices. This
 * is used to communicate keystrokes and mouse movements from the
 * Linux kernel to display systems, such as Xorg.
 */
       Dlt_Linux_EvDev          = 216,

/*
 * GSM Um and Abis interfaces, preceded by a "gsmtap" header.
 *
 * Requested by Harald Welte <laforge@gnumonks.org>.
 */
       Dlt_GsmTap_Um            = 217,
       Dlt_GsmTap_Abis          = 218,

/*
 * MPLS, with an MPLS label as the link-layer header.
 * Requested by Michele Marchetto <michele@openbsd.org> on behalf
 * of OpenBSD.
 */
       Dlt_Mpls                 = 219,

/*
 * USB packets, beginning with a Linux USB header, with the USB header
 * padded to 64 bytes; required for memory-mapped access.
 */
       Dlt_Usb_Linux_MMapped    = 220,

/*
 * DECT packets, with a pseudo-header; requested by
 * Matthias Wenzel <tcpdump@mazzoo.de>.
 */
       Dlt_Dect                 = 221,
/*
 * From: "Lidwa, Eric (GSFC-582.0)[SGT INC]" <eric.lidwa-1@nasa.gov>
 * Date: Mon, 11 May 2009 11:18:30 -0500
 *
 * Dlt_AOS. We need it for AOS Space Data Link Protocol.
 *   I have already written dissectors for but need an OK from
 *   legal before I can submit a patch.
 *
 */
       Dlt_Aos                  = 222,

/*
 * Wireless HART (Highway Addressable Remote Transducer)
 * From the HART Communication Foundation
 * IES/PAS 62591
 *
 * Requested by Sam Roberts <vieuxtech@gmail.com>.
 */
       Dlt_WiHart               = 223,

/*
 * Fibre Channel FC-2 frames, beginning with a Frame_Header.
 * Requested by Kahou Lei <kahou82@gmail.com>.
 */
       Dlt_Fc_2                 = 224,

/*
 * Fibre Channel FC-2 frames, beginning with an encoding of the
 * SOF, and ending with an encoding of the EOF.
 *
 * The encodings represent the frame delimiters as 4-byte sequences
 * representing the corresponding ordered sets, with K28.5
 * represented as 0xBC, and the D symbols as the corresponding
 * byte values; for example, SOFi2, which is K28.5 - D21.5 - D1.2 - D21.2,
 * is represented as 0xBC 0xB5 0x55 0x55.
 *
 * Requested by Kahou Lei <kahou82@gmail.com>.
 */
       Dlt_Fc_2_With_Frame_Delims = 225,
/*
 * Solaris ipnet pseudo-header; requested by Darren Reed <Darren.Reed@Sun.COM>.
 *
 * The pseudo-header starts with a one-byte version number; for version 2,
 * the pseudo-header is:
 *
 * struct dl_ipnetinfo {
 *     u_int8_t   dli_version;
 *     u_int8_t   dli_family;
 *     u_int16_t  dli_htype;
 *     u_int32_t  dli_pktlen;
 *     u_int32_t  dli_ifindex;
 *     u_int32_t  dli_grifindex;
 *     u_int32_t  dli_zsrc;
 *     u_int32_t  dli_zdst;
 * };
 *
 * dli_version is 2 for the current version of the pseudo-header.
 *
 * dli_family is a Solaris address family value, so it's 2 for IPv4
 * and 26 for IPv6.
 *
 * dli_htype is a "hook type" - 0 for incoming packets, 1 for outgoing
 * packets, and 2 for packets arriving from another zone on the same
 * machine.
 *
 * dli_pktlen is the length of the packet data following the pseudo-header
 * (so the captured length minus dli_pktlen is the length of the
 * pseudo-header, assuming the entire pseudo-header was captured).
 *
 * dli_ifindex is the interface index of the interface on which the
 * packet arrived.
 *
 * dli_grifindex is the group interface index number (for IPMP interfaces).
 *
 * dli_zsrc is the zone identifier for the source of the packet.
 *
 * dli_zdst is the zone identifier for the destination of the packet.
 *
 * A zone number of 0 is the global zone; a zone number of 0xffffffff
 * means that the packet arrived from another host on the network, not
 * from another zone on the same machine.
 *
 * An IPv4 or IPv6 datagram follows the pseudo-header; dli_family indicates
 * which of those it is.
 */
       Dlt_IpNet                = 226,

/*
 * CAN (Controller Area Network) frames, with a pseudo-header as supplied
 * by Linux SocketCAN.  See Documentation/networking/can.txt in the Linux
 * source.
 *
 * Requested by Felix Obenhuber <felix@obenhuber.de>.
 */
       Dlt_Can_SocketCan        = 227,

/*
 * Raw IPv4/IPv6; different from Dlt_RAW in that the Dlt_ value specifies
 * whether it's v4 or v6.  Requested by Darren Reed <Darren.Reed@Sun.COM>.
 */
    Dlt_IpV4                    = 228,
    Dlt_IpV6                    = 229,

/*
 * Ieee 802.15.4, exactly as it appears in the spec (no padding, no
 * nothing), and with no FCS at the end of the frame; requested by
 * Jon Smirl <jonsmirl@gmail.com>.
 */
    Dlt_Ieee802_15_4_NoFcs      = 230,

/*
 * Raw D-Bus:
 *
 *    http://www.freedesktop.org/wiki/Software/dbus
 *
 * messages:
 *
 *    http://dbus.freedesktop.org/doc/dbus-specification.html#message-protocol-messages
 *
 * starting with the endianness flag, followed by the message type, etc.,
 * but without the authentication handshake before the message sequence:
 *
 *    http://dbus.freedesktop.org/doc/dbus-specification.html#auth-protocol
 *
 * Requested by Martin Vidner <martin@vidner.net>.
 */
    Dlt_DBus                    = 231,

/*
 * Juniper-private data link type, as per request from
 * Hannes Gredler <hannes@juniper.net>.
 */
    Dlt_Juniper_Vs              = 232,
    Dlt_Juniper_Srx_E2E         = 233,
    Dlt_Juniper_FibreChannel    = 234,

/*
 * DVB-CI (DVB Common Interface for communication between a PC Card
 * module and a DVB receiver).  See
 *
 *    http://www.kaiser.cx/pcap-dvbci.html
 *
 * for the specification.
 *
 * Requested by Martin Kaiser <martin@kaiser.cx>.
 */
    Dlt_Dvb_Ci                  = 235,

/*
 * Variant of 3GPP TS 27.010 multiplexing protocol (similar to, but
 * *not* the same as, 27.010).  Requested by Hans-Christoph Schemmel
 * <hans-christoph.schemmel@cinterion.com>.
 */
    Dlt_Mux27010                = 236,

/*
 * STANAG 5066 D_PDUs.  Requested by M. Baris Demiray
 * <barisdemiray@gmail.com>.
 */
    Dlt_Stanag_5066_D_Pdu       = 237,

/*
 * Juniper-private data link type, as per request from
 * Hannes Gredler <hannes@juniper.net>.
 */
    Dlt_Juniper_Atm_Cemic       = 238,

/*
 * NetFilter LOG messages
 * (payload of netlink NFNL_SUBSYS_ULOG/NFULNL_MSG_PACKET packets)
 *
 * Requested by Jakub Zawadzki <darkjames-ws@darkjames.pl>
 */
    Dlt_NfLog                   = 239,

/*
 * Hilscher Gesellschaft fuer Systemautomation mbH link-layer type
 * for Ethernet packets with a 4-byte pseudo-header and always
 * with the payload including the FCS, as supplied by their
 * netANALYZER hardware and software.
 *
 * Requested by Holger P. Frommer <HPfrommer@hilscher.com>
 */
    Dlt_NetAnalyzer             = 240,

/*
 * Hilscher Gesellschaft fuer Systemautomation mbH link-layer type
 * for Ethernet packets with a 4-byte pseudo-header and FCS and
 * with the Ethernet header preceded by 7 bytes of preamble and
 * 1 byte of SFD, as supplied by their netANALYZER hardware and
 * software.
 *
 * Requested by Holger P. Frommer <HPfrommer@hilscher.com>
 */
    Dlt_NetAnalyzer_Transparent = 241,

/*
 * IP-over-Infiniband, as specified by RFC 4391.
 *
 * Requested by Petr Sumbera <petr.sumbera@oracle.com>.
 */
    Dlt_Ipoib                   = 242,

/*
 * Mpeg-2 transport stream (ISO 13818-1/ITU-T H.222.0).
 *
 * Requested by Guy Martin <gmsoft@tuxicoman.be>.
 */
    Dlt_Mpeg_2_Ts               = 243,

/*
 * ng4T GmbH's UMTS Iub/Iur-over-Atm and Iub/Iur-over-IP format as
 * used by their ng40 protocol tester.
 *
 * Requested by Jens Grimmer <jens.grimmer@ng4t.com>.
 */
    Dlt_Ng40                    = 244,

/*
 * Pseudo-header giving adapter number and flags, followed by an Nfc
 * (Near-Field Communications) Logical Link Control Protocol (Llcp) PDU,
 * as specified by Nfc Forum Logical Link Control Protocol Technical
 * Specification Llcp 1.1.
 *
 * Requested by Mike Wakerly <mikey@google.com>.
 */
    Dlt_Nfc_Llcp                = 245,

/*
 * 245 is used as LINKTYPE_PFSYNC; do not use it for any other purpose.
 *
 * Dlt_PFSYNC has different values on different platforms, and all of
 * them collide with something used elsewhere.  On platforms that
 * don't already define it, define it as 245.
 */
    Dlt_Matching_Max            = 246    /* highest value in the "matching" range */
};

/*
* Structure to retrieve available DLTs for the interface.
*/
struct BpfDltList
{
    uint32_t    bfl_len;        /* number of bfd_list array */
    uint32_t    *bfl_list;      /* array of DLTs */
};

/*
* DLT and savefile link type values are split into a class and
* a member of that class.  A class value of 0 indicates a regular
* DLT_/LINKTYPE_ value.
*/
inline BpfUInt32 Dlt_Class(BpfUInt32 x)
{
    return x & 0x03ff0000;
}

inline BpfCode operator & (BpfCode a, uint16_t b) NN_NOEXCEPT
{
    return static_cast<BpfCode>(static_cast<uint16_t>(a) & b);
}

inline BpfCode operator + (BpfCode a, BpfCode b) NN_NOEXCEPT
{
    return static_cast<BpfCode>(static_cast<uint16_t>(a) + static_cast<uint16_t>(b));
}

template <class T>
inline T operator + (BpfCode a, T b) NN_NOEXCEPT
{
    return static_cast<T>(static_cast<T>(a) + b);
}

/*
* The instruction encodings.
*/

/* instruction classes */
inline BpfCode Bpf_Class(BpfCode code)
{
    return code & 0x07;
}

/* ld/ldx fields */
inline BpfCode Bpf_Size(BpfCode code)
{
    return code & 0x18;
}

inline BpfCode Bpf_Mode(BpfCode code)
{
    return code & 0xe0;
}

/* alu/jmp fields */
inline BpfCode Bpf_Op(BpfCode code)
{
    return code & 0xf0;
}

inline BpfCode Bpf_Src(BpfCode code)
{
    return code & 0x08;
}

/* ret - Bpf_K and Bpf_X also apply */
inline BpfCode Bpf_Rval(BpfCode code)
{
    return code & 0x18;
}

/* misc */
inline BpfCode Bpf_MiscOp(BpfCode code)
{
    return code & 0xf8;
}

/*
* Macros for BpfInsn array initializers.
*/
inline BpfInsn Bpf_Stmt(BpfCode code, BpfUInt32 k)
{
    return BpfInsn{ code, 0, 0, k };
}

inline BpfInsn Bpf_Jump(BpfCode code, BpfUInt32 k, uint8_t jt, uint8_t jf)
{
    return BpfInsn{ code, jt, jf, k };
}

}} /* nn::socket */
