﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <memory>
#include <type_traits>
#include <nn/sf/sf_Types.h>
#include <nn/sf/impl/detail/sf_ServiceObjectImpl.h>
#include <nn/util/util_MacroForVariadic.h>
#include <nn/sf/detail/sf_CommonUtil.h>

namespace nn { namespace sf {

namespace detail {

struct StatelessDummyAllocator
{
};

template <typename Base, typename AllocationPolicy>
class ObjectImplFactoryWithStatelessAllocator
{
public:

    class Object;
    typedef StatelessDummyAllocator Allocator;
    typedef typename AllocationPolicy::template GetAllocator<Object>::type StatelessAllocator;

    class Object final : private impl::detail::ServiceObjectImplBase2, public Base
    {
        NN_DISALLOW_COPY(Object);
        NN_DISALLOW_MOVE(Object);
        friend class ObjectImplFactoryWithStatelessAllocator;
    private:

        template <typename... Args>
        explicit Object(Args&&... args) NN_NOEXCEPT
            : Base{std::forward<Args>(args)...}
        {
        }

        static void* operator new(size_t size) NN_NOEXCEPT
        {
            return AllocationPolicy::template AllocateAligned<Object>(size, NN_ALIGNOF(Object));
        }

        static void operator delete(void* p, size_t size) NN_NOEXCEPT
        {
            AllocationPolicy::template DeallocateAligned<Object>(p, size, NN_ALIGNOF(Object));
        }

        // delete
        static void* operator new(size_t size, Allocator* a) NN_NOEXCEPT;
        static void operator delete(void* p, Allocator* a) NN_NOEXCEPT;

        void DisposeImpl() NN_NOEXCEPT
        {
            delete this;
        }

    public:

        void AddReference() NN_NOEXCEPT
        {
            ServiceObjectImplBase2::AddReferenceImpl();
        }

        void Release() NN_NOEXCEPT
        {
            if (ServiceObjectImplBase2::ReleaseImpl())
            {
                DisposeImpl();
            }
        }

        Allocator* GetAllocator() const NN_NOEXCEPT
        {
            return nullptr;
        }

    };

    template <typename... Args>
    static Object* Create(Args&&... args)
    {
        return new Object(std::forward<Args>(args)...);
    }

    template <typename... Args>
    static Object* Create(Allocator* allocator, Args&&... args)
    {
        NN_UNUSED(allocator);
        return new Object(std::forward<Args>(args)...);
    }

};

template <typename Base, typename AllocationPolicy>
class ObjectImplFactoryWithStatefulAllocator
{
public:

    typedef typename AllocationPolicy::Allocator Allocator;

    class Object final : private impl::detail::ServiceObjectImplBase2, public Base
    {
        NN_DISALLOW_COPY(Object);
        NN_DISALLOW_MOVE(Object);
        friend class ObjectImplFactoryWithStatefulAllocator;
    private:

        template <typename... Args>
        explicit Object(Args&&... args) NN_NOEXCEPT
            : Base{std::forward<Args>(args)...}
        {
        }

        Allocator* m_Allocator;

        // delete
        static void* operator new(size_t size) NN_NOEXCEPT;
        static void operator delete(void* p, size_t size) NN_NOEXCEPT
        {
            // 未定義だとリンクエラーになるため、空実装
            NN_UNUSED(p);
            NN_UNUSED(size);
        }

        static void* operator new(size_t size, Allocator* a) NN_NOEXCEPT
        {
            return AllocationPolicy::AllocateAligned(a, size, NN_ALIGNOF(Object));
        }

        static void operator delete(void* p, Allocator* a) NN_NOEXCEPT
        {
            AllocationPolicy::DeallocateAligned(a, p, sizeof(Object), NN_ALIGNOF(Object));
        }

        void DisposeImpl() NN_NOEXCEPT
        {
            auto a = GetAllocator();
            this->~Object();
            operator delete(this, a);
        }

    public:

        void AddReference() NN_NOEXCEPT
        {
            ServiceObjectImplBase2::AddReferenceImpl();
        }

        void Release() NN_NOEXCEPT
        {
            if (ServiceObjectImplBase2::ReleaseImpl())
            {
                DisposeImpl();
            }
        }

        Allocator* GetAllocator() const NN_NOEXCEPT
        {
            return m_Allocator;
        }

    };

    template <typename... Args>
    static Object* Create(Allocator* allocator, Args&&... args)
    {
        auto p = new (allocator) Object(std::forward<Args>(args)...);
        if (!p)
        {
            return nullptr;
        }
        p->m_Allocator = allocator;
        return p;
    }

};

}

// Base クラスに AddReference/Release を実装して、
// Create 関数で AllocationPolicy を用いてインスタンス化するためのアダプタ。
//
// AllocationPolicy におけるアロケータが状態を持つかどうか(HasStatefulAllocator)によって特殊化され、
// ObjectImplFactoryWithStatelessAllocator または ObjectImplFactoryWithStatefulAllocator に実装が委譲される。
template <typename Base, typename AllocationPolicy, typename Enabled = void>
class ObjectImplFactory;
/*
{
public:

    // Create 関数で生成されるインスタンスの型
    class Object final : public Base
    {
    public:
        void AddReference() NN_NOEXCEPT;
        void Release() NN_NOEXCEPT;
        Allocator* GetAllocator() const NN_NOEXCEPT;
    private:
        static void* operator new(size_t size) NN_NOEXCEPT;
        static void operator delete(void* p, size_t size) NN_NOEXCEPT;
        static void* operator new(size_t size, Allocator* a) NN_NOEXCEPT;
        static void operator delete(void* p, Allocator* a) NN_NOEXCEPT;
    };

    // AllocationPolicy::HasStatefulAllocator == false のときのインスタンス生成関数
    //
    // args 引数を Base のコンストラクタの引数に転送し、オブジェクトを生成する。
    template <typename... Args>
    static Object* Create(Args&&... args) NN_NOEXCEPT;

    // AllocationPolicy::HasStatefulAllocator == false/true のとき共通のインスタンス生成関数
    //
    // args 引数を Base のコンストラクタの引数に転送し、
    // allocator を使って領域をアロケートしてオブジェクトを生成する。
    // ただし、AllocationPolicy::HasStatefulAllocator == false のとき、allocator は無視される。
    template <typename... Args>
    static Object* Create(Allocator* allocator, Args&&... args) NN_NOEXCEPT;

};
*/

template <typename Base, typename AllocationPolicy>
class ObjectImplFactory<Base, AllocationPolicy, typename std::enable_if<!AllocationPolicy::HasStatefulAllocator>::type>
    : public detail::ObjectImplFactoryWithStatelessAllocator<Base, AllocationPolicy>
{
};

template <typename Base, typename AllocationPolicy>
class ObjectImplFactory<Base, AllocationPolicy, typename std::enable_if<AllocationPolicy::HasStatefulAllocator>::type>
    : public detail::ObjectImplFactoryWithStatefulAllocator<Base, AllocationPolicy>
{
};

}}
