﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/**
    @file
    @brief サービスフレームワークで使用される共有オブジェクトインターフェイスとスマートポインタを定義します。
    @details
     このファイルを直接インクルードする必要はありません。 <nn/sf/sf_Types.h> をインクルードしてください。
*/

#include <nn/nn_Common.h>
#include <nn/sf/sf_Out.h>
#include <nn/nn_SdkAssert.h>
#include <type_traits>
#include <utility>

namespace nn { namespace sf {

/**
    @brief 寿命管理が可能なオブジェクトのインターフェイスです。

    @details
     AddReference() と Release() による寿命管理はスレッドセーフであり、
     参照がなくなったと判断した際には、デストラクタが一回だけ呼ばれます。

     このクラスのデストラクタは Release() 経由で呼ばれることを想定しており、
     直接呼ばれることを想定していません。
     このため、このクラスを基底クラスとして使用する際には、
     その派生先のクラスのデストラクタも protected にする必要があります。

     多くの場合、このクラスを直接使用する必要はなく IServiceObejct が継承することで使用します。

     オブジェクトに親子関係があり、子の動作のために親が必要な場合には、
     親クラスを ISharedObject から派生し、子クラスのメンバに SharedPointer<ISharedObject> を持たせることで、
     子クラスが破棄されるまで親クラスが破棄されないことが保証できます。
     この場合、親クラスは Release() 経由で破棄されることになるため、
     デストラクタを protected にする必要があることに注意してください。
*/
class ISharedObject
{
    NN_DISALLOW_COPY(ISharedObject);
    NN_DISALLOW_MOVE(ISharedObject);
public:

    /**
        @brief オブジェクト参照カウントを 1 加算します。

        @details
         この関数は内部用です。
         今後名前が変更になる可能性があるため、直接呼ばないようにしてください。
    */
    virtual void AddReference() NN_NOEXCEPT = 0;

    /**
        @brief オブジェクト参照カウントを 1 減算し、0 になった際にはオブジェクトを破棄します。

        @details
         この関数は内部用です。
         今後名前が変更になる可能性があるため、直接呼ばないようにしてください。
         かわりに ReleaseSharedObject() を呼ぶようにしてください。
    */
    virtual void Release() NN_NOEXCEPT = 0;

protected:

    ISharedObject() NN_NOEXCEPT {}

    /**
        @brief protected デストラクタ

        @details
         デストラクタを直接呼ぶことはできません。
         Release() などの内部で間接的に呼ばれます。
    */
    ~ISharedObject() NN_NOEXCEPT {}

};

namespace detail {

class SharedPointerBase
{
private:

    ISharedObject* m_P;

    void AddReferenceImpl() const NN_NOEXCEPT
    {
        if (m_P)
        {
            m_P->AddReference();
        }
    }

    void ReleaseImpl() const NN_NOEXCEPT
    {
        if (m_P)
        {
            m_P->Release();
        }
    }

public:

    SharedPointerBase() NN_NOEXCEPT
        : m_P(nullptr)
    {
    }

    SharedPointerBase(ISharedObject* p, bool addReference) NN_NOEXCEPT
        : m_P(p)
    {
        if (addReference)
        {
            AddReferenceImpl();
        }
    }

    SharedPointerBase(const SharedPointerBase& x) NN_NOEXCEPT
        : m_P(x.m_P)
    {
        AddReferenceImpl();
    }

    SharedPointerBase(SharedPointerBase&& x) NN_NOEXCEPT
        : m_P(x.m_P)
    {
        x.m_P = nullptr;
    }

    SharedPointerBase& operator=(const SharedPointerBase& rhs) NN_NOEXCEPT
    {
        SharedPointerBase tmp(rhs);
        tmp.swap(*this);
        return *this;
    }

    SharedPointerBase& operator=(SharedPointerBase&& rhs) NN_NOEXCEPT
    {
        SharedPointerBase tmp(std::move(rhs));
        tmp.swap(*this);
        return *this;
    }

    ~SharedPointerBase() NN_NOEXCEPT
    {
        ReleaseImpl();
    }

    void swap(SharedPointerBase& other) NN_NOEXCEPT
    {
        std::swap(this->m_P, other.m_P);
    }

    ISharedObject* Detach() NN_NOEXCEPT
    {
        auto ret = m_P;
        this->m_P = nullptr;
        return ret;
    }

    ISharedObject* Get() const NN_NOEXCEPT
    {
        return m_P;
    }

};

}

/**
    @brief ISharedObject 用の共有ポインタです。

    @tparam I インターフェイス型を指定します。ISharedObject を継承している必要があります。

    @details
     ISharedObject へのポインタを内部に持ち、必要に応じて ISharedObject::AddReference() と ISharedObject::Release() を呼びます。
     このクラスのデストラクト時にポインタが有効だった場合には ISharedObject::Release() が呼ばれます。

     効率のため、可能な限りコピーは使用せず、std::move を用いてムーブをするようにしてください。
*/
template <typename I>
class SharedPointer
{
    template <typename>
    friend class nn::sf::SharedPointer;
    template <typename, typename>
    friend class nn::sf::Out;
public:
    static constexpr bool IsTreatedAsSmartPointerBySf = true;
private:

    detail::SharedPointerBase m_Base;

    // operator bool 用ヘルパ
    typedef void (SharedPointer::*BoolType)() const;
    void FunctionForBoolType() const NN_NOEXCEPT {}

public:

    /**
        @brief 型引数にとった、対象となるインターフェイスの型の typedef です。
    */
    typedef I Interface;

    /**
        @brief デフォルトコンストラクタ: 何も指さない SharedPointer として初期化します。

        @post static_cast<bool>(*this) == false
    */
    SharedPointer() NN_NOEXCEPT
    {
    }

    /**
        @brief コンストラクタ: 何も指さない SharedPointer として初期化します。

        @post static_cast<bool>(*this) == false
    */
    NN_IMPLICIT SharedPointer(std::nullptr_t) NN_NOEXCEPT
    {
    }

    /**
        @brief コンストラクタ: 指定したインターフェイスポインタで初期化します。

        @param[in] p 内部インターフェイスへのポインタを指定します。
        @param[in] addReference p に対し ISharedObject::AddReference() を呼ぶかどうかを指定します。

        @post this->Get() == p

        @details
         インターフェイスポインタ p を使って初期化をします。
         addReference に true が指定された場合には、内部で ISharedObject::AddReference() が呼ばれます。

         サービスフレームワークの提供する関数は基本的に SharedPointer でインターフェイスポインタを包んで扱っており、
         インターフェイスポインタ Interface* を直接扱うことは稀であり、
         ほとんどの場合このコンストラクタを使用することはありません。

         ただし、オブジェクト間に親子関係を作りたいような場合には、
         親を ISharedObject から派生させて、子が親への参照を SharedPointer<ISharedObject> をメンバフィールドとして保持することがあり、
         この際には addReference を true としてこのコンストラクタを呼ぶことがあります。
    */
    SharedPointer(Interface* p, bool addReference) NN_NOEXCEPT
        : m_Base(static_cast<ISharedObject*>(p), addReference)
    {
    }

    /**
        @brief コピーコンストラクタ: SharedPointer をコピーし、有効な SharedPointer の場合には参照カウントを増やします。

        @param[in] x コピー元の SharedPointer を指定します。

        @post this->Get() == x.Get()

        @details
         x と同じオブジェクトを指すものとして *this を初期化します。
         内部的に ISharedObject::AddReference() が呼ばれます。
    */
    SharedPointer(const SharedPointer& x) NN_NOEXCEPT
        : m_Base(x.m_Base)
    {
    }

    /**
        @brief ムーブコンストラクタ: SharedPointer をムーブします。

        @param[in] x ムーブ元の SharedPointer を指定します。

        @post this->Get() == ムーブ前の x.Get()
        @post static_cast<bool>(x) == false

        @details
         x の指していたオブジェクトを指すものとして *this を初期化します。
         x は何も指さない状態になります。
         ISharedObject::AddReference() の呼び出しによる参照カウンタ操作がないため、効率的です。
    */
    SharedPointer(SharedPointer&& x) NN_NOEXCEPT
        : m_Base(std::move(x.m_Base))
    {
    }

    /**
        @brief コンストラクタ: SharedPointer をコピーし、有効な SharedPointer の場合には参照カウントを増やします。

        @tparam U コピー元の SharedPointer の指すインターフェイスを指定します。Interface は U の基底クラスであることが必要です。
        @param[in] x コピー元の SharedPointer を指定します。

        @post this->Get() == x.Get()

        @details
         x と同じオブジェクトを指すものとして *this を初期化します。
         内部的に ISharedObject::AddReference() が呼ばれます。
    */
    template <typename U>
    NN_IMPLICIT SharedPointer(const SharedPointer<U>& x) NN_NOEXCEPT
        : m_Base(x.m_Base)
    {
        static_assert(std::is_base_of<Interface, U>::value, "[SF-BASE-InvalidSharedPointerConversion] not std::is_base_of<Interface, U>::value");
    }

    /**
        @brief コンストラクタ: SharedPointer をムーブします。

        @tparam U ムーブ元の SharedPointer の指すインターフェイスを指定します。Interface は U の基底クラスであることが必要です。
        @param[in] x ムーブ元の SharedPointer を指定します。

        @post this->Get() == ムーブ前の x.Get()
        @post static_cast<bool>(x) == false

        @details
         x の指していたオブジェクトを指すものとして *this を初期化します。
         x は何も指さない状態になります。
         ISharedObject::AddReference() の呼び出しによる参照カウンタ操作がないため、効率的です。
    */
    template <typename U>
    NN_IMPLICIT SharedPointer(SharedPointer<U>&& x) NN_NOEXCEPT
        : m_Base(std::move(x.m_Base))
    {
        static_assert(std::is_base_of<Interface, U>::value, "[SF-BASE-InvalidSharedPointerConversion] not std::is_base_of<Interface, U>::value");
    }

    /**
        @brief SharedPointer を無効化します。

        @return *this を返します。

        @post static_cast<bool>(x) == false

        @details
         \*this が何らかのオブジェクトを指していた場合には、
         そのオブジェクトに対し ISharedObject::Release() が呼ばれます。
    */
    SharedPointer& operator=(std::nullptr_t) NN_NOEXCEPT
    {
        SharedPointer().swap(*this);
        return *this;
    }

    /**
        @brief コピー代入: コピー代入をします。

        @param rhs 代入元の SharedPointer を指定します。
        @return *this を返します。

        @post this->Get() == rhs.Get()

        @details
         rhs の指しているオブジェクトを指すように *this を書き換えます。
         \*this が何らかのオブジェクトとを指していた場合には、そのオブジェクトに対し ISharedObject::Release() が呼ばれます。
         また、内部的に ISharedObject::AddReference() が呼ばれます。
    */
    SharedPointer& operator=(const SharedPointer& rhs) NN_NOEXCEPT
    {
        SharedPointer tmp(rhs);
        tmp.swap(*this);
        return *this;
    }

    /**
        @brief ムーブ代入: ムーブ代入をします。

        @param rhs 代入元の SharedPointer を指定します。
        @return *this を返します。

        @post this->Get() == ムーブ前の rhs.Get()
        @post static_cast<bool>(x) == false

        @details
         rhs の指しているオブジェクトを指すように *this を書き換えます。
         \*this が何らかのオブジェクトとを指していた場合には、そのオブジェクトに対し ISharedObject::Release() が呼ばれます。
         rhs は何も指さない状態になります。
         ISharedObject::AddReference() の呼び出しによる参照カウンタ操作がないため、効率的です。
    */
    SharedPointer& operator=(SharedPointer&& rhs) NN_NOEXCEPT
    {
        SharedPointer tmp(std::move(rhs));
        tmp.swap(*this);
        return *this;
    }

    /**
        @brief 代入: コピー代入をします。

        @tparam U 代入元の SharedPointer の指すインターフェイスを指定します。
        @param rhs 代入元の SharedPointer を指定します。
        @return *this を返します。

        @post this->Get() == rhs.Get()

        @details
         rhs の指しているオブジェクトを指すように *this を書き換えます。
         *this が何らかのオブジェクトとを指していた場合には、そのオブジェクトに対し ISharedObject::Release() が呼ばれます。
         また、内部的に ISharedObject::AddReference() が呼ばれます。
    */
    template <typename U>
    SharedPointer& operator=(const SharedPointer<U>& rhs) NN_NOEXCEPT
    {
        static_assert(std::is_base_of<Interface, U>::value, "[SF-BASE-InvalidSharedPointerConversion] not std::is_base_of<Interface, U>::value");
        SharedPointer tmp(rhs);
        tmp.swap(*this);
        return *this;
    }

    /**
        @brief 代入: ムーブ代入をします。

        @tparam U 代入元の SharedPointer の指すインターフェイスを指定します。
        @param rhs 代入元の SharedPointer を指定します。
        @return *this を返します。

        @post this->Get() == ムーブ前の rhs.Get()
        @post static_cast<bool>(x) == false

        @details
         rhs の指しているオブジェクトを指すように *this を書き換えます。
         *this が何らかのオブジェクトとを指していた場合には、そのオブジェクトに対し ISharedObject::Release() が呼ばれます。
         rhs は何も指さない状態になります。
         ISharedObject::AddReference() の呼び出しによる参照カウンタ操作がないため、効率的です。
    */
    template <typename U>
    SharedPointer& operator=(SharedPointer<U>&& rhs) NN_NOEXCEPT
    {
        static_assert(std::is_base_of<Interface, U>::value, "[SF-BASE-InvalidSharedPointerConversion] not std::is_base_of<Interface, U>::value");
        SharedPointer tmp(std::move(rhs));
        tmp.swap(*this);
        return *this;
    }

    /**
        @brief swap: 指定した SharedPointer と swap します。

        @param[in] other swap の対象を指定します。

        @post this->Get() == swap 前の other.Get()
        @post other.Get() == swap 前の this->Get()
    */
    void swap(SharedPointer& other) NN_NOEXCEPT
    {
        m_Base.swap(other.m_Base);
    }

    /**
        @brief ポインタをデタッチします。

        @return 元々保持していたインターフェイスポインタを返します。

        @post static_cast<bool>(*this) == false

        @details
         \*this が無効化され、それまで指していたポインタを返します。
         参照カウンタは操作されず、参照の管理が単に放棄されるため、返ってきたポインタの管理を呼び出し側で行う必要があります。

         この関数で返ったポインタは ReleaseSharedObject() に渡すことで解放するか、
         SharedPointer(Interface*, bool) コンストラクタに false を伴って渡し、再度 SharedPointer に参照を戻すかのいずれかをする必要があります。
    */
    Interface* Detach() NN_NOEXCEPT
    {
        return static_cast<Interface*>(m_Base.Detach());
    }

    /**
        @brief 共有ポインタを無効化します。

        @details
         \*this が何らかのオブジェクトを指していた場合には、
         そのオブジェクトに対し ISharedObject::Release() が呼ばれます。
    */
    void Reset() NN_NOEXCEPT
    {
        *this = nullptr;
    }

    /**
        @brief (内部用) インターフェイスのポインタを取得します。

        @return インターフェイスポインタを返します。

        @details
         この関数を直接呼ぶ必要は一般にありません。
         内部オブジェクトにアクセスする場合には operator->() を使用してください。
    */
    Interface* Get() const NN_NOEXCEPT
    {
        return static_cast<Interface*>(m_Base.Get());
    }

    /**
        @brief ポインタが有効な場合に Interface への実ポインタを取得します。

        @return Interface への実ポインタを返します。

        @pre static_cast<bool>(*this) == true
    */
    Interface* operator->() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(Get());
        return Get();
    }

    /**
        @brief bool 変換: bool への明示的な変換演算子です。

        @return 有効なポインタであれば true を、そうでない場合には false を返します。
    */
    NN_IMPLICIT operator BoolType() const NN_NOEXCEPT
    {
        return Get() != nullptr ? &SharedPointer::FunctionForBoolType : nullptr;
    }

    /**
        @brief 否定演算子: 否定演算子です。

        @return !static_cast@<bool>(*this) を返します。
    */
    bool operator!() const NN_NOEXCEPT
    {
        return Get() == nullptr;
    }

};

//! @name SharedPointer 外部関数
//! @{

/**
    @brief swap: 二つの SharedPointer を swap します。

    @tparam Interface サービスインターフェイスを指定します。
    @param x swap する一つ目の SharedPointer を指定します。
    @param y swap する二つ目の SharedPointer を指定します。

    @post x.Get() == swap 前の y.Get()
    @post y.Get() == swap 前の x.Get()
*/
template <typename Interface>
void swap(SharedPointer<Interface>& x, SharedPointer<Interface>& y) NN_NOEXCEPT
{
    x.swap(y);
}

// TODO: operator== など

//! @}

//! @name 共有オブジェクトの寿命操作
//! @{

/**
    @brief 共有オブジェクトの参照を減らします。

    @param[in] p 共有オブジェクトへのポインタを指定します。

    @pre p が有効な参照をもつ共有オブジェクトへのポインタである

    @details
     共有オブジェクトへの参照を一つ減らします。
*/
inline void ReleaseSharedObject(ISharedObject* p) NN_NOEXCEPT
{
    p->Release();
}

//! @}

}}
