﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/**
    @file
    @brief サービスフレームワークで HIPC を使用する際に、メッセージバッファを扱う際に使用する API を定義します。
*/

#include <nn/nn_Common.h>

#include <nn/nn_SdkAssert.h>
#include <nn/sf/hipc/sf_HipcTlsBuffer.h>
#include <cstring>

namespace nn { namespace sf {

/**
    @brief HIPC メッセージバッファに必要なアライメントを表す定数です。

    @details
     グローバル変数をこの値でアラインしたい場合には NN_SF_ALIGNAS_HIPC_MESSAGE_BUFFER マクロを使用してください。

    @see NN_SF_ALIGNAS_HIPC_MESSAGE_BUFFER
*/
const size_t HipcMessageBufferAlignment = 4096;

/**
    @brief HIPC メッセージバッファをアラインして配置する際に使用するマクロです。

    @details
     このマクロを付加したグローバル変数は nn::sf::HipcMessageBufferAlignment でアラインされて配置されます。

    @see nn::sf::HipcMessageBufferAlignment
*/
#define NN_SF_ALIGNAS_HIPC_MESSAGE_BUFFER NN_ALIGNAS(4096)

/**
    @brief HIPC 用 TLS メッセージバッファのバイトサイズを表す定数です。
*/
const size_t HipcTlsMessageBufferSize = 256;

//! @name HIPC 用 TLS メッセージバッファアクセス
//! @{

/**
    @brief HIPC 用 TLS メッセージバッファからデータをコピーします。

    @param[out] buffer バッファのコピー先のポインタを指定します。
    @param[in] bufferSize バッファのサイズを指定します。

    @pre bufferSize >= HipcTlsMessageBufferSize
    @pre [buffer, buffer + HipcTlsMessageBufferSize) の領域に書き込みできる
    @pre HIPC 用 TLS メッセージバッファに適切な IPC メッセージが書き込まれている

    @post [buffer, buffer + HipcTlsMessageBufferSize) の領域に IPC メッセージが書き込まれている

    @details
     HIPC 用 TLS メッセージバッファに受信したメッセージを別のスレッドで処理する場合や、
     一旦別のメッセージを受け取りたい際には、この関数を使用して内容を退避することができます。
     この関数で退避した内容は CopyToHipcTlsMessageBuffer() を使用して書き戻すことができます。

    @see CopyToHipcTlsMessageBuffer()
*/
void CopyFromHipcTlsMessageBuffer(void* buffer, size_t bufferSize) NN_NOEXCEPT;

/**
    @brief HIPC 用 TLS メッセージバッファにデータをコピーします。

    @param[in] buffer バッファのコピー元のポインタを指定します。
    @param[in] bufferSize バッファのサイズを指定します。

    @pre bufferSize <= HipcTlsMessageBufferSize
    @pre [buffer, buffer + bufferSize) の領域を読み込める
    @pre [buffer, buffer + bufferSize) の領域に CopyFromHipcTlsMessageBuffer() によってコピーされたデータが書き込まれている

    @post HIPC 用 TLS メッセージバッファに IPC メッセージが書き込まれている

    @details
     CopyFromHipcTlsMessageBuffer() で退避したデータを HIPC 用 TLS メッセージバッファに書き戻します。

    @see CopyFromHipcTlsMessageBuffer()
*/
void CopyToHipcTlsMessageBuffer(const void* buffer, size_t bufferSize) NN_NOEXCEPT;

//! @}

// 以下実装
// TODO: 後に .cpp に移す

static_assert(HipcTlsMessageBufferSize == hipc::MessageBufferSizeOnTls, "[SF-Internal]");

inline void CopyFromHipcTlsMessageBuffer(void* buffer, size_t bufferSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(HipcTlsMessageBufferSize <= bufferSize);
    NN_UNUSED(bufferSize);
    std::memcpy(buffer, hipc::GetMessageBufferOnTls(), HipcTlsMessageBufferSize);
}

inline void CopyToHipcTlsMessageBuffer(const void* buffer, size_t bufferSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(bufferSize <= HipcTlsMessageBufferSize);
    NN_UNUSED(bufferSize);
    std::memcpy(hipc::GetMessageBufferOnTls(), buffer, HipcTlsMessageBufferSize);
}

}}
