﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/**
    @file
    @brief サービスフレームワークで HIPC を使用する際に、セッションハンドルによるプロキシオブジェクト生成を提供します。
    @details
     このファイルをインクルードすることで、以下のファイルで定義されているものを間接的にインクルードします。

     - <nn/sf/sf_HipcPortCommon.h>
     - <nn/sf/sf_HipcClientSession.h>
*/

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/sf/sf_ISharedObject.h>
#include <nn/sf/hipc/sf_HipcHandleTypes.h>
#include <nn/sf/sf_StandardAllocationPolicy.h>
#include <nn/sf/sf_HipcClientSession.h>
#include <nn/sf/sf_HipcPortCommon.h>

#include <nn/sf/hipc/sf_HipcDirectApi.h>
#include <nn/sf/cmif/client/sf_CmifProxyFactory.h>
#include <nn/sf/hipc/client/sf_HipcClientMessage.h>

namespace nn { namespace sf {

//! @name HIPC プロキシオブジェクト生成 (セッション指定)
//! @{

/**
    @brief クライアントセッションハンドルを閉じます。

    @param[in] handle 閉じるハンドルを指定します。
*/
void CloseHipcClientSessionHandle(HipcClientSessionHandle handle) NN_NOEXCEPT;

/**
    @brief セッションに関連づけられたサービスオブジェクトへのプロキシを取得します。

    @tparam Interface サービスの提供するインターフェイス型を指定します。
    @tparam AllocationPolicy 状態なしのアロケーションポリシーを指定します。

    @param[in] handle セッションハンドルを指定します。

    @return プロキシオブジェクトへの共有ポインタを指定します。

    @pre handle で表されるセッションが Interface を実装するオブジェクトに関連付けられている

    @details
     handle で表されるセッションに関連付けられたオブジェクトへのプロキシオブジェクトを生成して返します。
     handle の所有権はプロキシオブジェクトに移動します。
     この関数に渡した handle は閉じないでください。

     多くの場合、この関数を直接呼ぶ必要はありません。
*/
template <typename Interface, typename AllocationPolicy>
HipcRef<Interface> CreateHipcProxyByHandle(HipcClientSessionHandle handle) NN_NOEXCEPT;

/**
    @brief セッションに関連づけられたサービスオブジェクトへのプロキシを取得します。

    @tparam Interface サービスの提供するインターフェイス型を指定します。
    @tparam AllocationPolicy 状態ありのアロケーションポリシーを指定します。

    @param[in] pAllocator アロケータへのポインタを指定します。
    @param[in] handle セッションハンドルを指定します。

    @return プロキシオブジェクトへの共有ポインタを指定します。

    @pre handle で表されるセッションが Interface を実装するオブジェクトに関連付けられている

    @details
     handle で表されるセッションに関連付けられたオブジェクトへのプロキシオブジェクトを生成して返します。
     handle の所有権はプロキシオブジェクトに移動します。
     この関数に渡した handle は閉じないでください。

     多くの場合、この関数を直接呼ぶ必要はありません。
*/
template <typename Interface, typename AllocationPolicy>
HipcRef<Interface> CreateHipcProxyByHandle(typename AllocationPolicy::Allocator* pAllocator, HipcClientSessionHandle handle) NN_NOEXCEPT;

//! @}

// 以下実装

inline void CloseHipcClientSessionHandle(HipcClientSessionHandle handle) NN_NOEXCEPT
{
    hipc::CloseClientSessionHandle(handle);
}

template <typename Interface, typename AllocationPolicy>
inline HipcRef<Interface> CreateHipcProxyByHandle(HipcClientSessionHandle handle) NN_NOEXCEPT
{
    return detail::HipcRefAccessor::CreateHipcProxyByHandle<Interface, AllocationPolicy>(nullptr, handle);
}

template <typename Interface, typename AllocationPolicy>
inline HipcRef<Interface> CreateHipcProxyByHandle(typename AllocationPolicy::Allocator* pAllocator, HipcClientSessionHandle handle) NN_NOEXCEPT
{
    return detail::HipcRefAccessor::CreateHipcProxyByHandle<Interface, AllocationPolicy>(pAllocator, handle);
}

}}
