﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/**
    @file
    @brief サービスフレームワークで使用するヘルパクラス等を定義します。
*/

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Abort.h>
#include <utility>
#include <nn/result/result_HandlingUtility.h>

namespace nn { namespace sf {

/**
    @brief 関数が Result と値を返す際に使用する型です。

    @tparam T 値の型です。

    @details
     このクラステンプレートは Result を継承します。
     また IsSuccess() の際には、T の有効な型を内部に持ち Ref() で参照を取得することができます。

    @see NN_SF_ABORT_UNLESS_RESULT_AND_SUCCESS, NN_SF_RESULT_AND_DO, NN_SF_RESULT_AND_TRY
*/
template <typename T>
class ResultAnd
    : public Result
{
private:

    T m_Value;

public:

    /**
        @brief Result を使用して初期化します。

        @param result 初期化に使用する Result を指定します。

        @pre !result.IsSuccess()
        @post static_cast<Result>(*this) == result
    */
    NN_IMPLICIT ResultAnd(Result result)
        : Result(result)
    {
        NN_SDK_REQUIRES(!result.IsSuccess());
    }

    /**
        @brief 値を使用して初期化します。

        @param x 値を指定します。

        @post this->IsSuccess()
    */
    NN_IMPLICIT ResultAnd(const T& x)
        : Result(ResultSuccess())
        , m_Value(x)
    {
    }

    /**
        @brief 値を使用してムーブ初期化します。

        @param x 値を指定します。

        @post this->IsSuccess()
    */
    explicit ResultAnd(T&& x)
        : Result(ResultSuccess())
        , m_Value(std::move(x))
    {
        // explicit を NN_IMPLICIT にするべきだが、VC++ でコンパイルが通らなくなることがあるため、explicit としている。
    }

    /**
        @brief 保持している値への参照を取得します。

        @return 保持している値への参照を返します。

        @pre this->IsSuccess()
    */
    T& Ref() NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(*this);
        return m_Value;
    }

    /**
        @brief Ref() を返します。

        @return Ref() を返します。

        @pre this->IsSuccess()
    */
    T& operator*() NN_NOEXCEPT
    {
        return Ref();
    }


    /**
        @brief &Ref() を返します。

        @return &Ref() を返します。

        @pre this->IsSuccess()
    */
    T* operator->() NN_NOEXCEPT
    {
        return &Ref();
    }

    /**
        @brief 内部オブジェクトへの右辺値参照を返します。

        @return std::move(Ref()) を返します。

        @pre this->IsSuccess()
    */
    T&& MoveRef() NN_NOEXCEPT
    {
        return std::move(Ref());
    }

private:
    ResultAnd& operator=(const Result& other);
};

#if defined(NN_BUILD_FOR_DOCUMENT_GENERATION)
/**
    @brief nn::sf::ResultAnd に対して NN_ABORT_UNLESS_RESULT_SUCCESS を呼ぶ一方で、成功時には値を取得するためのマクロです。

    @param[out] variable 取得した値を格納する変数を指定します。この名前の変数が定義されます。
    @param[in] exp nn::sf::ResultAnd を返す、実行したい式を指定します。

    @details
     exp の返り値型 nn::sf::ResultAnd<T> に対し、変数 T variable を定義します。
     その上で、exp で表される式を実行し、その Result が成功でなかった場合には、その Result に対し NN_ABORT_UNLESS_RESULT_SUCCESS を実行します。
     成功だった場合には、variable に exp.MoveRef() を代入します。

    @see nn::sf::ResultAnd, NN_ABORT_UNLESS_RESULT_SUCCESS
*/
#define NN_SF_ABORT_UNLESS_RESULT_AND_SUCCESS(variable, exp)

/**
    @brief nn::sf::ResultAnd に対して NN_RESULT_DO を呼ぶ一方で、成功時には値を取得するためのマクロです。

    @param[out] variable 取得した値を格納する変数を指定します。この名前の変数が定義されます。
    @param[in] exp nn::sf::ResultAnd を返す、実行したい式を指定します。

    @details
     exp の返り値型 nn::sf::ResultAnd<T> に対し、変数 T variable を定義します。
     その上で、exp で表される式を実行し、その Result が成功でなかった場合には、その Result に対し NN_RESULT_DO を実行します。
     成功だった場合には、variable に exp.MoveRef() を代入します。

    @see nn::sf::ResultAnd
*/
#define NN_SF_RESULT_AND_DO(variable, exp)

/**
    @brief nn::sf::ResultAnd に対して NN_RESULT_TRY 相当のエラーハンドリングを記述します。

    @param[out] variable 取得した値を格納する変数を指定します。この名前の変数が定義されます。
    @param[in] exp nn::sf::ResultAnd を返す、実行したい式を指定します。

    @details
     exp の返り値型 nn::sf::ResultAnd<T> に対し、変数 T variable を定義します。
     その上で、exp で表される式を実行し、その Result が成功でなかった場合には、その Result に対し NN_RESULT_TRY を展開します。
     このマクロの後には対応する NN_SF_RESULT_AND_END_TRY で閉じる必要があり、
     その間には NN_RESULT_CATCH などのハンドリングマクロを記述することができます。

     NN_RESULT_CATCH などでハンドリングされた際には、変数 variable はデフォルトコンストラクトされたままとなります。
     NN_RESULT_CATCH 内などで NN_SF_RESULT_AND_END_TRY 以下に抜けるような記述をする場合には、
     variable に適切な値を設定することを強くお勧めします。

    @see nn::sf::ResultAnd, NN_SF_RESULT_AND_END_TRY
*/
#define NN_SF_RESULT_AND_TRY(variable, exp)

/**
    @brief NN_SF_RESULT_AND_TRY と組み合わせて使用します。

    @see NN_SF_RESULT_AND_TRY
*/
#define NN_SF_RESULT_AND_END_TRY

#else // of #if defined(NN_BUILD_FOR_DOCUMENT_GENERATION)

#define NN_SF_HANDLE_RESULT_AND(failureHandleMacro, variable, exp) \
    ::std::decay<decltype((exp).Ref())>::type variable; \
    { \
        auto _nn_sf_handle_result_and_tmp = (exp); \
        failureHandleMacro(_nn_sf_handle_result_and_tmp); \
        variable = ::std::move(_nn_sf_handle_result_and_tmp.Ref()); \
    }

#define NN_SF_ABORT_UNLESS_RESULT_AND_SUCCESS(variable, exp) NN_SF_HANDLE_RESULT_AND(NN_ABORT_UNLESS_RESULT_SUCCESS, variable, (exp))

#define NN_SF_RESULT_AND_DO(variable, exp) NN_SF_HANDLE_RESULT_AND(NN_RESULT_DO, variable, (exp))

#define NN_SF_RESULT_AND_TRY(variable, exp) \
    ::std::decay<decltype((exp).Ref())>::type variable; \
    { \
        auto _nn_sf_result_and_try_tmp = (exp); \
        if (_nn_sf_result_and_try_tmp.IsSuccess()) \
        { \
            variable = ::std::move(_nn_sf_result_and_try_tmp.Ref()); \
        } \
        else \
        { \
            NN_RESULT_TRY(static_cast<const ::nn::Result>(_nn_sf_result_and_try_tmp))

#define NN_SF_RESULT_AND_END_TRY \
            NN_RESULT_END_TRY \
        } \
    }

#endif // of #if defined(NN_BUILD_FOR_DOCUMENT_GENERATION)

}}
