﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/**
    @file
    @brief サービスフレームワークで通常使用するアロケーションポリシーとその操作関数を定義します。
*/

#include <nn/nn_Common.h>
#include <nn/nn_Allocator.h>
#include <nn/sf/sf_AllocationPolicies.h>
#include <type_traits>
#include <cstddef>
#include <nn/nn_SdkAssert.h>

namespace nn { namespace sf {

namespace detail {

void* DefaultAllocateImpl(size_t size, size_t alignment, size_t offset) NN_NOEXCEPT;
void DefaultDeallocateImpl(void* p, size_t size, size_t alignment, size_t offset) NN_NOEXCEPT;

template <typename T>
class DefaultAllocationPolicyAllocator
{
private:

    struct Holder
    {
        MemoryResource* pAllocator;
        typename std::aligned_storage<sizeof(T), NN_ALIGNOF(T)>::type x;
    };

public:

    void* Allocate(size_t size) NN_NOEXCEPT
    {
        NN_UNUSED(size);
        NN_SDK_REQUIRES_EQUAL(sizeof(T), size);
        return DefaultAllocateImpl(sizeof(Holder), NN_ALIGNOF(Holder), offsetof(Holder, x));
    }

    void Deallocate(void* p, size_t size) NN_NOEXCEPT
    {
        NN_UNUSED(size);
        NN_SDK_REQUIRES_EQUAL(sizeof(T), size);
        DefaultDeallocateImpl(p, sizeof(Holder), NN_ALIGNOF(Holder), offsetof(Holder, x));
    }

};

}

#if defined(NN_BUILD_FOR_DOCUMENT_GENERATION)
struct UnspecifiedDefaultAllocationPolicyType;
/**
    @brief サービスフレームワークのデフォルトアロケーションポリシーを表す型です。

    @details
     サービスフレームワークの各種 API のアロケーションポリシーとしてに DefaultAllocationPolicy を指定すると、
     サービスフレームワークデフォルトアロケーションポリシーによってアロケーションが行われます。

     単にアロケーションポリシーとして DefaultAllocationPolicy を指定した場合のデフォルトの動作は、
     グローバル operator new/delete (nothrow) の呼び出しとなります。

     これに加えて、必要に応じて、動的にアロケーター(MemoryResource)を指定することができます。

     サービスフレームワークデフォルトアロケーションポリシーでは、以下の二つによって実際に使用するアロケータを制御します。

     - グローバルデフォルトメモリリソース
     - カレントメモリリソース

     カレントメモリリソースの取得と設定は GetCurrentMemoryResource() と SetCurrentMemoryResource() で行うことができます。
     また、多くの場合はユーティリティクラス ScopedCurrentMemoryResourceSetter を使用することで、スコープ内でのみカレントメモリリソースの変更を行います。
     カレントメモリリソースは、スレッドローカルな変数であり、各スレッドごとに別の値を設定できます。
     カレントメモリリソースは、各スレッド開始時には nullptr が設定されています。

     グローバルデフォルトメモリリソースの取得と設定は GetGlobalDefaultMemoryResource() と SetGlobalDefaultMemoryResource() で行うことができます。
     グローバルデフォルトメモリリソースは、プログラムで一つ存在するグローバルな変数であり、多くの場合は、プログラム開始時に指定します。
     グローバルデフォルトメモリリソースは、プログラム開始時には nullptr が設定されています。
     SetGlobalDefaultMemoryResource() による設定はスレッドセーフではありません。
     すなわち SetGlobalDefaultMemoryResource() の呼び出しは、他の nn::sf 以下および nn::sf 以下を使用する関数と同時に実行することはできません。
     このため、プログラム開始直後など、一つのスレッドのみが動いている状況で呼ぶことが強く推奨されます。

     これらに対して、実際に使用する MemoryResource を取得する GetCurrentEffectiveMemoryResource() が以下のように定義されます。

     - GetCurrentMemoryResource() != nullptr のとき GetCurrentMemoryResource() を返す
     - そうでないときで GetGlobalDefaultMemoryResource() != nullptr のとき GetGlobalDefaultMemoryResource() を返す
     - いずれでもないとき GetNewDeleteMemoryResource() を返す

     サービスフレームワークデフォルトアロケーションポリシーを使用する場合、この GetCurrentEffectiveMemoryResource() で返された MemoryResource に対し、
     MemoryResource::allocate() を呼んでメモリを確保し、同じ MemoryResource オブジェクトの MemoryResource::deallocate() によってメモリを破棄します。

     ある MemoryResource を使用してメモリを確保された場合には、そのメモリが解放されるまで、その MemoryResource が有効である必要があります。
     多くの場合、 SetCurrentMemoryResource() や SetGlobalDefaultMemoryResource() に指定する MemoryResource オブジェクトは、
     プログラム開始時に生成して、プログラム終了時までの間、破棄しないことを推奨します。

    @see GetCurrentEffectiveMemoryResource(), ScopedCurrentMemoryResourceSetter, GetCurrentMemoryResource(), SetCurrentMemoryResource(), GetGlobalDefaultMemoryResource(), SetGlobalDefaultMemoryResource()
*/
typedef UnspecifiedDefaultAllocationPolicyType DefaultAllocationPolicy;
#else
typedef StatelessTypedAllocationPolicy<detail::DefaultAllocationPolicyAllocator> DefaultAllocationPolicy;
#endif

//! @name サービスフレームワークデフォルトアロケータ操作
//! @{

/**
    @brief 現在のコンテキストで DefaultAllocationPolicy で実際に使用される MomoryResource を取得します。

    @return 現在のコンテキストで DefaultAllocationPolicy で実際に使用される MomoryResource を返します。

    @details
     詳細は DefaultAllocationPolicy を参照してください。

    @see DefaultAllocationPolicy
*/
MemoryResource* GetCurrentEffectiveMemoryResource() NN_NOEXCEPT;

/**
    @brief サービスフレームワークのグローバルデフォルトメモリリソースを取得します。

    @return グローバルデフォルトメモリリソースを返します。未設定の場合には nullptr を返します。

    @details
     詳細は DefaultAllocationPolicy を参照してください。

    @see DefaultAllocationPolicy
*/
MemoryResource* GetGlobalDefaultMemoryResource() NN_NOEXCEPT;

/**
    @brief サービスフレームワークのグローバルデフォルトメモリリソースを設定します。

    @param[in] pMemoryResource 設定する MemoryResource を指定します。nullptr を指定すると未設定を指定したことになります。
    @return 設定前のグローバルデフォルトメモリリソースを返します。未設定だった場合には nullptr を返します。

    @details
     この関数はプログラムの開始時に呼ぶことを強く推奨します。
     詳細は DefaultAllocationPolicy を参照してください。

    @see DefaultAllocationPolicy
*/
MemoryResource* SetGlobalDefaultMemoryResource(MemoryResource* pMemoryResource) NN_NOEXCEPT;

/**
    @brief サービスフレームワークのカレントメモリリソースを取得します。

    @return カレントメモリリソースを返します。未設定の場合には nullptr を返します。

    @details
     詳細は DefaultAllocationPolicy を参照してください。

    @see DefaultAllocationPolicy
*/
MemoryResource* GetCurrentMemoryResource() NN_NOEXCEPT;

/**
    @brief サービスフレームワークのカレントメモリリソースを設定します。

    @param[in] pMemoryResource 設定する MemoryResource を指定します。nullptr を指定すると未設定を指定したことになります。
    @return 設定前のカレントメモリリソースを返します。未設定だった場合には nullptr を返します。

    @details
     特別な理由がない場合には、この関数を直接呼ぶ代わりに ScopedCurrentMemoryResourceSetter を使用することを推奨します。
     詳細は DefaultAllocationPolicy を参照してください。

    @see DefaultAllocationPolicy
*/
MemoryResource* SetCurrentMemoryResource(MemoryResource* pMemoryResource) NN_NOEXCEPT;

/**
    @brief サービスフレームワークデフォルトアロケータが特別な設定なしの際に使用する、グローバルな operator new/delete を使用するメモリリソースを取得します。

    @return グローバルな operator new/delete を使用するメモリリソースを返します。

    @details
     グローバルな operator new/delete (nothrow) を使用するメモリリソースを返します。
     グローバルデフォルトメモリリソースやカレントメモリリソースが設定されていない際に、このメモリリソースが使用されます。

     この関数の返り値は毎回同じです。
*/
MemoryResource* GetNewDeleteMemoryResource() NN_NOEXCEPT;

//! @}

/**
    @brief スコープ内において、サービスフレームワークのカレントメモリリソースを変更するためのユーティリティです。

    @details
     コンストラクタにおいて、指定した MemoryResource をカレントメモリリソースに設定し、
     デストラクタにおいて、元の MemoryResource をカレントメモリリソースに戻します。

    @see DefaultAllocationPolicy, GetCurrentMemoryResource(), SetCurrentMemoryResource()
*/
class ScopedCurrentMemoryResourceSetter
{
    NN_DISALLOW_COPY(ScopedCurrentMemoryResourceSetter);
    NN_DISALLOW_MOVE(ScopedCurrentMemoryResourceSetter);
public:

    /**
        @brief 指定した MemoryResource で初期化します。

        @param[in] pMemoryResource スコープ内で設定する MemoryResource を指定します。nullptr を指定した場合は、スコープ内で MemoryResource が未設定となります。

        @see SetCurrentMemoryResource()
    */
    explicit ScopedCurrentMemoryResourceSetter(MemoryResource* pMemoryResource) NN_NOEXCEPT;

    /**
        @brief カレントメモリリソースを元に戻します。

        @see GetCurrentMemoryResource()
    */
    ~ScopedCurrentMemoryResourceSetter() NN_NOEXCEPT;

private:

    MemoryResource* m_Previous;

};

}}
