﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>

#include <nn/util/util_BitPack.h>
#include <nn/sf/sf_NativeHandle.h>
#include <nn/sf/cmif/sf_CmifMessageCommon.h>
#include <nn/sf/hipc/detail/sf_HipcHandleRegistration.h>
#include <nn/os/os_NativeHandle.h>

namespace nn { namespace sf { namespace hipc { namespace detail {

enum HipcMessageReceiveBufferMode
{
    HipcMessageReceiveBufferMode_None,
    HipcMessageReceiveBufferMode_MessageBuffer,
    HipcMessageReceiveBufferMode_Single,
    HipcMessageReceiveBufferMode_Multi,
};

typedef cmif::PointerAndSize AddressValueAndSize;

struct MapTransferBufferInfo
{
    uintptr_t pointer;
    size_t size;
    uint32_t mapTransferAttribute;

    cmif::PointerAndSize ToPointerAndSize() const NN_NOEXCEPT
    {
        cmif::PointerAndSize ret = { pointer, size };
        return ret;
    }
};

struct HipcMessageHeaderInfo
{
    bool hasPid;
    int copyHandleCount;
    int moveHandleCount;
    int pointerCount;
    int sendCount;
    int receiveCount;
    int exchangeCount;
    size_t rawDataByteSize;
    int receiveListCount;
    HipcMessageReceiveBufferMode receiveBufferMode;
};

class HipcMessageBufferAccessor
{
private:

    char* m_MessageBuffer;
    size_t m_MessageBufferSize;

    bool SetHeadPointers(const HipcMessageHeaderInfo& headerInfo) NN_NOEXCEPT;

    size_t m_MessageSize;
    nn::util::BitPack32* m_HeaderHead;
    nn::util::BitPack32* m_SpecialHeaderHead;
    nn::util::BitPack32* m_PidHead;
    nn::util::BitPack32* m_CopyHandleHead;
    nn::util::BitPack32* m_MoveHandleHead;
    nn::util::BitPack32* m_PointerHead;
    nn::util::BitPack32* m_SendHead;
    nn::util::BitPack32* m_ReceiveHead;
    nn::util::BitPack32* m_ExchangeHead;
    char* m_RawDataHead;
    nn::util::BitPack32* m_ReceiveListHead;

public:

    typedef cmif::PointerAndSize PointerAndSize;

    static Bit16 GetTag(void *messageBuffer, size_t messageBufferSize) NN_NOEXCEPT;

    bool SetupHeader(void *messageBuffer, size_t messageBufferSize, const HipcMessageHeaderInfo& headerInfo) NN_NOEXCEPT;
    bool ParseHeader(HipcMessageHeaderInfo* pHeaderInfo, void *messageBuffer, size_t messageBufferSize) NN_NOEXCEPT;

    size_t GetMessageSize() const
    {
        return m_MessageSize;
    }

    void SetTag(Bit16 tag) NN_NOEXCEPT;
    Bit16 GetTag() const NN_NOEXCEPT;

    void SetProcessId(uint64_t processId) NN_NOEXCEPT;
    uint64_t GetProcessId() const NN_NOEXCEPT;

    void SetCopyHandle(int index, hipc::detail::InternalHandleValue handle) NN_NOEXCEPT;
    hipc::detail::InternalHandleValue GetCopyHandle(int index) const NN_NOEXCEPT;

    void SetMoveHandle(int index, InternalHandleValue handle) NN_NOEXCEPT;
    InternalHandleValue GetMoveHandle(int index) const NN_NOEXCEPT;

    void SetPointer(int index, InternalAddressValue addressValue, size_t size, int receiveIndex) NN_NOEXCEPT;
    AddressValueAndSize GetPointer(int index) const NN_NOEXCEPT;
    int GetPointerReceiveIndex(int index) const NN_NOEXCEPT;

    void SetSend(int index, InternalAddressValue addressValue, size_t size, uint32_t mapTransferAttribute) NN_NOEXCEPT;
    MapTransferBufferInfo GetSend(int index) const NN_NOEXCEPT;
    void SetReceive(int index, InternalAddressValue addressValue, size_t size, uint32_t mapTransferAttribute) NN_NOEXCEPT;
    MapTransferBufferInfo GetReceive(int index) const NN_NOEXCEPT;
    void SetExchange(int index, InternalAddressValue addressValue, size_t size, uint32_t mapTransferAttribute) NN_NOEXCEPT;
    MapTransferBufferInfo GetExchange(int index) const NN_NOEXCEPT;

    uintptr_t GetRawPointer() const NN_NOEXCEPT;

    void SetReceiveList(int index, InternalAddressValue addressValue, size_t size) NN_NOEXCEPT;
    AddressValueAndSize GetReceiveList(int index) const NN_NOEXCEPT;

};

// utility
inline cmif::PointerAndSize UnregisterAddress(AddressValueAndSize avas) NN_NOEXCEPT
{
    cmif::PointerAndSize ret = { hipc::detail::UnregisterAddress(avas.pointer), avas.size };
    return ret;
}

}}}} // ~ nn::sf::hipc::detail


