﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/sf/cmif/client/sf_CmifClientMessage.h>
#include <nn/sf/hipc/client/sf_HipcClientMessage.h>

namespace nn { namespace sf { namespace hipc { namespace client {

class HipcClientSessionAllocator
{
public:

    virtual Hipc2ProxyKind::ObjectInfo Allocate() NN_NOEXCEPT = 0;
    virtual void Deallocate(Hipc2ProxyKind::ObjectInfo info) NN_NOEXCEPT = 0;

    uint16_t GetPointerBufferSize() const NN_NOEXCEPT
    {
        return m_PointerBufferSize;
    }

protected:

    void SetPointerBufferSize(uint16_t pointerBufferSize) NN_NOEXCEPT
    {
        this->m_PointerBufferSize = pointerBufferSize;
    }

    ~HipcClientSessionAllocator()
    {
    }

private:

    uint16_t m_PointerBufferSize;

};

class HipcClientSessionManagedProxyBaseObject
    : public cmif::client::CmifBaseObject
{
private:

    HipcClientSessionAllocator* m_pClientSessionAllocator;

public:

    HipcClientSessionManagedProxyBaseObject() NN_NOEXCEPT
        : m_pClientSessionAllocator(nullptr)
    {
    }

    explicit HipcClientSessionManagedProxyBaseObject(HipcClientSessionAllocator* pClientSessionAllocator) NN_NOEXCEPT
        : m_pClientSessionAllocator(pClientSessionAllocator)
    {
    }

    HipcClientSessionAllocator* GetClientSessionAllocator() const NN_NOEXCEPT
    {
        return m_pClientSessionAllocator;
    }

};

template <typename HipcProxyKind>
struct Hipc2ClientSessionManagedProxyKindBase
{
    typedef HipcClientSessionAllocator* ObjectInfo;
    typedef HipcClientSessionManagedProxyBaseObject ProxyBaseObject;
    typedef HipcClientSessionAllocator* ProcessorArgument;

    class CmifClientMessage
        : public HipcProxyKind::CmifClientMessage
    {
    private:

        typedef typename HipcProxyKind::CmifClientMessage Base;
        HipcClientSessionAllocator* m_pClientSessionAllocator;

    public:

        NN_IMPLICIT CmifClientMessage(ObjectInfo objectInfo) NN_NOEXCEPT
            : Base(objectInfo->Allocate())
            , m_pClientSessionAllocator(objectInfo)
        {
            Base::SetPointerBufferSize(objectInfo->GetPointerBufferSize());
        }

        explicit CmifClientMessage(ProxyBaseObject *pProxy) NN_NOEXCEPT
            : Base(pProxy->GetClientSessionAllocator()->Allocate())
            , m_pClientSessionAllocator(pProxy->GetClientSessionAllocator())
        {
        }

        ~CmifClientMessage() NN_NOEXCEPT
        {
            auto clientHandle = Base::GetClientHandle();
            m_pClientSessionAllocator->Deallocate(clientHandle);
        }

        void AttachOutObjects(cmif::client::CmifBaseObject* outObjects[]) const NN_NOEXCEPT
        {
            NN_UNUSED(outObjects);
            NN_ABORT("[SF-CMIF-Unexpected]");
        }
    };
};

typedef Hipc2ClientSessionManagedProxyKindBase<Hipc2ProxyKind> Hipc2ClientSessionManagedProxyKind;

template <typename BaseProxyKind, typename CoreMethodInfo, typename ArgumentInfos>
class HipcSessionManagedProxyClientCoreProcessor;

template <typename BaseProxyKind, typename CoreMethodInfo, typename... ArgumentInfos_>
class HipcSessionManagedProxyClientCoreProcessor<BaseProxyKind, CoreMethodInfo, std::tuple<ArgumentInfos_...>>
{
private:

    using ProxyKind = sf::hipc::client::Hipc2ClientSessionManagedProxyKindBase<BaseProxyKind>;
    using ProxyBaseObject = typename ProxyKind::ProxyBaseObject;
    using ProcessorArgument = typename ProxyKind::ProcessorArgument;
    using BaseClientProcessor = cmif::client::ClientCoreProcessor<BaseProxyKind, CoreMethodInfo>;

    NN_FORCEINLINE static Result ProcessImpl(ProxyBaseObject* p, typename cmif::client::GetClientArgumentType<ArgumentInfos_>::type... args, cmif::MethodId methodId, SharedPointer<IServiceObject> pOutObjects[]) NN_NOEXCEPT
    {
        return ProcessImpl(p->GetClientSessionAllocator(), std::forward<typename cmif::client::GetClientArgumentType<ArgumentInfos_>::type>(args)..., methodId, pOutObjects, 0, 0);
    }

public:

    NN_FORCEINLINE static Result ProcessImpl(ProcessorArgument pAllocator, typename cmif::client::GetClientArgumentType<ArgumentInfos_>::type... args, cmif::MethodId methodId, SharedPointer<IServiceObject> pOutObjects[], size_t inRawPadding, size_t outRawPadding) NN_NOEXCEPT
    {
        auto handle = pAllocator->Allocate();
        auto pointerBufferSize = pAllocator->GetPointerBufferSize();
        NN_UTIL_SCOPE_EXIT
        {
            pAllocator->Deallocate(handle);
        };
        return BaseClientProcessor::ProcessImpl({handle, pointerBufferSize}, std::forward<typename cmif::client::GetClientArgumentType<ArgumentInfos_>::type>(args)..., methodId, pOutObjects, inRawPadding, outRawPadding);
    }

    NN_NOINLINE static Result Process(HipcClientSessionAllocator* p, typename cmif::client::GetClientArgumentType<ArgumentInfos_>::type... args, cmif::MethodId methodId) NN_NOEXCEPT
    {
        return ProcessImpl(p, std::forward<typename cmif::client::GetClientArgumentType<ArgumentInfos_>::type>(args)..., methodId, static_cast<SharedPointer<IServiceObject>*>(nullptr));
    }

    NN_NOINLINE static Result ProcessWithOutObjects(ProxyBaseObject* p, typename cmif::client::GetClientArgumentType<ArgumentInfos_>::type... args, cmif::MethodId methodId, SharedPointer<IServiceObject> pOutObjects[]) NN_NOEXCEPT
    {
        return ProcessImpl(p, std::forward<typename cmif::client::GetClientArgumentType<ArgumentInfos_>::type>(args)..., methodId, pOutObjects);
    }

};

}}}}

namespace nn { namespace sf { namespace cmif { namespace client {

template <typename HipcProxyKind, typename CoreMethodInfo, typename... ArgumentInfos_>
struct ClientCoreProcessor<sf::hipc::client::Hipc2ClientSessionManagedProxyKindBase<HipcProxyKind>, CoreMethodInfo, std::tuple<ArgumentInfos_...>>
    : public sf::hipc::client::HipcSessionManagedProxyClientCoreProcessor<HipcProxyKind, CoreMethodInfo, std::tuple<ArgumentInfos_...>>
{
};

}}}}

