﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cstdint>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <cstring>
#include <nn/sf/detail/sf_CommonUtil.h>
#include <nn/sf/cmif/sf_InlineContext.h>

namespace nn { namespace sf { namespace cmif {

NN_PRAGMA_PUSH_WARNINGS
NN_DISABLE_WARNING_SHADOW

typedef std::uint32_t MethodId;
const std::uint32_t InvalidMethodId = UINT32_MAX;

/**
    @berif cmif in ヘッダフォーマット構造体
*/
struct CmifInHeader
{
    nn::Bit32 signature;
    std::uint16_t version;
    char _reserved1[2];
    MethodId methodId;
    uint32_t envelopData;

    static const nn::Bit32 Signature = 0x49434653; // "SFCI" (LE)

    NN_SF_DETAIL_CONSTEXPR static CmifInHeader Create(MethodId methodId) NN_NOEXCEPT
    {
        CmifInHeader inHeader = {};
        inHeader.signature = Signature;
        inHeader.version = 1;
        inHeader.methodId = methodId;
        return inHeader;
    }

    bool Check() const NN_NOEXCEPT
    {
        return true
            && this->signature == Signature;
    }
};

static_assert(sizeof(CmifInHeader) == 16, "[SF-Internal]");
static_assert(NN_ALIGNOF(CmifInHeader) == 4, "[SF-Internal]");

/**
    @berif cmif out ヘッダフォーマット構造体
*/
struct CmifOutHeader
{
    nn::Bit32 signature;
    std::uint16_t version;
    char _reserved1[2];
    nn::Result result;
    char _reserved2[4];

    static const nn::Bit32 Signature = 0x4F434653; // "SFCO" (LE)

    void Set(nn::Result result) NN_NOEXCEPT
    {
        std::memset(this, 0, sizeof(*this));
        this->signature = Signature;
        this->version = 0;
        this->result = result;
    }

    bool Check() const NN_NOEXCEPT
    {
        return true
            && this->signature == Signature;
    }
};

static_assert(sizeof(CmifOutHeader) == 16, "[SF-Internal]");
static_assert(NN_ALIGNOF(CmifOutHeader) == 4, "[SF-Internal]");

NN_PRAGMA_POP_WARNINGS

}}}
