﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/sf/sf_Types.h>
#include <nn/sf/cmif/sf_CmifDomainCommon.h>
#include <nn/sf/cmif/sf_ICmifDomain.h>
#include <nn/sf/cmif/server/sf_CmifServerObjectInfo.h>
#include <nn/sf/cmif/server/sf_CmifDomainServerObject.h>

#include <type_traits>
#include <nn/os/os_Mutex.h>
#include <nn/util/util_IntrusiveList.h>
#include <nn/sf/impl/detail/sf_ServiceObjectImpl.h>

namespace nn { namespace sf { namespace cmif { namespace server {

class CmifServerDomainManager
{
private:

    class Domain;
    struct Entry;

    typedef util::IntrusiveListBaseNode<Entry, Domain> EntryNode;
    typedef util::IntrusiveListBaseNodeTraits<Entry, Domain> EntryNodeTraits;

    struct Entry
        : public util::IntrusiveListBaseNode<Entry, Domain>
    {
        Entry* pNext;
        Domain* pOwner;
        CmifServerObjectInfo target;
    };
    mutable os::Mutex m_EntryOwnerMutex;

    class Domain final
        : public CmifDomainServerObject
        , private impl::detail::ServiceObjectImplBase2
    {
        friend class CmifServerDomainManager;
    private:

        CmifServerDomainManager* m_pManager;
        util::IntrusiveList<Entry, EntryNodeTraits> m_Entries;

        // overrides CmifDomainServerObject::ICmifDomain (ISharedObject)
        virtual void AddReference() NN_NOEXCEPT NN_OVERRIDE
        {
            ServiceObjectImplBase2::AddReferenceImpl();
        }

        virtual void Release() NN_NOEXCEPT NN_OVERRIDE
        {
            if (ServiceObjectImplBase2::ReleaseImpl())
            {
                DisposeImpl();
            }
        }

        // overrides CmifDomainServerObject
        virtual CmifServerDomain* GetDomain() NN_NOEXCEPT NN_OVERRIDE
        {
            return this;
        }

        explicit Domain(CmifServerDomainManager* pManager) NN_NOEXCEPT;
        ~Domain() NN_NOEXCEPT;

        // overrides server::CmifServerDomain
        virtual Result ReserveEntry(CmifDomainObjectId ids[], int count) NN_NOEXCEPT NN_OVERRIDE;
        virtual void UnReserveEntry(const CmifDomainObjectId ids[], int count) NN_NOEXCEPT NN_OVERRIDE;
        virtual void RegisterObject(CmifDomainObjectId id, server::CmifServerObjectInfo&& x) NN_NOEXCEPT NN_OVERRIDE;
        virtual CmifServerObjectInfo UnregisterObject(CmifDomainObjectId id) NN_NOEXCEPT NN_OVERRIDE;
        virtual CmifServerObjectInfo GetObject(CmifDomainObjectId id) NN_NOEXCEPT NN_OVERRIDE;

        void DisposeImpl() NN_NOEXCEPT;

    };

public:

    typedef std::aligned_storage<sizeof(Domain), NN_ALIGNOF(Domain)>::type DomainStorage;
    typedef std::aligned_storage<sizeof(Entry), NN_ALIGNOF(Entry)>::type EntryStorage;

protected:

    CmifServerDomainManager(EntryStorage entryBuffer[], int entryCount) NN_NOEXCEPT;

    CmifDomainServerObject* CreateDomain() NN_NOEXCEPT;

private:

    virtual void* AllocateDomainStorage() NN_NOEXCEPT = 0;
    virtual void DeallocateDomainStorage(void* p) NN_NOEXCEPT = 0;

    class EntryAllocator
    {
    private:
        int m_EntryCount;
        Entry* m_Entries;
        mutable os::Mutex m_Mutex;
        Entry* m_FreeEntryHead;
    public:
        EntryAllocator(EntryStorage entryBuffer[], int entryCount) NN_NOEXCEPT;
        ~EntryAllocator() NN_NOEXCEPT;
        Entry* AllocateEntry() NN_NOEXCEPT;
        void DeallocateEntry(Entry* p) NN_NOEXCEPT;
        CmifDomainObjectId GetId(Entry* p) NN_NOEXCEPT;
        Entry* GetEntry(CmifDomainObjectId id) NN_NOEXCEPT;
    };

    EntryAllocator m_EntryAllocator;

};

}}}}
