﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/**
 * @file
 * @brief   デバッグモニタ向けの API を宣言するファイルです
 */

#include <nn/nn_Result.h>
#include <nn/os/os_Types.h>
#include <nn/os/os_NativeHandle.h>
#include <nn/svc/svc_Types.h>
#include <nn/ncm/ncm_ProgramId.h>

namespace nn { namespace pm {

    namespace detail
    {
        const char PortNameForDebugMonitor[] = "pm:dmnt";
    }

    /**
        @brief  プロセスを構成するモジュールのモジュール ID を表します
    */
    struct ModuleId
    {
        Bit8    value[16];
    };

    /**
        @brief  起動フックのタイプを表します。
    */
    enum HookType
    {
        HookType_ProgramId      = (1 << 0),     //!< プログラム ID を指定したフックを表します
        HookType_Application    = (1 << 1)      //!< アプリケーションを対象としたフックを表します
    };



//! @name 初期化と終了
//! @{

    /**
        @brief  ライブラリの使用を開始します

        @retresult
            @handleresult{nn::os::Result,}
        @endretresult

        @pre
            - ライブラリの使用が開始されていない

        @post
            - 返り値が ResultSuccess なら、ライブラリの使用が開始されている
    */
    Result InitializeForDebugMonitor() NN_NOEXCEPT;

    /**
        @brief  ライブラリの使用を終了します

        @retresult
            @handleresult{nn::os::Result,}
        @endretresult

        @details
            未実装です
    */
    Result FinalizeForDebugMonitor() NN_NOEXCEPT;

//! @}



//! @name プロセス操作
//! @{

    /**
        @brief  実行を開始していないプロセスの実行を開始します

        @param[in]  id  対象のプロセスを指定するプロセス ID

        @retresult
            @handleresult{nn::os::Result,}
        @endretresult

        @pre
            - id で指定するプロセスの状態が Initializing または PreAttached である

        @post
            - 返り値が ResultSuccess なら、id で指定したプロセスの実行が開始されている
    */
    Result StartProcess(os::ProcessId id) NN_NOEXCEPT;

//! @}



//! @name 情報取得
//! @{

    /**
        @brief  WaitAttach 状態のプロセスのリストを取得します

        @param[out] pStoredCount        pBuffer に格納されたプロセス ID の個数
        @param[out] pBuffer             WaitAttach 状態のプロセスのプロセス ID を格納するバッファ
        @param[in]  bufferEntryCount    pBuffer で指定される配列の要素数

        @retresult
            @handleresult{nn::os::Result,}
        @endretresult

        @post
            - ResultSuccess であれば
                - *pStoredCount >= 0
                - pBuffer[i] (0 <= i < *pStoredCount) は WaitAttach 状態のプロセスのプロセス ID
    */
    Result GetExceptionProcessIdList(int* pStoredCount, os::ProcessId* pBuffer, int bufferEntryCount) NN_NOEXCEPT;

    /**
        @brief  対象のプログラムから作成されたプロセスのプロセス ID を取得します

        @param[out] pOut        取得したプロセス ID を格納するバッファ
        @param[in]  programId   対象のプログラムを指定するプログラム ID

        @retresult
            @handleresult{nn::os::Result,}
        @endretresult

        @post
            - ResultSuccess であれば、*pOut は programId で指定されるプログラムから作成されたプロセスのプロセス ID

        @details
            同一のプログラムから作成されたプロセスが複数存在する場合は
            そのうちのいずれか一つのプロセスのプロセス ID を取得します。
    */
    Result GetProcessId(os::ProcessId* pOut, ncm::ProgramId programId) NN_NOEXCEPT;

    /**
        @brief  アプリケーションプログラムから作成されたプロセスのプロセス ID を取得します。

        @param[out] pOut        取得したプロセス ID を格納するバッファ

        @retresult
            @handleresult{nn::os::Result,}
        @endretresult

        @post
            - ResultSuccess であれば、*pOut はアプリケーションプログラムから作成されたプロセスのプロセス ID

        @details
            アプリケーションプログラムから作成されたプロセスが複数存在する場合は
            そのうちのいずれか一つのプロセスのプロセス ID を取得します。
    */
    Result GetApplicationProcessId(os::ProcessId* pOut) NN_NOEXCEPT;

//! @}



//! @name 起動フック
//! @{

    /**
        @brief  指定したプログラムの起動をフックします

        @param[out] pOut        指定したプログラムの起動が行われたときに
                                シグナルされる SystemEvent のハンドルを格納するバッファ。
                                この SystemEvent は os::EventClearMode_ManualClear で作成されています。
        @param[in]  programId   フックする対象のプログラムを指定するプログラム ID

        @retresult
            @handleresult{nn::os::Result,}
        @endretresult

        @pre
            - HookToCreateProcess を呼び出し中ではない

        @post
            - ResultSuccess ならば
                - programId を対象とした起動フックがセットされている
                - *pOut は SystemEvent のハンドル

        @details
            programId で指定したプログラムの起動が行われるときに、
            プログラム起動時のオプションに関わらず、
            作成されたプロセスの状態を Initializing に留めるようにし、
            加えて、*pOut で返される SystemEvent をシグナルするようにします。

            *pOut で返される SystemEvent がシグナルされていても
            指定したプログラムが起動されたとは限らないことに注意してください。
    */
    Result HookToCreateProcess(os::NativeHandle* pOut, ncm::ProgramId programId) NN_NOEXCEPT;

    /**
        @brief  アプリケーションの起動をフックします

        @param[out] pOut        アプリケーションの起動が行われたときに
                                シグナルされる SystemEvent のハンドルを格納するバッファ。
                                この SystemEvent は os::EventClearMode_ManualClear で作成されています。

        @retresult
            @handleresult{nn::os::Result,}
        @endretresult

        @pre
            - HookToCreateApplicationProcess を呼び出し中ではない

        @post
            - ResultSuccess ならば
                - アプリケーションを対象とした起動フックがセットされている
                - *pOut は SystemEvent のハンドル

        @details
            アプリケーションの起動が行われるときに、起動時のオプションに関わらず、
            作成されたプロセスの状態を Initializing に留めるようにし、
            加えて、*pOut で返される SystemEvent をシグナルするようにします。

            *pOut で返される SystemEvent がシグナルされていても
            アプリケーションが起動されたとは限らないことに注意してください。
    */
    Result HookToCreateApplicationProcess(os::NativeHandle* pOut) NN_NOEXCEPT;

    /**
        @brief  起動フックをクリアします

        @param[in]  hookTypes   クリアする起動フックのタイプを指定します。
                                HookType 列挙隊のメンバのビットごとの OR を取る形で指定します。

        @retresult
            @handleresult{nn::os::Result,}
        @endretresult

        @post
            - ResultSuccess ならば hookTypes で指定した起動フックがセットされていない

        @details
            HookToCreateProcess または HookToCreateApplicationProcess でセットした起動フックをクリアします。

            起動フックをクリアしてもそれぞれの関数で返される SystemEvent の状態は変化しません。
    */
    Result ClearHook(int hookTypes) NN_NOEXCEPT;

//! @}

}}  // namespace nn::pm

