﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   セキュアでないメモリの確保に関する公開ヘッダファイル（SDK 開発者向け）
 */

#pragma once

#include <nn/nn_Common.h>
#include <nn/os/os_SdkResult.h>
#include <nn/os/os_SdkThreadInfoTypes.h>
#include <nn/os/os_ThreadTypes.h>

namespace nn { namespace os {

//! @name セキュアでないメモリ確保関連 API（SDK 開発者専用）
//! @{

//-----------------------------------------------------------------------------
/**
 * @brief   セキュアでないメモリを確保します
 *
 * @param[out] pOutAddress  確保したセキュアでないメモリの先頭アドレスを格納します
 * @param[in]  size         確保するメモリサイズを指定します
 *
 * @retresult
 *   @handleresult{nn::os::ResultOutOfVirtualAddressSpace,仮想アドレス空間不足によりアドレス領域の確保に失敗しました。}
 *   @handleresult{nn::os::ResultOutOfMemory,割り当てる物理メモリの確保に失敗しました。}
 *   @handleresult{nn::os::ResultOutOfResource,物理メモリ操作に必要なリソースが不足しました。}
 * @endretresult
 *
 * @pre
 *  - size が nn::os::MemoryPageSize の整数倍である
 *
 * @details
 *  指定されたサイズのセキュアでないメモリを確保します。
 *
 *  本 API は、セキュアなメモリを利用するシステムプロセスが、
 *  一部セキュアでないメモリを利用する場合を想定しています。
 *
 */
Result AllocateUnsafeMemory(uintptr_t* pOutAddress, size_t size) NN_NOEXCEPT;

//-----------------------------------------------------------------------------
/**
 * @brief   セキュアでないメモリを解放します
 *
 * @param[in]  address  解放するメモリの先頭アドレスを指定します
 * @param[in]  thread   解放するメモリサイズを指定します
 *
 * @retresult
 *   @handleresult{nn::os::ResultInvalidParameter,address から size バイトの領域が有効なアドレス領域に含まれていません。}
 *   @handleresult{nn::os::ResultBusy,アドレス領域の中に使用中のメモリが含まれています。}
 *   @handleresult{nn::os::ResultOutOfResource,物理メモリ操作に必要なリソースが不足しました。}
 * @endretresult
 *
 * @pre
 *  - address が AllocateUnsafeMemory で取得したアドレスである
 *  - size が nn::os::MemoryPageSize の整数倍である
 *
 * @details
 *  指定されたセキュアでないメモリを解放します。
 *
 */
Result FreeUnsafeMemory(uintptr_t address, size_t size) NN_NOEXCEPT;

//! @}

//-----------------------------------------------------------------------------

}} // namespace nn::os

