﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/net/osl/osl_Mbuf.h>

namespace nn {
namespace net {
namespace osl {

class MbufPool
{
public:
    typedef uint16_t    IndexInPool;
    typedef int32_t     FreeIndex;
    class SharedArea;

    static const uint16_t INVALID_INDEX = 0xffff;

    MbufPool();
    ~MbufPool();

    void Initialize(uint16_t id, int16_t unitSize, int32_t count, uintptr_t base, bool bServer);
    void Finalize();

    int32_t GetMaxCount() const;
    int32_t GetFreeCount() const;

    size_t GetUnitSize() const;
    size_t GetDataSize() const;
    size_t GetUsingSize() const;

    Mbuf* TryAllocate();
    void Free(Mbuf* pMbuf);

    Mbuf* GetPtrFromIndex(int32_t index);
    IndexInPool GetIndexFromPtr(const Mbuf* pMbuf) const;

    static size_t GetRequiredMemorySize(int16_t unitSize, int32_t count);

    bool CheckClean();

private:
    void ConstructFreeChain();

    std::atomic<uint64_t>* GetFreeIndexPtr();

    void IncrementUsed();
    void DecrementUsed();

    SharedArea*             m_pSharedArea;
    uintptr_t               m_base;
    int32_t                 m_count;
    int16_t                 m_unitSize;
    uint16_t                m_id;
};

class MbufPool::SharedArea
{
public:
    SharedArea(size_t unitSize, int32_t count);
    bool CheckInitialized() const;

    uint32_t    m_magic;
    size_t      m_unitSize;
    int32_t     m_maxCount;
    std::atomic<int32_t> m_usedCount;
    std::atomic<uint64_t> m_taggedFreeIndex;

private:
    static const uint32_t MAGIC   = 0x0b0f0b0f;
};

inline size_t MbufPool::GetUnitSize() const
{
    return m_unitSize;
}

inline size_t MbufPool::GetDataSize() const
{
    return m_unitSize - Mbuf::HEADER_SIZE;
}

inline Mbuf* MbufPool::GetPtrFromIndex(int32_t index)
{
    NN_SDK_ASSERT(0 <= index && index < m_count, "index = %d, m_count = %d", index, m_count);
    return reinterpret_cast<Mbuf*>(m_base + m_unitSize * index);
}

inline std::atomic<uint64_t>* MbufPool::GetFreeIndexPtr()
{
    return &m_pSharedArea->m_taggedFreeIndex;
}

inline size_t MbufPool::GetUsingSize() const
{
    return GetRequiredMemorySize(m_unitSize, m_count);
}

inline int32_t MbufPool::GetMaxCount() const
{
    return m_pSharedArea->m_maxCount;
}

inline int32_t MbufPool::GetFreeCount() const
{
    return m_pSharedArea->m_maxCount - m_pSharedArea->m_usedCount;
}

}
}
}
