﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_TimeSpan.h>
#include <nn/os/os_Event.h>
#include <nn/os/os_Mutex.h>
#include <nn/net/osl/osl_CLibraryUtil.h>

typedef enum
{
    NN_NET_OSL_WAITMODE_AND,
    NN_NET_OSL_WAITMODE_OR
} nnnetOslWaitMode;

#ifdef __cplusplus

namespace nn { namespace net { namespace osl{

class EventFlag
{
public:
    enum WaitMode
    {
        ModeAnd = NN_NET_OSL_WAITMODE_AND,
        ModeOr  = NN_NET_OSL_WAITMODE_OR
    };
    static const uint32_t  INITIAL_ALLOCATE    = 0x3;
    static const size_t MAX_FLAG_BITS       = 4;
    static const uint32_t  VALID_MASK          = (1 << MAX_FLAG_BITS) - 1;

    EventFlag()
        : m_validMap(0)
    {
    }

    ~EventFlag()
    {
        Finalize();
    }

    void Initialize(void);
    Result TryInitialize(void);
    void Finalize(void);

    void Signal(uint32_t pattern);
    uint32_t Wait(uint32_t pattern, WaitMode mode, nn::TimeSpan timeout);
    uint32_t Wait(uint32_t pattern, WaitMode mode);

    uint32_t WaitAndClear(uint32_t pattern, WaitMode mode, uint32_t clear, nn::TimeSpan timeout);
    uint32_t WaitAndClear(uint32_t pattern, WaitMode mode, uint32_t clear);

    void ClearSignal(uint32_t pattern);

protected:
    Result AllocateEvents(uint32_t bitmap);
    void FreeEvents(uint32_t bitmap);

private:
    enum TimeoutMode
    {
        TimeoutMode_Timed,
        TimeoutMode_Infinite
    };

    uint32_t Wait(uint32_t pattern, WaitMode mode, TimeoutMode timeoutMode, nn::TimeSpan timeout);
    uint32_t WaitAndClear(uint32_t pattern, WaitMode mode, uint32_t clear, TimeoutMode timeoutMode, nn::TimeSpan timeout);

    nn::os::MutexType          m_cs;
    nn::os::EventType          m_events[MAX_FLAG_BITS];
    uint32_t                   m_validMap;
};

inline uint32_t EventFlag::Wait(uint32_t pattern, WaitMode mode)
{
    return Wait(pattern, mode, TimeoutMode_Infinite, 0);
}

inline uint32_t EventFlag::Wait(uint32_t pattern, WaitMode mode, nn::TimeSpan timeout)
{
    return Wait(pattern, mode, TimeoutMode_Timed, timeout);
}

inline uint32_t EventFlag::WaitAndClear(uint32_t pattern, WaitMode mode, uint32_t clear)
{
    return WaitAndClear(pattern, mode, clear, TimeoutMode_Infinite, 0);
}

inline uint32_t EventFlag::WaitAndClear(uint32_t pattern, WaitMode mode, uint32_t clear, nn::TimeSpan timeout)
{
    return WaitAndClear(pattern, mode, clear, TimeoutMode_Timed, timeout);
}

}}} // namesapce nn::net::osl

#endif // __cplusplus


// TORIAEZU: C言語を想定したインターフェースに std::aligned_storage を使う
typedef std::aligned_storage<sizeof(nn::net::osl::EventFlag), std::alignment_of<nn::net::osl::EventFlag>::value>::type nnnetOslEventFlag;
const int64_t NN_NET_OSL_TIMEOUT_INFINITY = -1;

//#include <nn/util/detail/util_CLibImpl.h>
//
//#define NN_NET_OSL_EVENTFLAG_SIZE                   (NN_OS_CRITICALSECTION_SIZE + (4 + 1) * 4)
//#define NN_NET_OSL_EVENTFLAG_ALIGNMENT_HOLDER_TYPE  u32
//
//NN_UTIL_DETAIL_CLIBIMPL_DEFINE_BUFFER_CLASS(nnnetOslEventFlag, nn::net::osl::EventFlag, NN_NET_OSL_EVENTFLAG_SIZE, NN_NET_OSL_EVENTFLAG_ALIGNMENT_HOLDER_TYPE);
//
NN_EXTERN_C void nnnetOslEventFlagInitialize(nnnetOslEventFlag* this_);
NN_EXTERN_C bool nnnetOslEventFlagTryInitialize(nnnetOslEventFlag* this_);
NN_EXTERN_C void nnnetOslEventFlagSignal(nnnetOslEventFlag* this_, uint32_t pattern);
NN_EXTERN_C uint32_t nnnetOslEventFlagWaitSignal(nnnetOslEventFlag* this_, uint32_t pattern, nnnetOslWaitMode mode, int64_t timeout);
NN_EXTERN_C void nnnetOslEventFlagClearSignal(nnnetOslEventFlag* this_, uint32_t pattern);
NN_EXTERN_C uint32_t nnnetOslEventFlagWaitAndClear(nnnetOslEventFlag* this_, uint32_t pattern, nnnetOslWaitMode mode, uint32_t clear, int64_t timeout);
NN_EXTERN_C void nnnetOslEventFlagFinalize(nnnetOslEventFlag* this_);
