﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <memory>
#include <nn/nn_Common.h>
#include <nn/nn_TimeSpan.h>
#include <nn/crypto/crypto_Sha256Generator.h>
#include <nn/fs/fs_RightsId.h>
#include <nn/os/os_Mutex.h>
#include <nn/util/util_Optional.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/ncm/ncm_Result.h>
#include <nn/ncm/ncm_AutoBuffer.h>
#include <nn/ncm/ncm_ContentStorage.h>
#include <nn/ncm/ncm_ContentMeta.h>
#include <nn/ncm/ncm_ContentMetaKey.h>
#include <nn/ncm/ncm_ContentInfo.h>
#include <nn/ncm/ncm_ContentInfoData.h>
#include <nn/ncm/ncm_InstallProgress.h>
#include <nn/ncm/ncm_InstallTaskData.h>
#include <nn/ncm/ncm_InstallTaskOccupiedSize.h>
#include <nn/ncm/ncm_SystemUpdateTaskApplyInfo.h>

namespace nn { namespace ncm {

    class ContentMetaDatabase;

    struct InstallContentMetaInfo
    {
        ContentId contentId;
        int64_t contentLength;
        ContentMetaType type;
        bool verifyHash;
        Bit8 reserved[6];
        Hash hash;

        static InstallContentMetaInfo MakeVerifiable(const ContentId& contentId, int64_t contentLength, ContentMetaType type, const Hash& hash) NN_NOEXCEPT
        {
            InstallContentMetaInfo info = { contentId, contentLength, type, true };
            info.hash = hash;

            return info;
        }

        static InstallContentMetaInfo MakeUnverifiable(const ContentId& contentId, int64_t contentLength, ContentMetaType type) NN_NOEXCEPT
        {
            InstallContentMetaInfo info = { contentId, contentLength, type, false };

            return info;
        }
    };

    enum InstallConfig
    {
        InstallConfig_LatestAddOnContentsAuto = 0x1u << 1,
        InstallConfig_SystemUpdateRecursive = 0x1u << 2,
        InstallConfig_RequiresExFatDriver = 0x1u << 3,
        InstallConfig_IgnoreTicket = 0x1u << 4,
        InstallConfig_SkipIndirectUpdateCheck = 0x1u << 5,
    };

    enum class ListContentMetaKeyFilter : Bit8
    {
        All,
        Committed,
        NotCommitted
    };

    struct InstallThroughput
    {
        int64_t installed;
        TimeSpanType elapsedTime;
    };

    class InstallTaskBase
    {
        NN_DISALLOW_COPY(InstallTaskBase);
        NN_DISALLOW_MOVE(InstallTaskBase);

    public:
        InstallTaskBase() NN_NOEXCEPT : m_Data(), m_Progress(), m_ProgressMutex(false), m_CancelMutex(false), m_CancelRequested(false), m_ThroughputMutex(false) {}
        virtual ~InstallTaskBase() NN_NOEXCEPT {}

        // ncm_InstallTaskBase では、Prepare 時の Placeholder 作成中、
        // および Execute 時の placeholder 書き込み時にキャンセルできる
        // それ以外のタイミングでキャンセルが必要であれば、別途実装が必要
        virtual void Cancel() NN_NOEXCEPT;
        virtual void ResetCancel() NN_NOEXCEPT;
        Result Prepare() NN_NOEXCEPT;
        Result GetPreparedPlaceHolderPath(ncm::Path* outValue, Bit64 id, ContentMetaType metaType, ContentType contentType) NN_NOEXCEPT;
        Result CalculateRequiredSize(int64_t* outValue) NN_NOEXCEPT;
        Result Cleanup() NN_NOEXCEPT;
        Result ListContentMetaKey(int* outCount, StorageContentMetaKey outList[], int count, int offset) NN_NOEXCEPT { return ListContentMetaKey(outCount, outList, count, offset, ListContentMetaKeyFilter::All);}
        Result ListContentMetaKey(int* outCount, StorageContentMetaKey outList[], int count, int offset, ListContentMetaKeyFilter filter) NN_NOEXCEPT;
        Result ListApplicationContentMetaKey(int* outCount, ApplicationContentMetaKey outList[], int count, int offset) NN_NOEXCEPT;
        Result Execute() NN_NOEXCEPT;
        Result PrepareAndExecute() NN_NOEXCEPT;
        Result Commit() NN_NOEXCEPT { return Commit(nullptr, 0); }
        // Partial Commit 用
        Result Commit(const StorageContentMetaKey* commitKeys, int count) NN_NOEXCEPT;
        virtual InstallProgress GetProgress() NN_NOEXCEPT;
        void ResetLastResult() NN_NOEXCEPT;
        Result IncludesExFatDriver(bool* outValue) NN_NOEXCEPT;
        InstallThroughput GetThroughput() NN_NOEXCEPT;
        Result CalculateContentsSize(int64_t* outValue, const ContentMetaKey& key, StorageId storageId) NN_NOEXCEPT;
        Result ListOccupiedSize(int* outCount, InstallTaskOccupiedSize* outList, int numList, int offset) NN_NOEXCEPT;

        Result FindMaxRequiredApplicationVersion(uint32_t* outValue) NN_NOEXCEPT;
        Result FindMaxRequiredSystemVersion(uint32_t* outValue) NN_NOEXCEPT;

    protected:
        Result Initialize(StorageId storage, InstallTaskDataBase* data, Bit32 config = 0) NN_NOEXCEPT;

        Result PrepareContentMeta(const InstallContentMetaInfo& info, util::optional<ContentMetaKey> expectedKey, util::optional<uint32_t> deltaSourceVersion = util::nullopt) NN_NOEXCEPT;
        Result PrepareContentMeta(ContentId id, int64_t size, ContentMetaType type, AutoBuffer* meta) NN_NOEXCEPT;
        Result WritePlaceHolderBuffer(InstallContentInfo* data, const void* buffer, size_t bufferSize) NN_NOEXCEPT;
        void PrepareAgain() NN_NOEXCEPT;

        Result CountInstallContentMetaData(int* outValue) NN_NOEXCEPT;
        Result GetInstallContentMetaData(InstallContentMeta* outValue, int index) NN_NOEXCEPT;
        Result DeleteInstallContentMetaData(const ContentMetaKey* list, int count) NN_NOEXCEPT;

        virtual Result PrepareDependency() NN_NOEXCEPT;
        Result PrepareSystemUpdateDependency() NN_NOEXCEPT;
        Result PrepareContentMetaIfLatest(const ContentMetaKey& key) NN_NOEXCEPT;
        Bit32 GetConfig() NN_NOEXCEPT {return m_Config;}
        Result WriteContentMetaToPlaceHolder(InstallContentInfo* installInfo, ContentStorage* storage, const InstallContentMetaInfo& info) NN_NOEXCEPT;

        StorageId GetInstallStorage() NN_NOEXCEPT { return m_InstallStorage; }

        virtual Result OnPrepareComplete() NN_NOEXCEPT
        {
            NN_RESULT_SUCCESS;
        }

        Result GetSystemUpdateTaskApplyInfo(SystemUpdateTaskApplyInfo* outValue) NN_NOEXCEPT;

        Result CanContinue() NN_NOEXCEPT;

    private:
        bool IsCancelRequested() NN_NOEXCEPT;
        Result PrepareImpl() NN_NOEXCEPT;
        Result ExecuteImpl() NN_NOEXCEPT;
        Result CommitImpl(const StorageContentMetaKey* commitKeys, int count) NN_NOEXCEPT;
        Result CleanupOne(const InstallContentMeta& data) NN_NOEXCEPT;

        Result VerifyAllNotCommitted(const StorageContentMetaKey* commitKeys, int count) NN_NOEXCEPT;

        // この中でサブクラスが AddInstallContentMetaData を呼んで、インストールしたいコンテンツメタを定義する。
        virtual Result PrepareInstallContentMetaData() NN_NOEXCEPT = 0;
        virtual Result GetInstallContentMetaInfo(InstallContentMetaInfo* outValue, const ContentMetaKey& key) NN_NOEXCEPT = 0;
        virtual Result GetLatestVersion(util::optional<uint32_t>* outValue, Bit64 id) NN_NOEXCEPT
        {
            NN_UNUSED(outValue);
            NN_UNUSED(id);
            NN_RESULT_THROW(ResultContentMetaNotFound());
        }

        virtual Result OnExecuteComplete() NN_NOEXCEPT
        {
            NN_RESULT_SUCCESS;
        }

        crypto::Sha256Generator m_CurrentSha256;
        Result WritePlaceHolder(InstallContentInfo* data) NN_NOEXCEPT;
        // プレースホルダをサブクラスが書き込む。
        virtual Result OnWritePlaceHolder(InstallContentInfo* data) NN_NOEXCEPT = 0;

        bool IsNecessaryInstallTicket(const nn::fs::RightsId& rightsId) NN_NOEXCEPT;
        virtual Result InstallTicket(const nn::fs::RightsId& rightsId, ContentMetaType contentMetaType) NN_NOEXCEPT = 0;

        Result IsNewerThanInstalled(bool* outValue, const ContentMetaKey& key) NN_NOEXCEPT;
        Result PreparePlaceHolder() NN_NOEXCEPT;

        void SetProgressState(InstallProgressState state) NN_NOEXCEPT;
        void IncrementProgress(int64_t installedSize) NN_NOEXCEPT;
        void SetTotalSize(int64_t totalSize) NN_NOEXCEPT;
        void SetLastResult(Result result) NN_NOEXCEPT;
        void CleanupProgress() NN_NOEXCEPT;

        void ResetThroughputMeasurement() NN_NOEXCEPT;
        void StartThroughputMeasurement() NN_NOEXCEPT;
        void UpdateThroughputMeasurement(size_t installed) NN_NOEXCEPT;

        Result GetInstallContentMetaDataFromPath(AutoBuffer* outValue, const Path& path, const InstallContentInfo& data, util::optional<uint32_t> deltaSourceVersion) NN_NOEXCEPT;

        InstallContentInfo MakeInstallContentInfoFrom(const InstallContentMetaInfo& info, const PlaceHolderId& placeHolderId) NN_NOEXCEPT;

        StorageId m_InstallStorage;
        InstallTaskDataBase* m_Data;

        InstallProgress m_Progress;
        os::Mutex m_ProgressMutex;

        Bit32 m_Config;
        os::Mutex m_CancelMutex;
        bool  m_CancelRequested;

        InstallThroughput m_Throughput;
        TimeSpan m_ThroughputMeasurementStartTime;
        os::Mutex m_ThroughputMutex;
    };
}}
