﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/fssystem/fs_DeltaHeader.h>
#include <nn/ncm/ncm_Result.h>

namespace nn { namespace ncm {

class ContentStorage;
struct PlaceHolderId;

/**
* @brief   パッチ間差分の適用処理を行うクラスです。
*
*  パッチに対してパッチの適用処理を行います。
*/
class DeltaApplier
{
public:
    static const int64_t ExtendSizeMax = 128 * 1024 * 1024;

private:
    /**
    * @brief   パッチ間差分の適用状態を記憶する構造体です。
    */
    struct StatusHeader
    {
        int64_t offsetDelta; //!< パッチ間差分の次の読み取り位置
        int64_t offsetPatch; //!< パッチ間差分を適用するパッチファイルの書き込み位置

        int32_t offsetHeader; //!< パッチ間差分データのヘッダー構造体の書き込み位置
        int32_t sizeHeader; //!< パッチ間差分データのヘッダー構造体のサイズ
    };

    NN_STATIC_ASSERT(std::is_pod<StatusHeader>::value);

public:
    static const int StatusSize = sizeof(StatusHeader) + fssystem::DeltaHeader::Size;

    /**
    * @brief   コンストラクタです。
    */
    DeltaApplier() NN_NOEXCEPT;

    /**
    * @brief   デストラクタです。
    */
    ~DeltaApplier() NN_NOEXCEPT;

    /**
    * @brief   初期化します。
    *
    * @param[out]  pOutOffsetDelta パッチ間差分データの開始オフセット
    * @param[in]   pStoragePatch                 パッチ間差分を適用するパッチのコンテントストレージ
    * @param[in]   placeHolderIdPatch            パッチ間差分を適用するパッチのプレースホルダー
    * @param[in]   pStatus                       パッチ間差分の適用状態を保存するバッファー
    * @param[in]   sizeStatus                    パッチ間差分の適用状態を保存するバッファーの有効なデータのサイズ
    *
    * @retval  ResuleSuccess                       正常に処理が終了しました。
    * @retval  ncm:::ResultInvalidApplyDeltaStatus パッチ間差分の適用状態が異常です。
    *
    * @pre
    *      - pOutOffsetDelta != nullptr
    *      - pStoragePatch != nullptr
    *      - pStatus != nullptr
    */
    Result Initialize(
        int64_t* pOutOffsetDelta,
        ncm::ContentStorage* pStoragePatch,
        const ncm::PlaceHolderId& placeHolderIdPatch,
        const char* pStatus,
        size_t sizeStatus
    ) NN_NOEXCEPT;

    /**
    * @brief   パッチ間差分を適用します。
    *
    * @param[out]  pOutStatus          パッチ間差分の適用状態を保存するバッファー
    * @param[in]   sizeStatus          パッチ間差分の適用状態を保存するバッファーのサイズ
    * @param[in]   pDelta              パッチ間差分データ
    * @param[in]   sizeDelta           パッチ間差分データのサイズ
    *
    * @retval  ResuleSuccess                         正常に処理が終了しました。
    * @retval  ncm::ResultInvalidDeltaFormat         パッチ間差分が異常です。
    * @retval  ncm::ResultNotEnoughSpaceToApplyDelta パッチ間差分を適用するためのスペースが足りません。
    * @retval  上記以外                              パッチ間差分の適用に失敗しました。
    *
    * @pre
    *      - pOutStatus != nullptr
    *      - StatusSize <= sizeStatus
    *      - pDelta != nullptr
    *      - 0 < sizeDelta
    */
    Result ApplyDelta(
                size_t* pOutProcessedSize,
                char* pOutStatus,
                size_t sizeStatus,
                const void* pDelta,
                size_t sizeDelta
            ) NN_NOEXCEPT;

private:
    /**
    * @brief   パッチ間差分の適用状態をバッファーから読み込みます。
    *
    * @param[in]   pStatus             パッチ間差分の適用状態バッファー
    * @param[in]   sizeStatus          パッチ間差分の適用状態バッファーのサイズ
    *
    * @retval  ResuleSuccess                      正常に処理が終了しました。
    * @retval  ncm::ResultInvalidApplyDeltaStatus パッチ間差分の適用状態が異常です。
    *
    * @pre
    *      - pStatus != nullptr
    */
    Result SetStatus(const char* pStatus, size_t sizeStatus) NN_NOEXCEPT;

    /**
    * @brief   パッチ間差分の適用状態をバッファーに書き込みます。
    *
    * @param[out]  pOutStatus          パッチ間差分の適用状態を保存するバッファー
    * @param[in]   sizeStatus          パッチ間差分の適用状態を保存するバッファーのサイズ
    *
    * @pre
    *      - pOutStatus != nullptr
    *      - StatusSize <= sizeStatus
    */
    void GetStatus(char* pOutStatus, size_t sizeStatus) NN_NOEXCEPT;

    /**
    * @brief   パッチ間差分のヘッダーを検証します。
    *
    * @retval  ResuleSuccess                 正常に処理が終了しました。
    * @retval  ncm::ResultInvalidDeltaFormat 検証に失敗しました。
    */
    Result VerifyHeader() NN_NOEXCEPT;

private:
    ncm::ContentStorage* m_pStoragePatch; //!< 適用するパッチのコンテントストレージ
    ncm::PlaceHolderId m_PlaceHolderIdPatch; //!< 適用するパッチのプレースホルダー
    fssystem::DeltaHeader m_Header; //!< パッチ間差分データのメタデータ
    StatusHeader m_StatusData; //!< パッチ間差分データ適用処理の状態
    unsigned char m_Command; //!< パッチ間差分データの実行中のコマンド
    int64_t m_CommandOffset; //!< パッチ間差分データのオフセット
    int64_t m_WriteCommandOffset; //!< パッチ間差分データの上書きコマンドのオフセット
    int64_t m_WriteCommandSize; //!< パッチ間差分データの上書きコマンドのデータサイズ
};

} }
