﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <string>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/os/os_Mutex.h>
#include <nn/fs/fs_Mount.h>
#include <nn/fs/fs_ContentStorage.h>
#include <nn/sf/sf_MemUtility.h>
#include <nn/fs/fs_Bis.h>
#include <nn/fs/fs_SystemSaveData.h>
#include <nn/ncm/ncm_IContentManager.h>
#include <nn/ncm/ncm_ContentManagerConfig.h>
#include <nn/ncm/ncm_ContentMetaDatabase.h>
#include <nn/ncm/ncm_BoundedMap.h>
#include <nn/kvdb/kvdb_FlatMapKeyValueStore.h>


namespace nn { namespace ncm {

    class ContentMetaMemoryResource
    {
    public:
        ContentMetaMemoryResource(void* heap, size_t heapSize) NN_NOEXCEPT :
            m_Allocator(heap, heapSize), m_MemoryResource(&m_Allocator) {}

        MemoryResource* Get() NN_NOEXCEPT
        {
            return &m_MemoryResource;
        }

    private:
        mem::StandardAllocator m_Allocator;
        sf::StandardAllocatorMemoryResource m_MemoryResource;
    };

    class ContentManagerImpl
    {
    public:
        ContentManagerImpl() NN_NOEXCEPT : m_Mutex(true), m_IsInitialized(false) {}

        ~ContentManagerImpl() NN_NOEXCEPT;

        Result Initialize() NN_NOEXCEPT;
        Result Initialize(const ContentManagerConfig &config) NN_NOEXCEPT;

        Result CreateContentStorage(StorageId storageId) NN_NOEXCEPT;

        Result CreateContentMetaDatabase(StorageId storageId) NN_NOEXCEPT;

        Result VerifyContentStorage(StorageId storageId) NN_NOEXCEPT;

        Result VerifyContentMetaDatabase(StorageId storageId) NN_NOEXCEPT;

        Result OpenContentStorage(sf::Out<sf::SharedPointer<IContentStorage>> outValue, StorageId storageId) NN_NOEXCEPT;

        Result OpenContentMetaDatabase(sf::Out<sf::SharedPointer<IContentMetaDatabase>> outValue, StorageId storageId) NN_NOEXCEPT;

        Result CleanupContentMetaDatabase(StorageId storageId) NN_NOEXCEPT;

        Result ActivateContentStorage(StorageId storageId) NN_NOEXCEPT;

        Result InactivateContentStorage(StorageId storageId) NN_NOEXCEPT;

        Result ActivateContentMetaDatabase(StorageId storageId) NN_NOEXCEPT;

        Result InactivateContentMetaDatabase(StorageId storageId) NN_NOEXCEPT;

    private:
        os::Mutex m_Mutex;
        bool m_IsInitialized;

        class ContentStorageRoot
        {
            NN_DISALLOW_COPY(ContentStorageRoot);
            NN_DISALLOW_MOVE(ContentStorageRoot);

        public:
            ContentStorageRoot() NN_NOEXCEPT {}
            char mountName[fs::MountNameLengthMax + 1];
            char path[128];
            StorageId storageId;
            fs::ContentStorageId contentStorageId;
            sf::SharedPointer<IContentStorage> storage;
        };

        struct SystemSaveDataInfo
        {
            fs::SystemSaveDataId id;
            int64_t size;
            int64_t journalSize;
            int flag;
            fs::SaveDataSpaceId spaceId;
        };

        class ContentMetaDatabaseRoot
        {
        public:
            ContentMetaDatabaseRoot() NN_NOEXCEPT{}
            char mountName[fs::MountNameLengthMax + 1];
            char path[128];
            StorageId storageId;
            SystemSaveDataInfo info;
            sf::SharedPointer<IContentMetaDatabase> db;
            util::optional<nn::kvdb::FlatMapKeyValueStore<ContentMetaKey>> kvs;
            ContentMetaMemoryResource* memoryResource;
            int maxContentMetaCount;
        };

        Result GetContentStorageRoot(ContentStorageRoot** outValue, StorageId id) NN_NOEXCEPT;
        Result GetContentMetaDatabaseRoot(ContentMetaDatabaseRoot** outValue, StorageId id) NN_NOEXCEPT;

        Result InitializeContentStorageRoot(ContentStorageRoot* outValue, StorageId storageId, fs::ContentStorageId contentStorageId) NN_NOEXCEPT;
        Result InitializeGameCardContentStorageRoot(ContentStorageRoot* outValue) NN_NOEXCEPT;
        Result InitializeContentMetaDatabaseRoot(
            ContentMetaDatabaseRoot* outValue,
            StorageId storageId,
            const SystemSaveDataInfo& info,
            int maxContentMetaCount,
            ContentMetaMemoryResource* memoryResource) NN_NOEXCEPT;
        Result InitializeGameCardContentMetaDatabaseRoot(ContentMetaDatabaseRoot* outValue, int maxContentMetaCount, ContentMetaMemoryResource* memoryResource) NN_NOEXCEPT;
        Result BuildContentMetaDataBase(StorageId storageId) NN_NOEXCEPT;
        Result BuildContentMetaDataBaseFromFile(StorageId storageId, fs::BisPartitionId bisPartitionId, const char* filePath) NN_NOEXCEPT;
        Result EnsureAndMountSystemSaveData(const char* mountName, const SystemSaveDataInfo& info) const NN_NOEXCEPT;

        ContentStorageRoot m_StorageRoot[8];
        ContentMetaDatabaseRoot m_MetaRoot[8];
        int m_StorageRootIndex{};
        int m_MetaRootIndex{};
    };
}}
