﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <string>
#include <nn/nn_Common.h>
#include <nn/ncm/ncm_ContentInfo.h>
#include <nn/ncm/ncm_ContentStorage.h>
#include <nn/ncm/ncm_StorageId.h>
#include <nn/ncm/ncm_ContentMetaKey.h>
#include <nn/crypto/crypto_Sha256Generator.h>

namespace nn { namespace ncm {

    struct Hash
    {
        Bit8 data[crypto::Sha256Generator::HashSize];
    };

    struct PackagedContentInfo
    {
        Hash        hash;
        ContentInfo info;

        const ContentId& GetId() const NN_NOEXCEPT
        {
            return info.id;
        }

        ContentType GetType() const NN_NOEXCEPT
        {
            return info.type;
        }

        uint8_t GetIdOffset() const NN_NOEXCEPT
        {
            return info.idOffset;
        }
    };

    enum class InstallState : Bit8
    {
        NotPrepared,
        Prepared,
        Installed,
        AlreadyExists
    };

    struct InstallContentInfo
    {
        Hash                    hash;
        crypto::Sha256Context   context;
        char                    bufferedData[crypto::Sha256Generator::BlockSize];
        uint64_t                bufferedDataSize;
        ContentInfo             info;
        PlaceHolderId           placeHolderId;
        ContentMetaType         metaType;
        InstallState            state;
        bool                    verifyHash;
        StorageId               storageId;
        bool                    isTemporary;
        bool                    isShaContextAvailable;
        Bit8                    reserved[2];
        int64_t                 written;

        const ContentId& GetId() const NN_NOEXCEPT
        {
            return info.id;
        }

        ContentType GetType() const NN_NOEXCEPT
        {
            return info.type;
        }

        static void MakeFrom(InstallContentInfo* outValue, const PackagedContentInfo& data, ContentMetaType metaType) NN_NOEXCEPT
        {
            std::memset(outValue, 0, sizeof(InstallContentInfo));
            std::memcpy(&outValue->hash, &data.hash, sizeof(outValue->hash));
            std::memcpy(&outValue->info, &data.info, sizeof(outValue->info));
            outValue->verifyHash = true;
            outValue->metaType = metaType;
        }
    };
}}
