﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
*   @file
*   @brief  htclow のモジュール内チャネルに関する API
*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>

#include <nn/htclow/htclow_ChannelTypes.h>

namespace nn { namespace htclow {

/**
*   @brief      チャネル構造体を初期化します
*
*   @param[out] pOutModule  チャネル構造体
*   @param[out] pModule     モジュール構造体
*   @param[in]  id          チャネル識別子
*
*   @details    チャネル構造体を、モジュールに紐づけて初期化します。
*
*   @pre        - 通常用 API の初期化を実行済
*               - チャネル構造体が未初期化
*               - モジュール構造体が初期化済
*
*   @post       - チャネル構造体が初期化済
*/
void InitializeChannel(ChannelType* pOutChannel, const ModuleType* pModule, ChannelId id) NN_NOEXCEPT;

/**
*   @brief      チャネルをオープンします
*
*   @param[in]  pChannel  チャネル構造体
*
*   @retresult
*       @handleresult{nn::htclow::ResultChannelStateError}
*       @handleresult{nn::htclow::ResultInternalError}
*   @endretresult
*
*   @pre        - 通常用 API の初期化を実行済
*               - チャネル構造体が初期化済
*               - チャネルがクローズ状態
*
*   @post       - チャネルがオープン状態
*
*   @details    チャネルをオープンします。@n
*               通信用バッファなどのリソースが確保され、Connect が可能になります。@n
*               チャネルがクローズ状態でなかった場合は ResultChannelStateError が返ります。
*/
nn::Result OpenChannel(const ChannelType* pChannel) NN_NOEXCEPT;

/**
*   @brief      ホストと通信してチャネルを接続状態にします
*
*   @param[in]  pChannel  チャネル構造体
*
*   @retresult
*       @handleresult{nn::htclow::ResultChannelStateError}
*       @handleresult{nn::htclow::ResultInternalError}
*   @endretresult
*
*   @pre        - 通常用 API の初期化を実行済
*               - チャネルがオープン状態
*
*   @post       - チャネルが接続状態
*
*   @details    ホストと通信を行い、チャネルを接続状態にします。@n
*               接続状態になると、送受信が可能になります。@n
*               チャネルがオープン状態でなかった場合は、ResultChannelStateError が返ります。
*/
nn::Result ConnectChannel(const ChannelType* pChannel) NN_NOEXCEPT;

/**
*   @brief      チャネルにデータを送信します
*
*   @param[out] pOutSize    送信したデータのサイズ
*   @param[in]  pBuffer     入力バッファ
*   @param[in]  bufferSize  入力バッファのサイズ
*   @param[in]  pChannel    チャネル構造体
*
*   @retresult
*       @handleresult{nn::htclow::ResultChannelStateError}
*       @handleresult{nn::htclow::ResultConnectionFailure}
*       @handleresult{nn::htclow::ResultInternalError}
*   @endretresult
*
*   @pre        - 通常用 API の初期化を実行済
*               - チャネルが接続状態
*
*   @details    入力バッファのデータをチャネルに送信します。@n
*               チャネルが接続状態でなかった場合は ResultChannelStateError が返ります。@n
*               ホストとの通信が切断された場合は ResultConnectionFailure が返ります。@n
*               ResultConnectionFailure が返ってきた場合は、チャネルをクローズする必要があります。@n
*               通信を再試行する場合はクローズ後に再度オープンしてください。
*
*/
nn::Result Send(size_t* pOutSize, const void* pBuffer, size_t bufferSize, const ChannelType* pChannel) NN_NOEXCEPT;

/**
*   @brief      チャネルからデータを受信します
*
*   @param[out] pOutSize    出力バッファに書き込まれたデータのサイズ
*   @param[out] pOutBuffer  出力バッファ
*   @param[in]  bufferSize  出力バッファのサイズ
*   @param[in]  pChannel    チャネル構造体
*
*   @retresult
*       @handleresult{nn::htclow::ResultChannelStateError}
*       @handleresult{nn::htclow::ResultConnectionFailure}
*       @handleresult{nn::htclow::ResultInternalError}
*   @endretresult
*
*   @pre        - 通常用 API の初期化を実行済
*               - チャネルが接続状態
*
*   @details    チャネルからデータを読み出し、出力バッファに書き込みます。@n
*               チャネルが接続状態でなかった場合は ResultChannelStateError が返ります。@n
*               ホストとの通信が切断された場合は ResultConnectionFailure が返ります。@n
*               ResultConnectionFailure が返ってきた場合は、チャネルをクローズする必要があります。@n
*               通信を再試行する場合はクローズ後に再度オープンしてください。
*/
nn::Result Receive(size_t* pOutSize, void* pOutBuffer, size_t bufferSize, const ChannelType* pChannel) NN_NOEXCEPT;

/**
*   @brief      チャネルを切断します
*
*   @param[in]  pOutChannel チャネル構造体
*
*   @pre        - 通常用 API の初期化を実行済
*               - チャネルが接続状態
*
*   @post       - チャネルがオープン状態
*
*   @details    ホストと接続しているチャネルの切断処理を行います。
*/
void ShutdownChannel(const ChannelType* pOutChannel) NN_NOEXCEPT;

/**
*   @brief      モジュール内チャネルをクローズします
*
*   @param[in]  pOutChannel チャネル構造体
*
*   @pre        - 通常用 API の初期化を実行済
*               - チャネルがオープン状態
*
*   @post       - チャネルがクローズ状態
*
*   @details    チャネルをクローズします。
*               通信用バッファなどのリソースが解放されます。
*/
void CloseChannel(const ChannelType* pOutChannel) NN_NOEXCEPT;

/**
*   @brief      チャネル構造体を未初期化状態にします
*
*   @param[out] pOutChannel チャネル構造体
*
*   @details    チャネル構造体を未初期化状態にします。
*
*   @pre        - 通常用 API の初期化を実行済
*               - チャネル構造体が初期化済
*
*   @post       - チャネル構造体が未初期化
*/
void FinalizeChannel(ChannelType* pOutChannel) NN_NOEXCEPT;

}}
