﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
*   @file
*   @brief  ホスト・ターゲット間通信に関する API
*/

#pragma once

#include<nn/nn_Common.h>
#include<nn/nn_Result.h>

namespace nn {
    class TimeSpan;
}

//--------------------------------------------------------------------------
//  C++ 向けの宣言
//--------------------------------------------------------------------------
namespace nn { namespace htc {

/**
*   @brief      ホストとの通信が確立されたことを通知するイベントオブジェクトにバインドします。
*
*   @param[out] pOutValue       イベントオブジェクトを指すポインタ
*
*   @pre
*               - ライブラリが初期化済の状態である
*               - pOutValue != nullptr
*               - *pOutValue は未初期化状態
*
*   @post
*               - *pOutValue は初期化状態
*
*   @details    ホストとの通信が確立されたことを通知するイベントオブジェクトにバインドします。@n
*               イベントは読み取り専用で手動クリアに設定されます。@n
*               ホストと接続中は常にシグナル状態で、切断時にクリアされます。@n
*               イベントはシステム側でクリアするため、ユーザ側でクリアしないでください。
*/
void BindHostConnectionEventForSystem(nn::os::SystemEvent* pOutValue) NN_NOEXCEPT;

/**
*   @brief      ホストとの通信が切断されたことを通知するイベントオブジェクトにバインドします。
*
*   @param[out] pOutValue       イベントオブジェクトを指すポインタ
*
*   @pre
*               - ライブラリが初期化済の状態である
*               - pOutValue != nullptr
*               - *pOutValue は未初期化状態
*
*   @post
*               - *pOutValue は初期化状態
*
*   @details    ホストとの通信が切断されたことを通知するイベントオブジェクトにバインドします。@n
*               イベントは読み取り専用で手動クリアに設定されます。@n
*               ホストと切断中は常にシグナル状態で、接続時にクリアされます。@n
*               イベントはシステム側でクリアするため、ユーザ側でクリアしないでください。
*/
void BindHostDisconnectionEventForSystem(nn::os::SystemEvent* pOutValue) NN_NOEXCEPT;

/**
*   @brief      ホスト PC との通信に使用されている IP アドレスを取得します。
*
*   @param[out] pOutValue       ホストブリッジの IP アドレス
*
*   @pre
*               - ライブラリが初期化済の状態である
*
*   @post
*               - *pOutValue は初期化状態
*
*   @details    ホスト PC との通信に使用されている IP アドレスを取得します。
*/
Result GetBridgeIpAddress(char* outBuffer, size_t bufferSize) NN_NOEXCEPT;

/**
*   @brief      ホスト PC との通信に使用されているネットワークのサブネットマスクを取得します。
*
*   @param[out] pOutValue       ホストブリッジのサブネットマスク
*
*   @pre
*               - ライブラリが初期化済の状態である
*
*   @post
*               - *pOutValue は初期化状態
*
*   @details    ホスト PC との通信に使用されているネットワークのサブネットマスクを取得します。
*/
Result GetBridgeSubnetMask(char* outBuffer, size_t bufferSize) NN_NOEXCEPT;

/**
*   @brief      ホストブリッジの MAC アドレスを取得します。
*
*   @param[out] pOutValue       ホストブリッジの MAC アドレス
*
*   @pre
*               - ライブラリが初期化済の状態である
*
*   @post
*               - *pOutValue は初期化状態
*
*   @details    ホストブリッジの MAC アドレスを取得します。
*/
Result GetBridgeMacAddress(char* outBuffer, size_t bufferSize) NN_NOEXCEPT;

/**
*   @brief      ホスト PC との通信に使用されているポート番号を取得します。
*
*   @param[out] pOutValue       ホストブリッジのポート番号
*
*   @pre
*               - ライブラリが初期化済の状態である
*
*   @post
*               - *pOutValue は初期化状態
*
*   @details     ホスト PC との通信に使用されているポート番号を取得します。
*/
Result GetBridgePort(char* outBuffer, size_t bufferSize) NN_NOEXCEPT;

}}
