﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
/**
*   @file
*   @brief  開発ゲームカード用ゲームカードドライバ API の宣言
*/

#pragma once

#include <nn/nn_Common.h>

#include <nn/gc/gc_Result.h>
#include <nn/gc/detail/gc_Types.h>
#include <nn/gc/detail/gc_Define.h>

namespace nn { namespace gc { namespace writer {

enum AsicMode
{
    AsicMode_ReadMode,
    AsicMode_WriteMode
};

enum MemorySize
{
    MemorySize_1GB  = 1,
    MemorySize_2GB  = 2,
    MemorySize_4GB  = 4,
    MemorySize_8GB  = 8,
    MemorySize_16GB = 16,
    MemorySize_32GB = 32,
};

//! @name 開発ゲームカード用 API
//! @{

/**
*   @brief      ASIC のモードを切り替えます。通常時は Read モードを使用してください。
*               開発カードへの書き込みを行いたいときのみ、Write モードを使用してください。
*
*   @param[in]  asicMode       使用したい ASIC のモード
*/
void ChangeMode(const AsicMode asicMode) NN_NOEXCEPT;

/**
*   @brief      ゲームカードのバスを有効化します。カードが挿入されたら初めに呼んでください。
*               ActivateForDebug 後にカードの抜去・再挿入があると内部でハードウェアをリセットするため復帰に時間がかかります。
*               ゲームカード未挿入時、および挿入されたゲームカードが開発カードでない時処理に失敗します。
*               WriteWithVerify 前に必ず呼んでください。
*
*   @return     処理の結果が返ります。
*/
nn::Result ActivateForWriter() NN_NOEXCEPT;

/**
*   @brief      カードメモリに書き込むことができる最大サイズを取得します。カードメモリへのパラメタ書き込みで設定したサイズとは関係のない、元々のサイズを取得します。
*
*   @param[in]  pOutValue   カードメモリサイズの格納先変数へのポインタ
*
*   @return     処理の結果が返ります。
*/

nn::Result GetCardAvailableRawSize(int64_t* pOutValue) NN_NOEXCEPT;


/**
*   @brief      全域 Erase の後、開発カードに設定パラメータを書き込みます。
*               パラメータを設定する場合、必ず WriteWithVerify の前に本 API を呼んでください。
*
*   @param[in]  memorySize   書き込み時に設定するカードメモリのサイズ。ここで指定した値によって書き込み・読み出し可能なサイズが制限されます。
*   @param[in]  secureAreaStartPageAddress   ノーマルエリアのページサイズです。
*
*   @return     処理の結果が返ります。
*/

nn::Result EraseAndWriteParameter(const MemorySize memorySize, const uint32_t secureAreaStartPageAddress) NN_NOEXCEPT;

/**
*   @brief      開発カードへのデータ書き込みを行います。
*               SetVerifyEnalbleFlag を呼んでいない場合、Verify は行いません。
*
*   @param[in]  inDataBuffer    書き込みデータの格納先バッファ。アラインされている必要があります。
*   @param[in]  dataBufferSize  書き込みデータの格納先バッファのサイズ
*   @param[in]  pageAddress     書き込みを開始するページアドレス
*   @param[in]  pageCount       書き込むページサイズ
*
*   @pre
*       - ActivateForWriter に成功している
*       - Verify を行う場合は SetVerifyEnalbleFlag で Verify を有効にしている。
*
*   @return     処理の結果が返ります。
*/
nn::Result Write(void* inDataBuffer, const size_t dataBufferSize, const uint32_t pageAddress, const uint32_t pageCount) NN_NOEXCEPT;

/**
*   @brief      書き込み後に Verify を行うかどうかのフラグを設定します。
*
*   @param[in]  isVerifyEnable 書き込み後に Verify を行うかどうかのフラグです。
*
*   @return     処理の結果が返ります。
*/
void SetVerifyEnalbleFlag(bool isVerifyEnable) NN_NOEXCEPT;


// *** 以下 特殊用途向き API ***

/**
*   @brief      ユーザによる任意の ASIC FW を設定します。ユーザ ASIC FW を使用するには ChangeMode で AsicMode_WriteMode を指定する必要があります。
*               一度でもユーザ ASIC FW を使用すると、開発カードへの書き込み用 ASIC FW が上書きされてしまいます。
*               この状態から再び書き込みを行えるようにするには、書き込み用 ASIC FW をユーザ ASIC FW として設定し直してください。
*
*   @param[in]  buffer         ユーザ ASIC FW の格納元バッファ
*   @param[in]  bufferLength   ユーザ ASIC FW の格納元バッファのサイズ
*
*/
void SetUserAsicFirmwareBuffer(const char* buffer, const size_t bufferLength) NN_NOEXCEPT;


/**
*   @brief      RMA 用の ASIC FW を使用している時に限り、RMA 情報を ASIC から取得します。
*
*   @param[in]  pOutValue      RMA 情報の格納先となる構造体へのポインタ
*
*   @return     処理の結果が返ります。
*/
nn::Result GetRmaInformation(RmaInformation* pOutValue) NN_NOEXCEPT;

/**
*   @brief      開発カードの設定値を直接書き込みます
*
*   @param[in]  devCardParameter  開発カードの設定値
*
*   @return     処理の結果が返ります。
*/
nn::Result WriteDevCardParam(detail::DevCardParameter& devCardParameter) NN_NOEXCEPT;

/**
*   @brief      開発カードの設定値を直接読みます
*
*   @param[in]  pOutValue  開発カードの設定値の格納先となる構造体へのポインタ
*
*   @return     処理の結果が返ります。
*/
nn::Result ReadDevCardParam(detail::DevCardParameter* pOutValue) NN_NOEXCEPT;

/**
*   @brief      開発カードの中身を強制的に Erase します。完全に中身が空になります。
*   本 API の後には EraseAndWriteParameter を呼んで下さい。
*
*   @return     処理の結果が返ります。
*/
nn::Result ForceErase() NN_NOEXCEPT;

//! @}

}}} // namespace nn::gc::dev
