﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/util/util_BitPack.h>
#include <nn/gc/gc.h>

namespace nn { namespace gc {
namespace detail {

// ゲームカードに対して発行するコマンドを扱うクラス
class AsicOperationCardCommandBuffer
{
    NN_DISALLOW_COPY(AsicOperationCardCommandBuffer);

    typedef struct
    {
        u8 operationId;
        u8 dataPageSize;
        u8 dataDirection;
        u8 reserved1;
        u32 dataPageNum;
        u8 commandBuffer[16];
        u8 reserved2[8];
        u8 cv[32];
    } AsicOperationCardCommand;

public:
    // AsicOperationBuffer の buffer を実体とする構造体でのアクセサ
    AsicOperationCardCommand bufferMap;

    AsicOperationCardCommandBuffer() NN_NOEXCEPT
    {
        NN_STATIC_ASSERT(sizeof(AsicOperationCardCommandBuffer::AsicOperationCardCommand) == GcMmcCmd60DataSize);
        memset(&bufferMap, 0, sizeof(AsicOperationCardCommand));
        bufferMap.operationId = static_cast<u8>(AsicOperationIndex_CARD_CMDOUT);
    }
    explicit AsicOperationCardCommandBuffer(const CardCommandId commandId)
    {
        memset(&bufferMap, 0, sizeof(AsicOperationCardCommand));
        bufferMap.operationId = static_cast<u8>(commandId);
    }

    void GetCommandBufferWithNoData(char* outOperationBuffer, size_t outOperationBufferSize, const CardCommandId command, const DataPageSize dataPageSize) NN_NOEXCEPT;
    void GetCommandBufferWithPageRead(char* outOperationBuffer, size_t outOperationBufferSize, const CardCommandId command, const u32 pageAddress, const u32 pageCount) NN_NOEXCEPT;
    void GetCommandBufferWithPageWrite(char* outOperationBuffer, size_t outOperationBufferSize, const CardCommandId command, const u32 pageAddress, const u32 pageCount) NN_NOEXCEPT;
    void GetCommandBufferDirectly(char* outOperationBuffer, size_t outOperationBufferSize, const DataPageSize dataPageSize, const DataTransDirectionForCard dataDirection, const u32 dataPageNum, const CardCommandId command, const u32 pageAddress, const u32 pageCount) NN_NOEXCEPT;

private:
    void SetCommandBufferForAsic(const DataPageSize dataPageSize, const DataTransDirectionForCard dataTransDirection, const u32 pageCount) NN_NOEXCEPT;
    void SetCommandBufferForGc(const CardCommandId command, const u32 pageAddress, const u32 pageCount) NN_NOEXCEPT;
    void CopyBuffer(char* outOperationBuffer, size_t outOperationBufferSize) NN_NOEXCEPT;
};

// ASIC オペレーションを扱うクラス
class AsicOperation
{
    NN_DISALLOW_COPY(AsicOperation);

private:
    // NOTE: DataIo にワークバッファのあるなしを統一したほうが暗復号化のバッファを分けることができるが、現時点では共通のバッファを使用する予定
    static char* g_WorkAlignedBuffer;
    static size_t g_WorkAlignedBufferLength;
    char m_AsicOperationBuffer[GcMmcCmd60DataSize];
    uint32_t m_SdmmcTimeoutMilliSeconds;

public:
    static const int GcAsicOperationCardCommandOffset = 8;
    static const int GcAsicOperationRegisterAddressOffset = 4;
    static const int GcAsicOperationDataStartOffset = 1;

public:
    // AsicOperation クラスの関数
    static AsicOperation& GetInstance() NN_NOEXCEPT; // Singleton パターン
    static void Initialize(char* workAlignedBuffer, const size_t workAlignedBufferLength) NN_NOEXCEPT;
    void Finalize() NN_NOEXCEPT {}

    // Operation コマンド発行用の API
    nn::Result SendFirmware() NN_NOEXCEPT;
    nn::Result ReceiveCertificate(char* outBuffer, const size_t bufferLength) NN_NOEXCEPT;
    nn::Result SendSocCertificate(const char* certBuffer, const size_t bufferLength) NN_NOEXCEPT;
    nn::Result ReceiveRandomValue(char* outBuffer, const size_t bufferLength) NN_NOEXCEPT;
    nn::Result SendRandomValue(const char* dataBuffer, const size_t bufferLength) NN_NOEXCEPT;
    nn::Result ReceiveDeviceChallenge(char* outBuffer, const size_t bufferLength) NN_NOEXCEPT;
    nn::Result RespondDeviceChallenge(const char* hashBuffer, const size_t bufferLength) NN_NOEXCEPT;
    nn::Result SendHostChallenge(const char* randomValueBuffer, const size_t bufferLength) NN_NOEXCEPT;
    nn::Result ReceiveChallengeResponse(char* outBuffer, const size_t bufferLength) NN_NOEXCEPT;
    nn::Result ChangeModeToSecure() NN_NOEXCEPT;
    nn::Result EnableCardBus() NN_NOEXCEPT;
    nn::Result ExchangeRandomValuesInSecureMode(char* outBuffer, const size_t receiveBufferLength, const char* randomValueBuffer, const size_t sendBufferLength) NN_NOEXCEPT; // バッファの長さは outBuffer, randomValueBuffer で共通
    nn::Result GetCardHeader(char* outBuffer, const size_t bufferLength) NN_NOEXCEPT;
    nn::Result ChangeGcModeToSecure(char* outBuffer, const size_t bufferLength) NN_NOEXCEPT;
    nn::Result ChangeGcModeToDebug(char* outBuffer, const size_t bufferLength) NN_NOEXCEPT;

    // *** 特定用途 API ***
    nn::Result ReadRmaInformation(RmaInformation* pOutInformation) NN_NOEXCEPT;

    // レジスタ操作
    // Write はアドレスごとの書き込みの為、AsicRegister クラスを利用する
    nn::Result WriteRegister(u32 address, u32 value) NN_NOEXCEPT;
    // レジスタのリードは一括処理
    nn::Result ReadRegister() NN_NOEXCEPT;

    // ゲームカードに対してコマンドを投げるAPI
    nn::Result SendCardReadId1(CardId1* pOutId1) NN_NOEXCEPT;
    nn::Result SendCardReadId2(CardId2* pOutId2) NN_NOEXCEPT;
    nn::Result SendCardReadId3(CardId3* pOutId3) NN_NOEXCEPT;
    nn::Result SendCardReadUid(CardUid* pOutCardUid) NN_NOEXCEPT;
    nn::Result SendCardReadPage(char* outAlignedDataBuffer, const size_t bufferLength, const u32 pageAddress, const u32 pageCount) NN_NOEXCEPT;
    nn::Result SendCardReadPageWithNotAlignedBuffer(char* outDataBuffer, const size_t bufferLength, const u32 pageAddress, const u32 pageCount) NN_NOEXCEPT;
    nn::Result SendCardWritePage(char* workAlignedDataBuffer, const size_t bufferLength, const u32 pageAddress, const u32 pageCount) NN_NOEXCEPT;
    nn::Result SendCardRefresh(RefreshResponse* pOutRefreshResponse) NN_NOEXCEPT;
    nn::Result SendCardSetKey() NN_NOEXCEPT;
    nn::Result SendCardSelfRefresh(SelfRefreshResponse* pOutRefreshResponse) NN_NOEXCEPT;
    nn::Result SendCardReadRefreshStatus(SelfRefreshStatus* pOutSelfRefreshStatus) NN_NOEXCEPT;
    nn::Result SendCardReadCrc(uint32_t* pCrc, size_t crcLength, const u32 pageAddress, const u32 pageCount) NN_NOEXCEPT;
    nn::Result SendCardErase() NN_NOEXCEPT;
    nn::Result SendCardReadParameter(DevCardParameter* pOutDevCardParameter) NN_NOEXCEPT;
    nn::Result SendCardWriteParameter(const DevCardParameter& devCardParameter) NN_NOEXCEPT;
    nn::Result SendCardReadErrorCount(DevCardErrorCount* pOutErrCount, const u32 pageAddress) NN_NOEXCEPT;
    nn::Result SendCardChangeDebugDirect() NN_NOEXCEPT;

    // Read したデータを復号化する API
    void DecryptReadData(char* workBuffer, const size_t bufferLength) NN_NOEXCEPT;

private:
    // Index のみ必要な Operation API
    nn::Result ProcessWriteOperation(const char* inDataBuffer, const size_t bufferLength, const AsicOperationIndex index) NN_NOEXCEPT;
    nn::Result ProcessReadOperation(char* outDataBuffer, const size_t bufferLength, const AsicOperationIndex index) NN_NOEXCEPT;
    nn::Result ProcessNoDataOperation(const AsicOperationIndex index) NN_NOEXCEPT;

    // ASIC へ送るコマンドバッファを直接指定する Operation API
    nn::Result ProcessWriteOperation(const char* inDataBuffer, const size_t bufferLength, const char* operationBuffer, const size_t operationBufferSize) NN_NOEXCEPT;
    nn::Result ProcessReadOperation(char* outDataBuffer, const size_t bufferLength, const char* operationBuffer, const size_t operationBufferSize) NN_NOEXCEPT;
    nn::Result ProcessNoDataOperation(const char* operationBuffer, const size_t operationBufferSize) NN_NOEXCEPT;

    // 使用する側でアラインされたバッファに直接 R/W する  Operation API
    nn::Result ProcessWriteOperationWithAlignedBuffer(char* inAlignedDataBuffer, const size_t alignedBufferLength, const char* operationBuffer, const size_t operationBufferSize) NN_NOEXCEPT;
    nn::Result ProcessReadOperationWithAlignedBuffer(char* outAlignedDataBuffer, const size_t alignedBufferLength, const char* operationBuffer, const size_t operationBufferSize) NN_NOEXCEPT;

    // 実コマンドを発行する API
    nn::Result ExecuteOperation(char* workAlignedBuffer, const size_t workAlignedBufferLength, const char* operationBuffer, const size_t operationBufferSize, const DataDirection direction) NN_NOEXCEPT;

    // OperationIndex だけを埋める API
    void SetAsicOperationIndex(const AsicOperationIndex operationIndex) NN_NOEXCEPT
    {
        memset(m_AsicOperationBuffer, 0, sizeof(m_AsicOperationBuffer));
        m_AsicOperationBuffer[0] = static_cast<u8>(operationIndex);
    }

    void SetCmd60TimeoutDefault()
    {
        m_SdmmcTimeoutMilliSeconds = Cmd60DefaultTimeOutMilliSeconds;
    }

    void SetCmd60TimeoutForErase()
    {
        m_SdmmcTimeoutMilliSeconds = EraseTimeOutMilliSeconds;
    }

    AsicOperation() NN_NOEXCEPT;
};


} } }
