﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/fssystem/fs_SparseStorage.h>
#include <nn/fssystem/utilTool/fs_BucketTreeRange.h>

namespace nn { namespace fssystem { namespace utilTool {

/**
 * @brief   SparseStorage で扱うデータを構築するクラスです。
 */
class SparseStorageBuilder
{
    NN_DISALLOW_COPY(SparseStorageBuilder);

public:
    typedef MemoryResource IAllocator;

    // スパース化で残すべきデータの領域を表す構造体
    typedef BucketTreeRange Range;

    // Range の配列を扱うクラス
    typedef BucketTreeRangeArray RangeArray;

    static const size_t BlockSizeMin = 512;

public:
    /**
     * @brief   コンストラクタです。
     */
    SparseStorageBuilder() NN_NOEXCEPT;

    /**
     * @brief   コンストラクタです。
     *
     * @param[in]   blockSize   ストレージの分割サイズ
     * @param[in]   eraseSize   ストレージの最小削除サイズ
     *
     * @pre
     *      - 512 <= blockSize
     *      - ispow2(blockSize) != false
     *      - blockSize <= eraseSize
     *      - eraseSize % blockSize == 0
     */
    SparseStorageBuilder(size_t blockSize, size_t eraseSize) NN_NOEXCEPT;

    /**
     * @brief   エントリ数 0 の SparseStorage を作成します。
     *
     * @param[in]   originalSize    元データのサイズ
     *
     * @pre
     *      - 0 <= originalSize
     */
    void Build(int64_t originalSize) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_GREATER_EQUAL(originalSize, 0);
        NN_SDK_REQUIRES_EQUAL(m_VirtualStorageSize, 0);

        m_VirtualStorageSize = originalSize;
    }

    /**
     * @brief   IndirectStorage を解析してテーブルを作成します。
     *
     * @param[in]   pAllocator              アロケータのポインタ
     * @param[in]   indirectNodeStorage     IndirectStorage のテーブルノードのストレージ
     * @param[in]   indirectEntryStorage    IndirectStorage のテーブルエントリのストレージ
     * @param[in]   indirectEntryCount      IndirectStorage のエントリ数
     * @param[in]   originalSize            元データのサイズ
     *
     * @return  実行結果を返します。
     *
     * @pre
     *      - pAllocator != nullptr
     *      - 0 < indirectEntryCount
     *      - 0 <= originalSize
     */
    Result Build(
               IAllocator* pAllocator,
               fs::SubStorage indirectNodeStorage,
               fs::SubStorage indirectEntryStorage,
               int indirectEntryCount,
               int64_t originalSize
           ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pAllocator);

        BucketTree indirectTable;
        NN_RESULT_DO(indirectTable.Initialize(
            pAllocator,
            indirectNodeStorage,
            indirectEntryStorage,
            IndirectStorage::NodeSize,
            sizeof(IndirectStorage::Entry),
            indirectEntryCount
        ));

        NN_RESULT_DO(Build(indirectTable, originalSize));

        NN_RESULT_SUCCESS;
    }

    /**
     * @brief   IndirectStorage を解析してテーブルを作成します。
     *
     * @param[in]   indirectTable   IndirectStorage のテーブル情報
     * @param[in]   originalSize    元データのサイズ
     *
     * @return  実行結果を返します。
     *
     * @pre
     *      - 0 <= originalSize
     */
    Result Build(const BucketTree& indirectTable, int64_t originalSize) NN_NOEXCEPT;

    /**
     * @brief   IndirectStorage を解析してテーブルを再構築します。
     *
     * @param[in]   pAllocator              アロケータのポインタ
     * @param[in]   indirectNodeStorage     IndirectStorage のテーブルノードのストレージ
     * @param[in]   indirectEntryStorage    IndirectStorage のテーブルエントリのストレージ
     * @param[in]   indirectEntryCount      IndirectStorage のエントリ数
     * @param[in]   originalSize            元データのサイズ
     *
     * @return  実行結果を返します。
     *
     * @pre
     *      - pAllocator != nullptr
     *      - 0 < indirectEntryCount
     *      - 0 <= originalSize
     */
    Result Rebuild(
               IAllocator* pAllocator,
               fs::SubStorage indirectNodeStorage,
               fs::SubStorage indirectEntryStorage,
               int indirectEntryCount,
               int64_t originalSize
           ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pAllocator);

        BucketTree indirectTable;
        NN_RESULT_DO(indirectTable.Initialize(
            pAllocator,
            indirectNodeStorage,
            indirectEntryStorage,
            IndirectStorage::NodeSize,
            sizeof(IndirectStorage::Entry),
            indirectEntryCount
        ));

        NN_RESULT_DO(Rebuild(indirectTable, originalSize));

        NN_RESULT_SUCCESS;
    }

    /**
     * @brief   IndirectStorage を解析してテーブルを再構築します。
     *
     * @param[in]   indirectTable   IndirectStorage のテーブル情報
     * @param[in]   originalSize    元データのサイズ
     *
     * @return  実行結果を返します。
     *
     * @pre
     *      - 0 <= originalSize
     */
    Result Rebuild(const BucketTree& indirectTable, int64_t originalSize) NN_NOEXCEPT;

    /**
     * @brief   構築済みの SparseStorage を取り込みます。
     *
     * @param[in]   pStorage    構築済みのストレージのポインタ
     *
     * @return  実行結果を返します。
     *
     * @pre
     *      - pStorage != nullptr
     */
    Result Import(SparseStorage* pStorage) NN_NOEXCEPT;

    /**
     * @brief   元データを含むストレージを設定します。
     *
     * @param[in]   storage     ストレージ
     *
     * @return  実行結果を返します。
     */
    Result SetDataStorage(fs::SubStorage storage) NN_NOEXCEPT;

    /**
     * @brief   元データを含むストレージを設定します。
     *
     * @param[in]   pStorage    ストレージのポインタ
     * @param[in]   offset      ストレージのオフセット
     * @param[in]   size        ストレージのサイズ
     */
    template< typename T >
    void SetDataStorage(T pStorage, int64_t offset, int64_t size) NN_NOEXCEPT
    {
        // 事前検証は SubStorage に任せる
        m_DataStorage = fs::SubStorage(pStorage, offset, size);
        m_DataStorageSize = size;
    }

    /**
     * @brief   スパース化済みのデータストレージを設定します。
     *
     * @param[in]   pStorage    ストレージのポインタ
     * @param[in]   offset      ストレージのオフセット
     * @param[in]   size        ストレージのサイズ
     */
    void SetDataStorage(std::unique_ptr<fs::IStorage>&& pStorage, int64_t offset, int64_t size) NN_NOEXCEPT;

    /**
     * @brief   テーブルのヘッダ情報のみを書き出します。
     *
     * @param[out]  outBuffer       ヘッダ情報を書き出すバッファのポインタ
     * pparam[in]   outBufferSize   ヘッダ情報を書き出すバッファのサイズ
     *
     * @pre
     *      - outBuffer != nullptr
     *      - 16 <= outBufferSize
     */
    void WriteHeader(void* outBuffer, size_t outBufferSize) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(outBuffer);
        NN_SDK_REQUIRES_GREATER_EQUAL(outBufferSize, sizeof(BucketTree::Header));
        NN_UNUSED(outBufferSize);

        BucketTree::Header header;
        header.Format(m_EntryCount);

        std::memcpy(outBuffer, &header, sizeof(header));
    }

    /**
     * @brief   テーブルを書き出します。
     *
     * @param[in]   pAllocator      アロケータのポインタ
     * @param[in]   headerStorage   テーブルヘッダのストレージ
     * @param[in]   nodeStorage     テーブルノードのストレージ
     * @param[in]   entryStorage    テーブルエントリのストレージ
     *
     * @return  実行結果を返します。
     *
     * @pre
     *      - pAllocator != nullptr
     */
    Result WriteTable(
               IAllocator* pAllocator,
               fs::SubStorage headerStorage,
               fs::SubStorage nodeStorage,
               fs::SubStorage entryStorage
           ) const NN_NOEXCEPT;

    /**
     * @brief   テーブルを元に必要なデータを読み込みます。
     *
     * @param[in]   offset  読み込むデータのオフセット
     * @param[in]   buffer  データを読み込むバッファのアドレス
     * @param[in]   size    読み込むデータのサイズ
     *
     * @return  実行結果を返します。
     *
     * @pre
     *      - 0 <= offset
     *      - size == 0 || buffer != nullptr
     *
     * offset,size で指定された領域が [0, QueryDataStorageSize()) を超えている場合、その領域は 0 埋めします。
     */
    Result ReadData(int64_t offset, void* buffer, size_t size) NN_NOEXCEPT
    {
        NN_RESULT_DO(m_RangeArray.ReadData(&m_DataStorage, m_PhysicalStorageSize, offset, buffer, size, nullptr));
        NN_RESULT_SUCCESS;
    }

    /**
     * @brief   テーブルでヘッダ部分のストレージのサイズを取得します。
     *
     * @return  ヘッダ部分のストレージサイズを返します。
     */
    int64_t QueryTableHeaderStorageSize() const NN_NOEXCEPT
    {
        return SparseStorage::QueryHeaderStorageSize();
    }

    /**
     * @brief   テーブルでノード部分のストレージのサイズを取得します。
     *
     * @return  ノード部分のストレージサイズを返します。
     */
    int64_t QueryTableNodeStorageSize() const NN_NOEXCEPT
    {
        return SparseStorage::QueryNodeStorageSize(m_EntryCount);
    }

    /**
     * @brief   テーブルでエントリ部分のストレージサイズを取得します。
     *
     * @return  エントリ部分のストレージサイズを返します。
     */
    int64_t QueryTableEntryStorageSize() const NN_NOEXCEPT
    {
        return SparseStorage::QueryEntryStorageSize(m_EntryCount);
    }

    /**
     * @brief   テーブルのストレージサイズを取得します。
     *
     * @return  テーブルのストレージサイズを返します。
     */
    int64_t QueryTableStorageSize() const NN_NOEXCEPT
    {
        return QueryTableHeaderStorageSize() +
               QueryTableNodeStorageSize() +
               QueryTableEntryStorageSize();
    }

    /**
     * @brief   必要なデータのストレージサイズを取得します。
     *
     * @return  必要なデータのストレージサイズを返します。
     */
    int64_t QueryDataStorageSize() const NN_NOEXCEPT
    {
        return m_PhysicalStorageSize;
    }

    /**
     * @brief   元データのストレージサイズを取得します。
     *
     * @return  元データのストレージサイズを返します。
     */
    int64_t GetOriginalStorageSize() const NN_NOEXCEPT
    {
        return m_VirtualStorageSize;
    }

    /**
     * @brief   SparseStorage のビルド情報を出力します。
     *
     * @param[in]   pAllocator          アロケータのポインタ
     * @param[in]   pTableFileStorage   テーブル情報を書き出すストレージのポインタ
     * @param[in]   pNodeFileStorage    ノード情報を書き出すストレージのポインタ
     * @param[in]   pEntryFileStorage   エントリ情報を書き出すストレージのポインタ
     *
     * @pre
     *      - pAllocator != nullptr
     *      - pTableFileStorage != nullptr
     *      - pNodeFileStorage != nullptr
     *      - pEntryFileStorage != nullptr
     */
    Result OutputBuildLog(
               IAllocator* pAllocator,
               fs::IStorage* pTableFileStorage,
               fs::IStorage* pNodeFileStorage,
               fs::IStorage* pEntryFileStorage
           ) const NN_NOEXCEPT;

public:
    // オーサリングツール・テスト向け
    const RangeArray& GetRangeArray() const NN_NOEXCEPT
    {
        return m_RangeArray;
    }

    int GetEntryCount() const NN_NOEXCEPT
    {
        return m_EntryCount;
    }

private:
    class RangeContainer;

private:
    const size_t m_BlockSize;
    const int64_t m_EraseSize;
    int m_EntryCount;
    RangeArray m_RangeArray;
    fs::SubStorage m_DataStorage;
    int64_t m_DataStorageSize;
    int64_t m_VirtualStorageSize;
    int64_t m_PhysicalStorageSize;
    std::unique_ptr<fs::IStorage> m_pStorageHolder;

    friend class SparseStorageBuilderTest;
};

}}}
