﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <algorithm>
#include <iterator>
#include <vector>
#include <nn/fssystem/utilTool/fs_BinaryMatch.h>

namespace nn { namespace fssystem { namespace utilTool {

// 最適化の進捗状況
enum class BinaryMatchOptimizePhase
{
    ShiftByAnchor,
    ShiftByLargeSplit,
    ShiftBySmallFit,
    ShiftByLeftJustify,
    ShiftBySoftSplit,
    Complete,
    Count = Complete
};

// std::partition 向けの整列ファンクタ
struct PartitionOrderFunctor
{
    bool operator()(const BinaryMatchResult& lhs, const BinaryMatchResult& rhs) const NN_NOEXCEPT
    {
        if (lhs.oldOffset == BinaryMatchResult::InvalidOffset)
        {
            if (rhs.oldOffset == BinaryMatchResult::InvalidOffset)
            {
                return lhs.newOffset < rhs.newOffset;
            }
            else
            {
                return false; // lhs > rhs
            }
        }
        else
        {
            if (rhs.oldOffset == BinaryMatchResult::InvalidOffset)
            {
                return true; // lhs < rhs
            }
            else
            {
                return lhs.oldOffset < rhs.oldOffset;
            }
        }
    }
};

// std::partition 向けの判定ファンクタ
struct PartitionPointFunctor
{
    bool operator()(const BinaryMatchResult& result) const NN_NOEXCEPT
    {
        return result.oldOffset != BinaryMatchResult::InvalidOffset;
    }
};

// 移動が必要な領域を表すクラス
class RelocationShiftRange
{
public:
    RelocationShiftRange() NN_NOEXCEPT
        : m_Offset(0)
        , m_Size(0)
    {
    }

    RelocationShiftRange(int64_t offset, int64_t size) NN_NOEXCEPT
        : m_Offset(offset)
        , m_Size(size)
    {
        NN_SDK_REQUIRES_GREATER_EQUAL(offset, 0);
        NN_SDK_REQUIRES_GREATER_EQUAL(size, 0);
    }

    bool operator<(const RelocationShiftRange& other) const NN_NOEXCEPT
    {
        return m_Offset < other.m_Offset;
    }

    void Empty() NN_NOEXCEPT
    {
        m_Size = 0;
    }

    void Advance(int64_t size) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_LESS_EQUAL(size, m_Size);
        m_Offset += size;
        m_Size -= size;
    }

    int64_t GetSize() const NN_NOEXCEPT
    {
        return m_Size;
    }

    int64_t GetOffset() const NN_NOEXCEPT
    {
        return m_Offset;
    }

private:
    int64_t m_Offset;
    int64_t m_Size;
};

// 移動させる移動が必要な領域を保持するコンテナ
class RelocationShiftRangeContainer
{
public:
    static const int64_t SplitSizeMin = 16 * 1024;
    static const int64_t LargeSplitSizeMin = 256 * 1024;

public:
    typedef std::vector<RelocationShiftRange> Container;
    typedef Container::iterator iterator;

public:
    RelocationShiftRangeContainer(
        std::vector<BinaryMatchResult>* pResults,
        int64_t endOffset
    ) NN_NOEXCEPT;

    ~RelocationShiftRangeContainer() NN_NOEXCEPT;

    // BinaryMatchResult の分割・移動を行うのに必要な情報を生成
    bool Build(int64_t expandedSize) NN_NOEXCEPT;

    // BinaryMatchResult がマッチ済みの領域の後続になるように移動する
    void ShiftByAnchor(int64_t limitNewOffset) NN_NOEXCEPT;

    // 小さな BinaryMatchResult をファーストフィットで移動する
    void ShiftBySmallFit() NN_NOEXCEPT;

    // 大きな BinaryMatchResult を分割して移動する
    void ShiftByLargeSplit() NN_NOEXCEPT;

    // 分割しないで BinaryMatchResult を前詰めしていく
    void ShiftByLeftJustify() NN_NOEXCEPT;

    // BinaryMatchResult を分割して前詰めしていく
    bool ShiftBySoftSplit(int64_t expandedSize) NN_NOEXCEPT;

    // BinaryMatchResult を後詰めしていく
    void ShiftByRightJustify() NN_NOEXCEPT;

private:
    bool ShiftByLargeSplit(const RelocationShiftRange& moveRange) NN_NOEXCEPT;

    bool ShiftBySoftSplit(
        std::vector<BinaryMatchResult>* pResults,
        const RelocationShiftRange& moveRange
    ) NN_NOEXCEPT;

    void SplitCommon(
        std::vector<BinaryMatchResult>* pResults,
        const RelocationShiftRange& moveRange,
        Container::iterator begin,
        Container::iterator end,
        BinaryMatchDetail detail
    ) NN_NOEXCEPT;

    void SetupSplit() NN_NOEXCEPT;

    void CleanupSplit() NN_NOEXCEPT;

    int64_t GetTotalFreeSize() const NN_NOEXCEPT
    {
        int64_t size = 0;
        for( const auto& range : m_FreeRanges )
        {
            size += range.GetSize();
        }
        return size;
    }

private:
    const int64_t m_EndOffset;
    std::vector<BinaryMatchResult>& m_Results;
    std::vector<RelocationShiftRange> m_MoveRanges;
    std::vector<RelocationShiftRange> m_FreeRanges;
    std::vector<RelocationShiftRange> m_FreeBackup;
    int64_t m_FreeRangeLastOffset;
};

// BinaryMatch の結果から oldOffset について重複を取り除く
void OptimizeBinaryMatchResult(
    std::vector<BinaryMatchResult>* pResults
) NN_NOEXCEPT;

// BinaryMatchResult の分割・移動を行う
void ShiftBinaryMatchResult(
    std::vector<BinaryMatchResult>* pResults,
    int64_t limitOldOffset,
    int64_t limitNewOffset,
    int64_t expandableSize,
    BinaryMatchProgress* pProgress
) NN_NOEXCEPT;

}}}
