﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <memory>
#include <nn/crypto/crypto_Sha256Generator.h>
#include <nn/fssystem/fs_NcaHeader.h>
#include <nn/util/util_BinTypes.h>

namespace nn { namespace fssystem { namespace utilTool {

class BinaryRegionArray;

/**
 * @brief   BinaryRegion の外部ファイルのヘッダを表します。
 */
struct BinaryRegionFileHeader
{
    static const uint32_t Signature = NN_UTIL_CREATE_SIGNATURE_4('B', 'R', 'F', '0');
    static const uint32_t Version = 2;
    static const char* const Extension;

    uint32_t signature;
    uint32_t version;
    char headerHash[crypto::Sha256Generator::HashSize];
    char header[NcaFsHeader::Size];
    char dataHash[crypto::Sha256Generator::HashSize];
    int64_t dataOffset;
    uint32_t regionSize;
    uint32_t regionCount;

    bool operator==(const BinaryRegionFileHeader& other) const NN_NOEXCEPT
    {
        return std::memcmp(&this->signature, &other.signature, offsetof(BinaryRegionFileHeader, regionCount)) == 0;
    }

    // 初期化
    void Initialize() NN_NOEXCEPT
    {
        this->signature = Signature;
        this->version = Version;
        this->dataOffset = 0;
        this->regionSize = 0;
        this->regionCount = 0;
    }

    // ヘッダー部分を設定
    void SetHeader(const char* buffer, size_t size) NN_NOEXCEPT;

    // データ部分を設定
    void SetData(int64_t offset, const char* buffer, size_t size) NN_NOEXCEPT;

    // ファイル名を生成
    void GenerateFileName(char* outBuffer, size_t outSize) const NN_NOEXCEPT;
};
NN_STATIC_ASSERT(std::is_pod<BinaryRegionFileHeader>::value);
NN_STATIC_ASSERT((sizeof(BinaryRegionFileHeader) % sizeof(int64_t)) == 0);

/**
 * @brief   BinaryRegion の外部ファイルを表します。
 */
class BinaryRegionFile
{
    NN_DISALLOW_COPY(BinaryRegionFile);

public:
    typedef BinaryRegionFileHeader Header;

public:
    // コンストラクタ
    BinaryRegionFile() NN_NOEXCEPT
        : m_Buffer(nullptr)
    {
        m_Header.Initialize();
    }

    // ヘッダを設定
    void SetHeader(const char* buffer, size_t size) NN_NOEXCEPT
    {
        // NOTE: 事前検証は BinaryRegionFileHeader::SetHeader() に任せる
        m_Header.SetHeader(buffer, size);
    }

    // データを設定
    void SetData(int64_t offset, const char* buffer, size_t size) NN_NOEXCEPT
    {
        // NOTE: 事前検証は BinaryRegionFileHeader::SetData() に任せる
        m_Header.SetData(offset, buffer, size);
    }

    // 読み込んだファイルのリージョンハッシュをチェック
    bool CheckRegion(char* buffer, size_t bufferSize) NN_NOEXCEPT;

    // リージョンハッシュを設定（CheckRegion() に失敗した場合に実行する）
    void SetRegion(const BinaryRegionArray& region) NN_NOEXCEPT;

    // リージョンハッシュを取得（CheckRegion() に成功した場合に実行できる）
    const BinaryRegionArray GetRegion() const NN_NOEXCEPT;

    // ヘッダを取得
    const Header& GetHeader() const NN_NOEXCEPT
    {
        return m_Header;
    }

    // データオフセットを取得
    int64_t GetDataOffset() const NN_NOEXCEPT
    {
        return m_Header.dataOffset;
    }

    // ファイル名を生成
    void GenerateFileName(char* outBuffer, size_t outSize) const NN_NOEXCEPT
    {
        m_Header.GenerateFileName(outBuffer, outSize);
    }

private:
    Header m_Header;
    char* m_Buffer;
};

}}}
