﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <algorithm>
#include <nn/fs/fs_SubStorage.h>

namespace nn { namespace fssystem {

    /**
    * @brief ストレージオブジェクト、オフセット、サイズの組み合わせを管理するためのクラスです。
    *
    *        SubStorage とほぼ同一の実装ですが、こちらはサイズを truncate (丸め込み)します。
    *        おもに BlockBufferfCacheFile と組み合わせて使用します。
    */
    class TruncatedSubStorage : public fs::SubStorage
    {
    public:
        TruncatedSubStorage() NN_NOEXCEPT
            : SubStorage()
        {
        }

        NN_IMPLICIT TruncatedSubStorage(const SubStorage& other) NN_NOEXCEPT
            : SubStorage(other)
        {
        }

        /**
        * @brief        ストレージの内容をバッファに読み込みます。
        *
        * @param[in]    offset  読み込み開始位置
        * @param[out]   buffer  読み込んだ内容をコピーするバッファ
        * @param[in]    size    読み込むデータサイズ
        *
        * @return       関数の処理結果を返します。
        */
        virtual Result Read(
                           int64_t offset,
                           void* buffer,
                           size_t size
                       ) NN_NOEXCEPT NN_OVERRIDE
        {
            if( size == 0 )
            {
                NN_RESULT_SUCCESS;
            }
            NN_RESULT_THROW_UNLESS(buffer != nullptr, nn::fs::ResultNullptrArgument());

            int64_t sizeStorage = 0;
            NN_RESULT_DO(GetSize(&sizeStorage));

            NN_SDK_ASSERT(offset <= sizeStorage);

            const auto sizeRead = static_cast<int64_t>(size);
            const auto sizeMax = static_cast<int64_t>(sizeStorage - offset);
            size_t sizeTruncated = static_cast<size_t>(std::min(sizeRead, sizeMax));

            return SubStorage::Read(offset, buffer, sizeTruncated);
        }

        /**
        * @brief        バッファの内容をファイルに書き込みます。
        *
        * @param[in]    offset  書き込み開始位置
        * @param[in]    buffer  書き込むデータ
        * @param[in]    size    書き込むデータサイズ
        * @param[in]    option  書き込みオプション
        *
        * @return       関数の処理結果を返します。
        */
        virtual Result Write(
                           int64_t offset,
                           const void* buffer,
                           size_t size
                       ) NN_NOEXCEPT NN_OVERRIDE
        {
            if( size == 0 )
            {
                NN_RESULT_SUCCESS;
            }
            NN_RESULT_THROW_UNLESS(buffer != nullptr, nn::fs::ResultNullptrArgument());

            int64_t sizeStorage = 0;
            NN_RESULT_DO(GetSize(&sizeStorage));

            NN_SDK_ASSERT(offset <= sizeStorage);

            const auto sizeWrite = static_cast<int64_t>(size);
            const auto sizeMax = static_cast<int64_t>(sizeStorage - offset);
            size_t sizeTruncated = static_cast<size_t>(std::min(sizeWrite, sizeMax));

            return SubStorage::Write(offset, buffer, sizeTruncated);
        }

        TruncatedSubStorage& operator=(const SubStorage& other) NN_NOEXCEPT
        {
            SubStorage::operator=(other);
            return *this;
        }
    };

}}
