﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>

#include <nn/fs/detail/fs_Newable.h>
#include <nn/fssystem/buffers/fs_BufferManagerUtility.h>
#include <nn/fssystem/save/fs_ISaveFileSystemDriver.h>

namespace nn { namespace fssystem { namespace save {

class ProxyFileSystemWithRetryingBufferAllocationFile : public IFile, nn::fs::detail::Newable
{
    NN_DISALLOW_COPY(ProxyFileSystemWithRetryingBufferAllocationFile);

    friend class ProxyFileSystemWithRetryingBufferAllocation;

public:
    /**
    * @brief        コンストラクタ
    *
    * @param[in]    pFile   下位ファイル
    */
    explicit ProxyFileSystemWithRetryingBufferAllocationFile() NN_NOEXCEPT
    : m_pFile(nullptr)
    {
    }

public:
    /**
    * @brief 初期化します。
    *
    * @param[in] pFile 下位ファイル
    */
    void Initialize(IFile* pFile) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pFile);
        NN_SDK_ASSERT(m_pFile == nullptr);
        m_pFile = pFile;
    }

    /**
    * @brief        ファイルの内容をバッファに読み込みます。
    *
    * @param[in]    offset  読み込み開始位置
    * @param[out]   buffer  読み込んだ内容をコピーするバッファ
    * @param[in]    size    読み込むデータサイズ
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result ReadBytes(
                       int64_t offset,
                       void* buffer,
                       size_t size
                   ) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_ASSERT_NOT_NULL(m_pFile);
        NN_RESULT_DO(buffers::DoContinouslyUntilBufferIsAllocated(
            [=]() NN_NOEXCEPT -> Result
            {
                buffers::ScopedBufferManagerContextRegistration scopedRegistration;
                NN_RESULT_DO(m_pFile->ReadBytes(offset, buffer, size));
                NN_RESULT_SUCCESS;
            },
            NN_CURRENT_FUNCTION_NAME
        ));
        NN_RESULT_SUCCESS;
    }

    /**
    * @brief        バッファの内容をファイルに書き込みます。
    *
    * @param[in]    offset  書き込み開始位置
    * @param[in]    buffer  書き込むデータ
    * @param[in]    size    書き込むデータサイズ
    * @param[in]    option  書き込みオプション
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result WriteBytes(
                       int64_t offset,
                       const void* buffer,
                       size_t size
                   ) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_ASSERT_NOT_NULL(m_pFile);
        NN_RESULT_DO(buffers::DoContinouslyUntilBufferIsAllocated(
            [=]() NN_NOEXCEPT -> Result
            {
                buffers::ScopedBufferManagerContextRegistration scopedRegistration;
                NN_RESULT_DO(m_pFile->WriteBytes(offset, buffer, size));
                NN_RESULT_SUCCESS;
            },
            NN_CURRENT_FUNCTION_NAME
        ));
        NN_RESULT_SUCCESS;
    }

    /**
    * @brief        ファイルサイズを取得します。
    *
    * @param[out]   outValue    ファイルサイズ
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result GetSize(int64_t* outValue) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_ASSERT_NOT_NULL(m_pFile);
        NN_RESULT_DO(buffers::DoContinouslyUntilBufferIsAllocated(
            [=]() NN_NOEXCEPT -> Result
            {
                buffers::ScopedBufferManagerContextRegistration scopedRegistration;
                NN_RESULT_DO(m_pFile->GetSize(outValue));
                NN_RESULT_SUCCESS;
            },
            NN_CURRENT_FUNCTION_NAME
        ));
        NN_RESULT_SUCCESS;
    }

    /**
    * @brief        ファイルサイズを変更します。
    *
    * @param[in]    size    変更後のファイルサイズ
    *
    * @return       失敗の結果を返します。
    */
    virtual Result SetSize(int64_t size) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_ASSERT_NOT_NULL(m_pFile);
        NN_RESULT_DO(buffers::DoContinouslyUntilBufferIsAllocated(
            [=]() NN_NOEXCEPT -> Result
            {
                buffers::ScopedBufferManagerContextRegistration scopedRegistration;
                NN_RESULT_DO(m_pFile->SetSize(size));
                NN_RESULT_SUCCESS;
            },
            NN_CURRENT_FUNCTION_NAME
        ));
        NN_RESULT_SUCCESS;
    }

    /**
    * @brief        フラッシュします。
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result Flush() NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_ASSERT_NOT_NULL(m_pFile);
        NN_RESULT_DO(buffers::DoContinouslyUntilBufferIsAllocated(
            [=]() NN_NOEXCEPT -> Result
            {
                buffers::ScopedBufferManagerContextRegistration scopedRegistration;
                NN_RESULT_DO(m_pFile->Flush());
                NN_RESULT_SUCCESS;
            },
            NN_CURRENT_FUNCTION_NAME
        ));
        NN_RESULT_SUCCESS;
    }

    /**
    * @brief        範囲指定処理を行います。
    *
    * @param[out]   outBuffer       範囲指定処理の結果を格納するバッファ
    * @param[in]    outBufferSize   範囲指定処理の結果を格納するバッファのサイズ
    * @param[in]    operationId     範囲指定処理の種類
    * @param[in]    offset          範囲指定処理開始位置
    * @param[in]    size            範囲指定処理を行うデータサイズ
    * @param[in]    inBuffer        範囲指定処理に渡すバッファ
    * @param[in]    inBufferSize    範囲指定処理に渡すバッファのサイズ
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result OperateRange(
                       void* outBuffer,
                       size_t outBufferSize,
                       fs::OperationId operationId,
                       int64_t offset,
                       int64_t size,
                       const void* inBuffer,
                       size_t inBufferSize
                   ) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_ASSERT_NOT_NULL(m_pFile);
        NN_RESULT_DO(buffers::DoContinouslyUntilBufferIsAllocated(
            [=]() NN_NOEXCEPT -> Result
            {
                buffers::ScopedBufferManagerContextRegistration scopedRegistration;
                NN_RESULT_DO(m_pFile->OperateRange(
                    outBuffer, outBufferSize, operationId, offset, size, inBuffer, inBufferSize));
                NN_RESULT_SUCCESS;
            },
            NN_CURRENT_FUNCTION_NAME
        ));
        NN_RESULT_SUCCESS;
    }

private:
    IFile* m_pFile;     //!< 下位ファイル
};

class ProxyFileSystemWithRetryingBufferAllocationDirectory : public IDirectory, nn::fs::detail::Newable
{
    NN_DISALLOW_COPY(ProxyFileSystemWithRetryingBufferAllocationDirectory);

    friend class ProxyFileSystemWithRetryingBufferAllocation;

public:
    /**
    * @brief        コンストラクタ
    */
    explicit ProxyFileSystemWithRetryingBufferAllocationDirectory() NN_NOEXCEPT
    : m_pDirectory(nullptr)
    {
    }

public:
    /**
    * @brief 初期化します。
    *
    * @param[in] pDirectory 下位ディレクトリ
    */
    void Initialize(IDirectory* pDirectory) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pDirectory);
        NN_SDK_ASSERT(m_pDirectory == nullptr);
        m_pDirectory = pDirectory;
    }

    /**
    * @brief        ディレクトリ内の子ディレクトリエントリーを取得します。
    *
    * @param[out]   outNumEntries   取得したエントリー数
    * @param[out]   outEntries      取得したエントリー情報
    *                               配列のサイズは numEntries 以上である必要があります。
    * @param[in]    numEntries      取得するエントリー数
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result Read(
                       int32_t* outNumEntries,
                       nn::fs::DirectoryEntry outEntries[],
                       int32_t numEntries
                   ) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_RESULT_DO(buffers::DoContinouslyUntilBufferIsAllocated(
            [=]() NN_NOEXCEPT -> Result
            {
                buffers::ScopedBufferManagerContextRegistration scopedRegistration;
                NN_RESULT_DO(m_pDirectory->Read(outNumEntries, outEntries, numEntries));
                NN_RESULT_SUCCESS;
            },
            NN_CURRENT_FUNCTION_NAME
        ));
        NN_RESULT_SUCCESS;
    }

private:
    IDirectory* m_pDirectory;
};

/**
 * @brief 下位ファイルシステムに対して ResultBufferAllocationFailed に対してはリトライをかける
 *        プロキシファイルシステムです。
 */
class ProxyFileSystemWithRetryingBufferAllocation : public IFileSystem
{
    NN_DISALLOW_COPY(ProxyFileSystemWithRetryingBufferAllocation);

public:
    /**
    * @brief コンストラクタ
    */
    ProxyFileSystemWithRetryingBufferAllocation() NN_NOEXCEPT
    : m_pFileSystem(nullptr)
    {
    }

public:
    /**
    * @brief マウントします。
    *
    * @param[in] pFileSystem 下位ファイルシステム
    */
    void Initialize(IFileSystem* pFileSystem) NN_NOEXCEPT
    {
        m_pFileSystem = pFileSystem;
    }

    /**
    * @brief アンマウントします。
    */
    void Finalize() NN_NOEXCEPT
    {
        m_pFileSystem = nullptr;
    }

    /**
    * @brief        ファイルを作成します。
    *
    * @param[in]    path    ファイルパス
    * @param[in]    size    ファイルサイズ
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result CreateFile(const Path& path, int64_t size) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_ASSERT_NOT_NULL(m_pFileSystem);
        NN_RESULT_DO(buffers::DoContinouslyUntilBufferIsAllocated(
            [=]() NN_NOEXCEPT -> Result
            {
                buffers::ScopedBufferManagerContextRegistration scopedRegistration;
                NN_RESULT_DO(m_pFileSystem->CreateFile(path, size));
                NN_RESULT_SUCCESS;
            },
            NN_CURRENT_FUNCTION_NAME
        ));
        NN_RESULT_SUCCESS;
    }

    /**
    * @brief        ディレクトリを作成します。
    *
    * @param[in]    path ディレクトリパス
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result CreateDirectory(const Path& path) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_ASSERT_NOT_NULL(m_pFileSystem);
        NN_RESULT_DO(buffers::DoContinouslyUntilBufferIsAllocated(
            [=]() NN_NOEXCEPT -> Result
            {
                buffers::ScopedBufferManagerContextRegistration scopedRegistration;
                NN_RESULT_DO(m_pFileSystem->CreateDirectory(path));
                NN_RESULT_SUCCESS;
            },
            NN_CURRENT_FUNCTION_NAME
        ));
        NN_RESULT_SUCCESS;
    }

    /**
    * @brief        ファイルを開きます。
    *
    * @param[out]   outValue    ファイルオブジェクト
    * @param[in]    path        ファイルパス
    * @param[in]    mode        オープンモード
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result OpenFile(
                       IFile** outValue,
                       const Path& path,
                       int mode
                   ) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_ASSERT_NOT_NULL(m_pFileSystem);

        *outValue = nullptr;

        NN_RESULT_DO(buffers::DoContinouslyUntilBufferIsAllocated(
            [=]() NN_NOEXCEPT -> Result
            {
                buffers::ScopedBufferManagerContextRegistration scopedRegistration;

                auto pFileWrapper = new ProxyFileSystemWithRetryingBufferAllocationFile();
                NN_RESULT_THROW_UNLESS(pFileWrapper != nullptr, fs::ResultAllocationMemoryFailedNew());

                IFile* pInnerFile = nullptr;
                auto result = m_pFileSystem->OpenFile(&pInnerFile, path, mode);
                if( result.IsFailure() )
                {
                    delete pFileWrapper;
                    NN_RESULT_THROW(result);
                }

                pFileWrapper->Initialize(pInnerFile);
                *outValue = pFileWrapper;

                NN_RESULT_SUCCESS;
            },
            NN_CURRENT_FUNCTION_NAME
        ));
        NN_RESULT_SUCCESS;
    }

    /**
    * @brief        ディレクトリを開きます。
    *
    * @param[out]   outValue    ディレクトリオブジェクト
    * @param[in]    path        ディレクトリパス
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result OpenDirectory(
                       IDirectory** outValue,
                       const Path& path
                   ) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_ASSERT_NOT_NULL(m_pFileSystem);

        *outValue = nullptr;

        NN_RESULT_DO(buffers::DoContinouslyUntilBufferIsAllocated(
            [=]() NN_NOEXCEPT -> Result
            {
                buffers::ScopedBufferManagerContextRegistration scopedRegistration;

                auto pDirectoryWrapper = new ProxyFileSystemWithRetryingBufferAllocationDirectory();
                NN_RESULT_THROW_UNLESS(pDirectoryWrapper != nullptr, fs::ResultAllocationMemoryFailedNew());

                IDirectory* pInnerDirectory = nullptr;
                auto result = m_pFileSystem->OpenDirectory(&pInnerDirectory, path);
                if( result.IsFailure() )
                {
                    delete pDirectoryWrapper;
                    NN_RESULT_THROW(result);
                }

                pDirectoryWrapper->Initialize(pInnerDirectory);
                *outValue = pDirectoryWrapper;

                NN_RESULT_SUCCESS;
            },
            NN_CURRENT_FUNCTION_NAME
        ));
        NN_RESULT_SUCCESS;
    }

    /**
    * @brief        ファイルを閉じます。
    *
    * @param[in]    pFile   ファイルオブジェクト
    */
    virtual void CloseFile(IFile* pFile) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_ASSERT_NOT_NULL(m_pFileSystem);
        auto pFileWrapper = static_cast<ProxyFileSystemWithRetryingBufferAllocationFile*>(pFile);
        m_pFileSystem->CloseFile(pFileWrapper->m_pFile);
        delete pFileWrapper;
    }

    /**
    * @brief        ディレクトリを閉じます。
    *
    * @param[in]    pDirectory  ディレクトリオブジェクト
    */
    virtual void CloseDirectory(IDirectory* pDirectory) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_ASSERT_NOT_NULL(m_pFileSystem);
        auto pDirectoryWrapper = static_cast<ProxyFileSystemWithRetryingBufferAllocationDirectory*>(pDirectory);
        m_pFileSystem->CloseDirectory(pDirectoryWrapper->m_pDirectory);
        delete pDirectoryWrapper;
    }

    /**
    * @brief        ファイルを削除します。
    *
    * @param[in]    path    ファイルパス
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result DeleteFile(const Path& path) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_ASSERT_NOT_NULL(m_pFileSystem);
        NN_RESULT_DO(buffers::DoContinouslyUntilBufferIsAllocated(
            [=]() NN_NOEXCEPT -> Result
            {
                buffers::ScopedBufferManagerContextRegistration scopedRegistration;
                NN_RESULT_DO(m_pFileSystem->DeleteFile(path));
                NN_RESULT_SUCCESS;
            },
            NN_CURRENT_FUNCTION_NAME
        ));
        NN_RESULT_SUCCESS;
    }

    /**
    * @brief        ディレクトリを削除します。
    *
    * @param[in]    path    ディレクトリパス
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result DeleteDirectory(const Path& path) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_ASSERT_NOT_NULL(m_pFileSystem);
        NN_RESULT_DO(buffers::DoContinouslyUntilBufferIsAllocated(
            [=]() NN_NOEXCEPT -> Result
            {
                buffers::ScopedBufferManagerContextRegistration scopedRegistration;
                NN_RESULT_DO(m_pFileSystem->DeleteDirectory(path));
                NN_RESULT_SUCCESS;
            },
            NN_CURRENT_FUNCTION_NAME
        ));
        NN_RESULT_SUCCESS;
    }

    /**
    * @brief        ディレクトリを再帰的に削除します。
    *
    * @param[in]    path    ディレクトリパス
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result DeleteDirectoryRecursively(const Path& path) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_ASSERT_NOT_NULL(m_pFileSystem);
        NN_RESULT_DO(buffers::DoContinouslyUntilBufferIsAllocated(
            [=]() NN_NOEXCEPT -> Result
            {
                buffers::ScopedBufferManagerContextRegistration scopedRegistration;
                NN_RESULT_DO(m_pFileSystem->DeleteDirectoryRecursively(path));
                NN_RESULT_SUCCESS;
            },
            NN_CURRENT_FUNCTION_NAME
        ));
        NN_RESULT_SUCCESS;
    }

    /**
    * @brief        ファイルをリネームします。
    *
    * @param[in]    oldPath 変更対象のファイルパス
    * @param[in]    newPath 変更後のファイルパス
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result RenameFile(const Path& oldPath, const Path& newPath) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_ASSERT_NOT_NULL(m_pFileSystem);
        NN_RESULT_DO(buffers::DoContinouslyUntilBufferIsAllocated(
            [=]() NN_NOEXCEPT -> Result
            {
                buffers::ScopedBufferManagerContextRegistration scopedRegistration;
                NN_RESULT_DO(m_pFileSystem->RenameFile(oldPath, newPath));
                NN_RESULT_SUCCESS;
            },
            NN_CURRENT_FUNCTION_NAME
        ));
        NN_RESULT_SUCCESS;
    }

    /**
    * @brief        ディレクトリをリネームします。
    *
    * @param[in]    oldPath 変更対象のディレクトリパス
    * @param[in]    newPath 変更後のディレクトリパス
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result RenameDirectory(const Path& oldPath, const Path& newPath) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_ASSERT_NOT_NULL(m_pFileSystem);
        NN_RESULT_DO(buffers::DoContinouslyUntilBufferIsAllocated(
            [=]() NN_NOEXCEPT -> Result
            {
                buffers::ScopedBufferManagerContextRegistration scopedRegistration;
                NN_RESULT_DO(m_pFileSystem->RenameDirectory(oldPath, newPath));
                NN_RESULT_SUCCESS;
            },
            NN_CURRENT_FUNCTION_NAME
        ));
        NN_RESULT_SUCCESS;
    }

    /**
    * @brief        指定したファイルが存在しているかどうかを取得します。
    *
    * @param[out]   outValue    ファイルが存在しているかどうか
    * @param[in]    path        ファイルパス
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result HasFile(bool* outValue, const Path& path) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_ASSERT_NOT_NULL(m_pFileSystem);
        NN_RESULT_DO(buffers::DoContinouslyUntilBufferIsAllocated(
            [=]() NN_NOEXCEPT -> Result
            {
                buffers::ScopedBufferManagerContextRegistration scopedRegistration;
                NN_RESULT_DO(m_pFileSystem->HasFile(outValue, path));
                NN_RESULT_SUCCESS;
            },
            NN_CURRENT_FUNCTION_NAME
        ));
        NN_RESULT_SUCCESS;
    }

    /**
    * @brief        指定したディレクトリが存在しているかどうかを取得します。
    *
    * @param[out]   outValue    ディレクトリが存在しているかどうか
    * @param[in]    path        ディレクトリパス
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result HasDirectory(bool* outValue, const Path& path) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_ASSERT_NOT_NULL(m_pFileSystem);
        NN_RESULT_DO(buffers::DoContinouslyUntilBufferIsAllocated(
            [=]() NN_NOEXCEPT -> Result
            {
                buffers::ScopedBufferManagerContextRegistration scopedRegistration;
                NN_RESULT_DO(m_pFileSystem->HasDirectory(outValue, path));
                NN_RESULT_SUCCESS;
            },
            NN_CURRENT_FUNCTION_NAME
        ));
        NN_RESULT_SUCCESS;
    }

    /**
    * @brief        空き領域のサイズをバイト数単位で取得します。
    *
    * @param[out]   outValue    空き領域のサイズ
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result GetFreeBytes(int64_t* outValue) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_ASSERT_NOT_NULL(m_pFileSystem);
        NN_RESULT_DO(buffers::DoContinouslyUntilBufferIsAllocated(
            [=]() NN_NOEXCEPT -> Result
            {
                buffers::ScopedBufferManagerContextRegistration scopedRegistration;
                NN_RESULT_DO(m_pFileSystem->GetFreeBytes(outValue));
                NN_RESULT_SUCCESS;
            },
            NN_CURRENT_FUNCTION_NAME
        ));
        NN_RESULT_SUCCESS;
    }

private:
    IFileSystem* m_pFileSystem;
};

}}}

