﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>

#include <nn/util/util_Optional.h>
#include <nn/crypto/crypto_Sha256Generator.h>
#include <nn/fs/fs_MemoryStorage.h>
#include <nn/fssystem/buffers/fs_BufferManagerUtility.h>
#include <nn/fssystem/save/fs_ISaveFileSystemDriver.h>
#include <nn/fssystem/save/fs_SaveFileStorage.h>
#include <nn/fssystem/save/fs_SaveFileInFile.h>
#include <nn/fssystem/save/fs_JournalIntegritySaveDataStorage.h>
#include <nn/fssystem/save/fs_HierarchicalDuplexStorage.h>
#include <nn/fssystem/save/fs_HierarchicalIntegrityVerificationStorage.h>
#include <nn/fssystem/save/fs_SaveDataFileSystemCore.h>
#include <nn/fssystem/save/fs_BlockCacheBufferedStorage.h>
#include <nn/fssystem/save/fs_JournalStorage.h>
#include <nn/fssystem/save/fs_RemapStorage.h>
#include <nn/fssystem/fs_IMacGenerator.h>

namespace nn { namespace fssystem { namespace save {

class FsJournalIntegritySaveDataFileSystemTest;
class IInternalStorageFileSystemVisitor;

/**
* @brief ジャーナリング+完全性検証を行うセーブデータファイルシステムクラスです。
*/
class JournalIntegritySaveDataFileSystem : public IFileSystem
{
    NN_DISALLOW_COPY(JournalIntegritySaveDataFileSystem);
public:
    //! タイムスタンプ、位置情報などを書き込むための拡張データ
    struct ExtraData
    {
        char data[512];
    };
    NN_STATIC_ASSERT(std::is_pod<ExtraData>::value);
    static const char InternalStorageFileNameIntegritySeed[];

public:
    typedef void (*GenerateRandomFunction)(void* pData, size_t size);

private:
    static const size_t ControlAreaHashSize = nn::crypto::Sha256Generator::HashSize;

    /**
    * ファイルシステム レイアウト情報ヘッダ
    *
    * ファイルシステム レイアウト情報ヘッダに相当する部分は
    * 二重化、ジャーナリング、完全性検証範囲外とせざるを得ません。
    * マスター管理領域内に配置し、マスター管理領域によって保護されます。
    */
    struct FileSystemLayoutHeader
    {
        int32_t magic;                                  //!< 初期化チェック用コード
        int32_t version;                                //!< アーカイバーのバージョン

        char hash[ControlAreaHashSize];                 //!< 管理領域のハッシュ

        // ファイルシステム内
        int64_t offsetFileSystemRemapMeta;
        int64_t sizeFileSystemRemapMeta;
        int64_t offsetMetaRemapMeta;
        int64_t sizeMetaRemapMeta;
        int64_t offsetFileSystemRemapData;
        int64_t sizeFileSystemRemapData;

        int64_t offsetsBitmapL2[2];
        int64_t sizeBitmapL2;
        int64_t offsetsMeta[2];
        int64_t sizeMeta;
        int64_t offsetSaveData;
        int64_t sizeSaveDataCoreOriginal;
        int64_t sizeSaveDataCore;
        int64_t sizeReservedArea;

        // マスター管理領域内
        int64_t offsetsBitmapL1[2];
        int64_t sizeBitmapL1;
        int64_t offsetMasterHash;
        int64_t offsetOriginalMasterHash;
        int64_t sizeMasterHash;

        // メタデータ内
        int64_t offsetJournalTable;
        int64_t sizeJournalTable;
        int64_t offsetJournalBitmapPhysical;
        int64_t sizeJournalBitmapPhysical;
        int64_t offsetJournalBitmapVirtual;
        int64_t sizeJournalBitmapVirtual;
        int64_t offsetJournalBitmapUnassigned;
        int64_t sizeJournalBitmapUnassigned;
        int64_t offsetHierarchicalIntegrityLayeredHashL1;
        int64_t sizeHierarchicalIntegrityLayeredHashL1;
        int64_t offsetHierarchicalIntegrityLayeredHashL2;
        int64_t sizeHierarchicalIntegrityLayeredHashL2;
        int64_t offsetHierarchicalIntegrityLayeredHashL3;
        int64_t sizeHierarchicalIntegrityLayeredHashL3;
        int64_t offsetSaveDataMeta;
        int64_t sizeSaveDataMeta;

        bool masterSelectBitmap;
        char reserved1[7];

        int64_t offsetSaveDataMetaMasterHash;
        int64_t offsetOriginalSaveDataMetaMasterHash;
        int64_t offsetLayeredHashSaveDataMetaL1;
        int64_t sizeLayeredHashSaveDataMetaL1;
        int64_t offsetLayeredHashSaveDataMetaL2;
        int64_t sizeLayeredHashSaveDataMetaL2;

        char reserved2[112];
    };
    NN_STATIC_ASSERT(sizeof(FileSystemLayoutHeader) == 512);
    NN_STATIC_ASSERT(std::is_pod<FileSystemLayoutHeader>::value);

    //! マスター管理領域
    struct MasterHeader
    {
        static const int SignatureSize = 256; //!< 署名サイズ

        struct ControlArea
        {
            char signature[SignatureSize];    //!< 署名
            FileSystemLayoutHeader header;    //!< ファイルシステムヘッダ
        } controlArea;
        NN_STATIC_ASSERT(std::is_pod<ControlArea>::value);

        HierarchicalDuplexMetaInformation hierarchicalDuplexControlArea;
        HierarchicalIntegrityVerificationMetaInformation hierarchicalIntegrityControlArea;
        JournalStorage::ControlArea journalControlArea;
        SaveDataFileSystemCore::FileSystemHeader saveDataControlArea;
        RemapStorage::ControlArea fileSystemRemapControlArea;
        RemapStorage::ControlArea metaRemapControlArea;

        struct CommitData
        {
            int64_t counterForBundledCommit;
        } commitData;
        NN_STATIC_ASSERT(std::is_pod<CommitData>::value);

        ExtraData extraData;                    //!< 拡張データ
        ExtraData originalExtraData;            //!< 更新前の拡張データ

        HierarchicalIntegrityVerificationMetaInformation saveDataMetaHashControlArea;

        // マスター選択ビットマップ
        // マスターハッシュ
        // 仮コミットデータ
        // セーブデータメタのマスターハッシュ
        // 16 KB まで 0 埋め
        char reserved[16 * 1024
            - sizeof(ControlArea)
            - (sizeof(HierarchicalDuplexMetaInformation) + sizeof(int32_t))
            - sizeof(HierarchicalIntegrityVerificationMetaInformation)
            - sizeof(JournalStorage::ControlArea)
            - sizeof(SaveDataFileSystemCore::FileSystemHeader)
            - sizeof(RemapStorage::ControlArea) * 2
            - sizeof(CommitData)
            - sizeof(ExtraData) * 2
            - sizeof(HierarchicalIntegrityVerificationMetaInformation)
            ];
    };
    NN_STATIC_ASSERT(sizeof(MasterHeader) == 16 * 1024);
    NN_STATIC_ASSERT(std::is_pod<MasterHeader>::value);

    //! 管理領域を保持するためのクラス
    class ControlAreaHolder
    {
    public:
        ControlAreaHolder() NN_NOEXCEPT;
        Result Reset(const MasterHeader* pAtomic) NN_NOEXCEPT;
        void Clear() NN_NOEXCEPT;
        void Commit(MasterHeader* pAtomic) const NN_NOEXCEPT;
        MasterHeader::CommitData* GetCommitData() const NN_NOEXCEPT;
        ExtraData* GetExtraData() NN_NOEXCEPT;
        const nn::fs::SubStorage GetDuplexControlAreaStorage() NN_NOEXCEPT;
        const nn::fs::SubStorage GetFileSystemRemapControlAreaStorage() NN_NOEXCEPT;
        const nn::fs::SubStorage GetMetaRemapControlAreaStorage() NN_NOEXCEPT;
        const nn::fs::SubStorage GetIntegrityControlAreaStorage() NN_NOEXCEPT;
        const nn::fs::SubStorage GetJournalControlAreaStorage() NN_NOEXCEPT;
        const nn::fs::SubStorage GetSaveDataControlAreaStorage() NN_NOEXCEPT;
        const nn::fs::SubStorage GetMasterHashStorage() NN_NOEXCEPT;
        const nn::fs::SubStorage GetL1BitmapStorage(int index) NN_NOEXCEPT;
        const nn::fs::SubStorage GetSaveDataMetaMasterHashStorage() NN_NOEXCEPT;
        const nn::fs::SubStorage GetLayeredHashSaveDataMetaControlAreaStorage() NN_NOEXCEPT;

        int32_t GetVersion() const NN_NOEXCEPT
        {
            return m_Version;
        }

    private:
        std::unique_ptr<char[], nn::fs::detail::Deleter> m_pHeader;
        std::unique_ptr<nn::fs::MemoryStorage> m_pStorage;
        size_t m_BufferSize;
        size_t m_BitmapSize;
        int32_t m_Version;
    };

public:
    /**
    * @brief        ジャーナリング+完全性検証機能付きセーブデータファイルシステムをフォーマットするために必要なサイズを取得します。
    *
    * @param[out]   outSize                 ジャーナリング+完全性検証に必要なサイズを取得します。
    * @param[in]    countExpandMax          拡張できる回数の最大
    * @param[in]    inputParamDuplex        二重化パラメータ
    * @param[in]    inputParamIntegrity     完全性検証パラメータ
    * @param[in]    sizeBlock               ブロックサイズ
    * @param[in]    countDataBlock          データブロック数
    * @param[in]    countReservedBlock      予約領域のブロック数
    *
    * @return       関数の処理結果を返します。
    */
    static Result QuerySize(
                      int64_t* outSize,
                      int countExpandMax,
                      const HierarchicalDuplexStorageControlArea::InputParam& inputParamDuplex,
                      const HierarchicalIntegrityVerificationStorageControlArea::InputParam& inputParamIntegrity,
                      uint32_t sizeBlock,
                      uint32_t countDataBlock,
                      uint32_t countReservedBlock
                  ) NN_NOEXCEPT;

    /**
    * @brief        ジャーナリング+完全性検証機能付きセーブデータファイルシステムをフォーマットするために必要なサイズを取得します。
    *
    * @param[out]   outSize                 ジャーナリング+完全性検証に必要なサイズを取得します。
    * @param[in]    countExpandMax          拡張できる回数の最大
    * @param[in]    inputParamDuplex        二重化パラメータ
    * @param[in]    inputParamIntegrity     完全性検証パラメータ
    * @param[in]    sizeBlock               ブロックサイズ
    * @param[in]    countDataBlock          データブロック数
    * @param[in]    countReservedBlock      予約領域のブロック数
    * @param[in]    version             セーブデータバージョン
    *
    * @return       関数の処理結果を返します。
    */
    static Result QuerySize(
                      int64_t* outSize,
                      int countExpandMax,
                      const HierarchicalDuplexStorageControlArea::InputParam& inputParamDuplex,
                      const HierarchicalIntegrityVerificationStorageControlArea::InputParam& inputParamIntegrity,
                      uint32_t sizeBlock,
                      uint32_t countDataBlock,
                      uint32_t countReservedBlock,
                      uint32_t version
                  ) NN_NOEXCEPT;

    /**
    * @brief        ジャーナリング+完全性検証機能付きセーブデータファイルシステムとしてフォーマットします。
    *
    * @param[in]    storage ストレージ
    * @param[in]    countExpandMax 拡張できる回数の最大
    * @param[in]    inputParamDuplex 二重化パラメータ
    * @param[in]    inputParamIntegrity 完全性検証パラメータ
    * @param[in]    sizeBlock ブロックサイズ
    * @param[in]    countDataBlock データブロック数
    * @param[in]    countlReservedBlock 予約領域のブロック数
    * @param[in]    pIntegrityCacheBufferSet キャッシュマネージャ
    * @param[in]    pDuplicateCacheBuffer キャッシュマネージャ
    * @param[in]    pLocker キャッシュマネージャのスレッド間同期オブジェクト
    * @param[in]    pMacGenerator MAC 生成クラス
    *
    * @return       関数の処理結果を返します。
    */
    static Result Format(
                      fs::SubStorage storage,
                      int countExpandMax,
                      const HierarchicalDuplexStorageControlArea::InputParam& inputParamDuplex,
                      const HierarchicalIntegrityVerificationStorageControlArea::InputParam& inputParamIntegrity,
                      uint32_t sizeBlock,
                      uint32_t countDataBlock,
                      uint32_t countReservedBlock,
                      FilesystemBufferManagerSet* pIntegrityCacheBufferSet,
                      IBufferManager* pDuplicateCacheBuffer,
                      os::Mutex* pLocker,
                      IMacGenerator* pMacGenerator,
                      const nn::fs::SaveDataHashSalt& hashSalt
                  ) NN_NOEXCEPT;

    /**
    * @brief        ジャーナリング+完全性検証機能付きセーブデータファイルシステムを拡張します。
    *
    * @param[in]    storage ストレージ
    * @param[in]    sizeBlock ブロックサイズ
    * @param[in]    countDataBlock 拡張後のデータブロック数
    * @param[in]    countlReservedBlock 拡張後の予約領域のブロック数
    * @param[in]    pDuplicateCacheBuffer キャッシュマネージャ
    * @param[in]    pLocker キャッシュマネージャのスレッド間同期オブジェクト
    * @param[in]    pMacGenerator MAC 生成クラス
    * @param[in]    minimumVersion 最低バージョン
    *
    * @return       関数の処理結果を返します。
    */
    static Result Expand(
                      fs::SubStorage storage,
                      uint32_t sizeBlock,
                      uint32_t countDataBlock,
                      uint32_t countReservedBlock,
                      IBufferManager* pDuplicateCacheBuffer,
                      os::Mutex* pLocker,
                      IMacGenerator* pMacGenerator,
                      uint32_t minimumVersion
                  ) NN_NOEXCEPT;

    /**
    * @brief        拡張データを取得します。
    *
    * @param[out]   outData         拡張データ
    * @param[in]    storage         ストレージ
    * @param[in]    pBufferManager  バッファマネージャ
    * @param[in]    pMacGenerator   MAC 生成クラス
    * @param[in]    minimumVersion  最低バージョン
    *
    * @return       関数の処理結果を返します。
    *
    * @details      最後にコミットした際の拡張データを読み込みます。
    *               すなわち、WriteExtraData で書き込んだデータは
    *               コミットするまでこの関数では読み込めません。
    *               マウントしていなくても実行可能です。
    */
    static Result ReadExtraData(
                      ExtraData* outData,
                      fs::SubStorage storage,
                      IBufferManager* pBufferManager,
                      IMacGenerator* pMacGenerator,
                      uint32_t minimumVersion
                  ) NN_NOEXCEPT;

    /**
    * @brief        コンストラクタ
    */
    JournalIntegritySaveDataFileSystem() NN_NOEXCEPT;

    /**
    * @brief        デストラクタ
    */
    virtual ~JournalIntegritySaveDataFileSystem() NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        ジャーナリング+完全性検証付きセーブデータファイルシステムをマウントします。
    *
    * @param[in]    storage                     ストレージ
    * @param[in]    pIntegrityCacheBufferSet    キャッシュマネージャ
    * @param[in]    pDuplicateCacheBuffer       キャッシュマネージャ
    * @param[in]    pLocker                     ファイルシステムに対する同期オブジェクト
    * @param[in]    pMacGenerator               MAC 生成クラス
    * @param[in]    minimumVersion              最低バージョン
    *
    * @return       関数の処理結果を返します。
    */
    Result Initialize(
               fs::SubStorage storage,
               FilesystemBufferManagerSet* pIntegrityCacheBufferSet,
               IBufferManager* pDuplicateCacheBuffer,
               os::Mutex* pLocker,
               IMacGenerator* pMacGenerator,
               uint32_t minimumVersion
           ) NN_NOEXCEPT;

    /**
    * @brief        ジャーナリング+完全性検証付きセーブデータファイルシステムをアンマウントします。
    *
    * @details      アンマウント時に自動Commitは行いません。
    */
    void Finalize() NN_NOEXCEPT;

    /**
    * @brief        ファイルシステムに対してコミット操作を行います。
    *
    * @return       関数の処理結果を返します。
    */
    Result CommitFileSystem() NN_NOEXCEPT
    {
        return CommitFileSystemCore(0);
    }

    /**
    * @brief        ファイルシステムに対して仮コミット操作を行います。
    *               仮コミットについてはセーブデータ内部仕様書を参照してください。
    *
    * @param[in]    counterForBundledCommit 非ゼロのカウンタ値
    *
    * @return       関数の処理結果を返します。
    *
    * @pre
    *               - counterForBundledCommit != 0
    */
    Result CommitFileSystemProvisionally(int64_t counterForBundledCommit) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_EQUAL(counterForBundledCommit, 0);
        NN_RESULT_THROW_UNLESS(counterForBundledCommit != 0, nn::fs::ResultInvalidAlignment());
        return CommitFileSystemCore(counterForBundledCommit);
    }

    /**
    * @brief        ファイルシステムを巻き戻します。
    *
    * @return       関数の処理結果を返します。
    */
    Result RollbackFileSystem() NN_NOEXCEPT;

    /**
    * @brief        複数セーブデータの一括コミット用のカウンタ値を取得します。
    *
    * @return       カウンタ値を返します。
    */
    int64_t GetCounterForBundledCommit() const NN_NOEXCEPT
    {
        return m_ControlArea.GetCommitData()->counterForBundledCommit;
    }

    /**
    * @brief        拡張データを更新します。
    *
    * @param[in]    extraData   拡張データ
    *
    * @return       関数の処理結果を返します。
    */
    Result WriteExtraData(
               const ExtraData& extraData
           ) NN_NOEXCEPT;

    /**
    * @brief        拡張データを取得します。
    *
    * @param[out]   outData     拡張データ
    *
    * @details      WriteExtraData で書き込んだ値が読み込まれます。
    */
    void ReadExtraData(
             ExtraData* outData
         ) NN_NOEXCEPT;

    /**
    * @brief        管理領域のパラメータを抽出します。
    *
    * @param[out]   outParamDuplex          二重化のパラメータ
    * @param[out]   outParamIntegrity       完全性検証のパラメータ
    * @param[out]   outCountDataBlock       データ領域のブロック数
    * @param[out]   outCountJournalBlock    ジャーナル領域のブロック数
    * @param[out]   outBlockSize            ブロックサイズ
    * @param[out]   outCountExpandMax       拡張回数の最大
    * @param[out]   outVersion              セーブデータバージョン
    */
    void ExtractParameters(
        HierarchicalDuplexStorageControlArea::InputParam* outParamDuplex,
        HierarchicalIntegrityVerificationStorageControlArea::InputParam* outParamIntegrity,
        uint32_t* outCountDataBlock,
        uint32_t* outCountJournalBlock,
        int32_t* outBlockSize,
        int* outCountExpandMax,
        uint32_t* outVersion
        ) const NN_NOEXCEPT;

    /**
    * @brief        セーブデータの署名 (MAC) を更新します。
    *
    * @param[in]    storage                      セーブデータのストレージ
    * @param[in]    pMacGenerator                MAC 生成クラス
    * @param[in]    minimumVersion               最低バージョン
    *
    * @return       関数の処理結果を返します。
    */
    static Result UpdateMac(
        fs::SubStorage storage,
        IMacGenerator* pMacGenerator,
        uint32_t minimumVersion
    ) NN_NOEXCEPT;

    /**
    * @brief        セーブデータのマスターヘッダーを修復します。
    *
    * @param[in]    storage                      セーブデータのストレージ
    * @param[in]    pDuplicateCacheBuffer        キャッシュマネージャ
    * @param[in]    pMacGenerator                MAC 生成クラス
    * @param[in]    minimumVersion               最低バージョン
    *
    * @return       関数の処理結果を返します。
    */
    static Result RecoverMasterHeader(
        fs::SubStorage storage,
        IBufferManager* pDuplicateCacheBuffer,
        IMacGenerator* pMacGenerator,
        uint32_t minimumVersion
    ) NN_NOEXCEPT;

public:
    /**
    * @brief        ファイルを作成します。
    *
    * @param[in]    path    ファイルパス
    * @param[in]    size    ファイルサイズ
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result CreateFile(const Path& path, int64_t size) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        ディレクトリを作成します。
    *
    * @param[in]    path ディレクトリパス
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result CreateDirectory(const Path& path) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        ファイルを開きます。
    *
    * @param[out]   outValue    ファイルオブジェクト
    * @param[in]    path        ファイルパス
    * @param[in]    mode        オープンモード
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result OpenFile(
                       IFile** outValue,
                       const Path& path,
                       int mode
                   ) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        ディレクトリを開きます。
    *
    * @param[out]   outValue    ディレクトリオブジェクト
    * @param[in]    path        ディレクトリパス
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result OpenDirectory(
                       IDirectory** outValue,
                       const Path& path
                   ) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        ファイルを閉じます。
    *
    * @param[in]    pFile   ファイルオブジェクト
    */
    virtual void CloseFile(IFile* pFile) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        ディレクトリを閉じます。
    *
    * @param[in]    pDirectory  ディレクトリオブジェクト
    */
    virtual void CloseDirectory(IDirectory* pDirectory) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        ファイルを削除します。
    *
    * @param[in]    path    ファイルパス
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result DeleteFile(const Path& path) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        ディレクトリを削除します。
    *
    * @param[in]    path    ディレクトリパス
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result DeleteDirectory(const Path& path) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        ディレクトリを再帰的に削除します。
    *
    * @param[in]    path    ディレクトリパス
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result DeleteDirectoryRecursively(const Path& path) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        全ての子エントリを再帰的に削除します。
    *
    * @param[in]    path    ディレクトリパス
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result CleanDirectoryRecursively(const Path& path) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        ファイルをリネームします。
    *
    * @param[in]    oldPath 変更対象のファイルパス
    * @param[in]    newPath 変更後のファイルパス
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result RenameFile(const Path& oldPath, const Path& newPath) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        ディレクトリをリネームします。
    *
    * @param[in]    oldPath 変更対象のディレクトリパス
    * @param[in]    newPath 変更後のディレクトリパス
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result RenameDirectory(const Path& oldPath, const Path& newPath) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        指定したファイルが存在しているかどうかを取得します。
    *
    * @param[out]   outValue    ファイルが存在しているかどうか
    * @param[in]    path        ファイルパス
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result HasFile(bool* outValue, const Path& path) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        指定したディレクトリが存在しているかどうかを取得します。
    *
    * @param[out]   outValue    ディレクトリが存在しているかどうか
    * @param[in]    path        ディレクトリパス
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result HasDirectory(bool* outValue, const Path& path) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        空き領域のサイズをバイト数単位で取得します。
    *
    * @param[out]   outValue    空き領域のサイズ
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result GetFreeBytes(int64_t* outValue) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        実データ領域のサイズをバイト数単位で取得します。
    *
    * @param[out]   outValue    実データ領域のサイズ
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result GetDataAreaBytes(int64_t* outValue) NN_NOEXCEPT NN_OVERRIDE;

   /**
    * @brief        乱数列生成関数をセットします。
    *
    * @return       なし
    */
    static void SetGenerateRandomFunction(
        GenerateRandomFunction function
    ) NN_NOEXCEPT;

    //! セーブデータ内部ストレージアクセス用の Vistor を受け入れます。
    nn::Result AcceptVisitor(IInternalStorageFileSystemVisitor* pVisitor) NN_NOEXCEPT;

private:
    typedef decltype(buffers::MakeUniqueBufferFromBufferManager<MasterHeader>(nullptr, nullptr)) MasterHeaderBufferType;

private:
    /**
    * @brief        マスター管理領域を読み込みます。
    *
    * @param[out]   outHeader                    マスター管理領域の読み込み先
    * @param[out]   outIsProvisionallyCommitted  仮コミット中かどうか
    * @param[out]   outIsFirstControlAreaMounted 最初の管理領域を読み込んだかどうか
    * @param[in]    headerBufferSize             読み込み先バッファのサイズ
    * @param[in]    storage                      マスター管理領域を読み込むストレージ
    * @param[in]    pMacGenerator                MAC 生成クラス
    * @param[in]    minimumVersion               最低バージョン
    *
    * @return       関数の処理結果を返します。
    *
    * @pre
    *               - outHeader != nullptr
    *               - outIsProvisionallyCommitted != nullptr
    *               - outIsFirstControlAreaMounted != nullptr
    */
    static Result ReadMasterControlArea(
                      MasterHeader* outHeader,
                      bool* outIsProvisionallyCommitted,
                      bool* outIsFirstControlAreaMounted,
                      size_t headerBufferSize,
                      fs::SubStorage storage,
                      IMacGenerator* pMacGenerator,
                      uint32_t minimumVersion
                  ) NN_NOEXCEPT;


    /**
    * @brief        マスター管理領域の MAC を更新します。
    *
    * @param[in]    storage                      マスター管理領域のストレージ
    * @param[in]    pMacGenerator                MAC 生成クラス
    * @param[in]    minimumVersion               最低バージョン
    *
    * @return       関数の処理結果を返します。
    */
    static Result UpdateMasterControlAreaMac(
        fs::SubStorage storage,
        IMacGenerator* pMacGenerator,
        uint32_t minimumVersion
    ) NN_NOEXCEPT;

    /**
    * @brief        メタデータを拡張します。
    *
    * @param[out]   outNeedsCommit              コミット処理を行う必要があるかどうかを格納する領域
    * @param[in]    storageMetaRemapMeta        メタデータのリマップストレージのメタストレージ
    * @param[in]    storageRemappedFileSystem   リマップが敷かれたファイルシステムストレージ
    * @param[in]    storageControlArea          マスター管理領域ストレージ
    * @param[in]    layoutHeader                拡張前のレイアウトヘッダ
    * @param[in]    layoutHeaderNew             拡張後のレイアウトヘッダ
    * @param[in]    sizeArchiveBlock            ファイルシステムコアのブロックサイズ
    * @param[in]    pDuplicateCacheBuffer       キャッシュマネージャ
    * @param[in]    pLocker                     ファイルシステムに対する同期オブジェクト
    *
    * @return       関数の処理結果を返します。
    *
    * @pre
    *               - outNeedsCommit != nullptr
    */
    static Result ExpandMeta(
                      bool* outNeedsCommit,
                      fs::SubStorage storageMetaRemapMeta,
                      fs::SubStorage storageRemappedFileSystem,
                      fs::SubStorage storageControlArea,
                      const FileSystemLayoutHeader& layoutHeader,
                      const FileSystemLayoutHeader& layoutHeaderNew,
                      uint32_t sizeArchiveBlock,
                      IBufferManager* pDuplicateCacheBuffer,
                      os::Mutex* pLocker
                  ) NN_NOEXCEPT;

    /**
    * @brief        セーブデータをマウントします。
    *
    * @param[in]    layoutHeader                レイアウトヘッダ
    * @param[in]    storageFileSystem           セーブデータファイルシステムをマウントするストレージ
    * @param[in]    sizeBlock                   ブロックサイズ
    * @param[in]    pIntegrityCacheBufferSet    キャッシュマネージャ
    * @param[in]    pDuplicateCacheBuffer       キャッシュマネージャ
    * @param[in]    pLocker                     ファイルシステムに対する同期オブジェクト
    *
    * @return       関数の処理結果を返します。
    */
    Result MountSaveData(
               const FileSystemLayoutHeader& layoutHeader,
               fs::SubStorage storageFileSystem,
               size_t sizeBlock,
               FilesystemBufferManagerSet* pIntegrityCacheBufferSet,
               IBufferManager* pDuplicateCacheBuffer,
               os::Mutex* pLocker
           ) NN_NOEXCEPT;

    /**
    * @brief        完全性検証+ジャーナリングセーブデータをマウントします。
    *
    * @param[in]    layoutHeader                レイアウトヘッダ
    * @param[in]    pIntegrityCacheBufferSet    キャッシュマネージャ
    * @param[in]    pSaveDataCoreCacheBufferSet キャッシュマネージャ
    * @param[in]    pLocker                     ファイルシステムに対する同期オブジェクト
    *
    * @return       関数の処理結果を返します。
    */
    Result MountIntegritySaveData(
               const FileSystemLayoutHeader& layoutHeader,
               FilesystemBufferManagerSet* pIntegrityCacheBufferSet,
               IBufferManager* pSaveDataCoreCacheBuffer,
               os::Mutex* pLocker
           ) NN_NOEXCEPT;

    /**
    * @brief        セーブデータを再マウントします。
    *
    * @return       関数の処理結果を返します。
    */
    Result RemountSaveData() NN_NOEXCEPT;

    /**
    * @brief        ファイルシステムをコミットします。
    *
    * @param[in]    counterForBundledCommit 仮コミット時に書き込むカウンタ値
    *
    * @return       関数の処理結果を返します。
    */
    Result CommitFileSystemCore(int64_t counterForBundledCommit) NN_NOEXCEPT;

    /**
    * @brief        署名を付けます。
    *
    * @param[in]    pStorage      計算したハッシュを格納するストレージ
    * @param[in]    pMacGenerator MAC 生成クラス
    * @param[in]    pHash         計算したハッシュを格納するバッファ
    * @param[in]    hashSize      pHash のサイズ
    * @param[in]    pData         計算対象データ
    * @param[in]    dataSize      pData のサイズ
    *
    * @return       関数の処理結果を返します。
    */
    static Result Sign(
                      fs::IStorage* pStorage,
                      IMacGenerator* pMacGenerator,
                      void* pHash,
                      size_t hashSize,
                      const void* pData,
                      size_t dataSize
                  ) NN_NOEXCEPT;

    /**
    * @brief        データの正当性を検証します。
    *
    * @param[in]    pMacGenerator MAC 生成クラス
    * @param[in]    pData         計算対象データ
    * @param[in]    dataSize      pData のサイズ
    * @param[in]    pSignature    比較に使用する署名
    * @param[in]    signatureSize pSignature のサイズ
    *
    * @return       関数の処理結果を返します。
    */
    static Result Verify(
                      IMacGenerator* pMacGenerator,
                      const void* pData,
                      size_t dataSize,
                      const void* pSignature,
                      size_t signatureSize
                  ) NN_NOEXCEPT;

    /**
    * @brief        マスターヘッダの正当性を検証します。
    *
    * @param[in]    pMasterHeader   マスターヘッダ
    *
    * @return       正当か否かを返します
    */
    static bool VerifyMasterHeaderContent(
                    MasterHeader* pMasterHeader
                ) NN_NOEXCEPT;

    /**
    * @brief        指定のパラメータのセーブデータにおける各データの配置を計算します。
    *
    * @param[out]   outFileSystemLayoutHeader           レイアウトヘッダ
    * @param[in]    version                             レイアウトデータのバージョン
    * @param[in]    sizeRemappedFileSystemEntryTable    ファイルシステム用論物変換エントリテーブルのサイズ
    * @param[in]    sizeRemappedMetaEntryTable          メタデータ用論物変換エントリテーブルのサイズ
    * @param[in]    inputParamDuplex                    二重化パラメータ
    * @param[in]    inputParamIntegrity                 完全性検証パラメータ
    * @param[in]    sizeBlock                           ブロックサイズ
    * @param[in]    countDataBlock                      データブロック数
    * @param[in]    countReservedBlock                  予約領域のブロック数
    *
    * @return       関数の処理結果を返します。
    */
    static Result FormatFileSystemLayoutHeader(
                      FileSystemLayoutHeader* outFileSystemLayoutHeader,
                      uint32_t version,
                      int64_t sizeRemappedFileSystemEntryTable,
                      int64_t sizeRemappedMetaEntryTable,
                      const HierarchicalDuplexStorageControlArea::InputParam& inputParamDuplex,
                      const HierarchicalIntegrityVerificationStorageControlArea::InputParam& inputParamIntegrity,
                      uint32_t sizeBlock,
                      uint32_t countDataBlock,
                      uint32_t countReservedBlock
                  ) NN_NOEXCEPT;

    Result GetMasterHeader(
               MasterHeaderBufferType* outAtomic,
               uint32_t minimumVersion
           ) NN_NOEXCEPT;

    void StoreMasterHeader(MasterHeader* pAtomic, size_t size) NN_NOEXCEPT;

    os::Mutex* GetLocker() NN_NOEXCEPT
    {
        return m_pCommitLocker;
    }

private:
    ControlAreaHolder m_ControlArea;             //!< 管理領域
    fs::SubStorage m_BaseStorageControlArea;     //!< 管理領域ストレージ
    BufferedStorage m_StorageBufferingRemap;     //!< リマップストレージ用バッファ
    BufferedStorage m_StorageBufferingMeta;      //!< メタ情報ストレージ用バッファ
    RemapStorage m_RemappedFileSystemStorage;    //!< ファイルシステム用リマップストレージ
    RemapStorage m_RemappedMetaStorage;          //!< メタデータ用リマップストレージ
    HierarchicalDuplexStorage m_DuplexStorage;   //!< 二重化メタデータストレージ
    JournalIntegritySaveDataStorage m_IntegrityStorage; //!< 完全性検証+ジャーナリングファイル
    HierarchicalIntegrityVerificationStorage m_IntegrityStorageSaveDataMeta; //!< 完全性検証ファイル
    SaveDataFileSystemCore m_FileSystem;         //!< 完全性検証+ジャーナリング対象のファイルシステム
    IBufferManager* m_pBufferManager;            //!< バッファマネージャ
    IBufferManager::CacheHandle m_MasterHandle;  //!< マスターヘッダを格納したメモリのハンドル
    os::Mutex* m_pCommitLocker;                  //!< ファイルシステムに対する同期オブジェクト
    bool m_IsInitialized;                        //!< 初期化されているかどうか
    bool m_IsFirstControlAreaMounted;            //!< 最初の管理領域を使ってマウントされているかどうか
    bool m_IsProvisionallyCommitted;             //!< 仮コミットされているかどうか
    bool m_IsExtraDataModified;                  //!< 拡張データが更新されたか
    IMacGenerator* m_pMacGenerator;              //!< MAC 生成クラス
    uint32_t m_SaveDataMinimumVersion;           //!< 最低バージョン

private:
    friend class IntegrityFilteredFile;
    friend class IntegrityFilteredDirectory;

private:
    // テスト用クラスには実装を公開します。
    friend class FsJournalIntegritySaveDataFileSystemTest;
};

/**
* @brief コミット対応ファイルオブジェクトです。
*/
class IntegrityFilteredFile : public FileInFile, public nn::fs::detail::Newable
{
    NN_DISALLOW_COPY(IntegrityFilteredFile);

private:
    /**
    * @brief        コンストラクタ
    */
    explicit IntegrityFilteredFile(
                 IFile* pFile,
                 JournalIntegritySaveDataFileSystem* pFileSystem
             ) NN_NOEXCEPT;

public:
    /**
    * @brief        デストラクタ
    */
    virtual ~IntegrityFilteredFile() NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        下位ファイルオブジェクトへの参照を解除します。
    */
    void Finalize() NN_NOEXCEPT;

    /**
    * @brief        ファイルの内容をバッファに読み込みます。
    *
    * @param[in]    offset  読み込み開始位置
    * @param[out]   buffer  読み込んだ内容をコピーするバッファ
    * @param[in]    size    読み込むデータサイズ
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result ReadBytes(
                       int64_t offset,
                       void* buffer,
                       size_t size
                   ) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        バッファの内容をファイルに書き込みます。
    *
    * @param[in]    offset  書き込み開始位置
    * @param[in]    buffer  書き込むデータ
    * @param[in]    size    書き込むデータサイズ
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result WriteBytes(
                       int64_t offset,
                       const void* buffer,
                       size_t size
                   ) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        ファイルサイズを変更します。
    *
    * @param[in]    size    変更後のファイルサイズ
    *
    * @return       失敗の結果を返します。
    */
    virtual Result SetSize(int64_t size) NN_NOEXCEPT NN_OVERRIDE;

private:
    JournalIntegritySaveDataFileSystem* m_pFileSystem;

private:
    friend class JournalIntegritySaveDataFileSystem;
};

/**
* @brief コミット対応ディレクトリオブジェクトです。
*/
class IntegrityFilteredDirectory : public IDirectory, nn::fs::detail::Newable
{
    NN_DISALLOW_COPY(IntegrityFilteredDirectory);

private:
    /**
    * @brief        コンストラクタ
    */
    explicit IntegrityFilteredDirectory(
                 IDirectory* pDirectory,
                 JournalIntegritySaveDataFileSystem* pFileSystem
             ) NN_NOEXCEPT;

public:
    /**
    * @brief        デストラクタ
    */
    virtual ~IntegrityFilteredDirectory() NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        下位ディレクトリオブジェクトへの参照を解除します。
    */
    void Finalize() NN_NOEXCEPT;

    /**
    * @brief        下位ディレクトリオブジェクトを取得します。
    */
    IDirectory* GetDirectory() NN_NOEXCEPT
    {
        return m_pDirectory;
    }

    /**
    * @brief        ディレクトリ内の子ディレクトリエントリーを取得します。
    *
    * @param[out]   outNumEntries   取得したエントリー数
    * @param[out]   outEntries      取得したエントリー情報
    *                               配列のサイズは numEntries 以上である必要があります。
    * @param[in]    numEntries      取得するエントリー数
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result Read(
                       int32_t* outNumEntries,
                       nn::fs::DirectoryEntry outEntries[],
                       int32_t numEntries
                   ) NN_NOEXCEPT NN_OVERRIDE;

private:
    JournalIntegritySaveDataFileSystem* m_pFileSystem;
    IDirectory* m_pDirectory;

private:
    friend class JournalIntegritySaveDataFileSystem;
};

}}}

