﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cstring>

#include <nn/nn_Common.h>

#include <nn/crypto/crypto_Sha256Generator.h>
#include <nn/fs/fs_SaveDataTypes.h>
#include <nn/fs/fs_StorageType.h>
#include <nn/fs/fs_SubStorage.h>
#include <nn/fssystem/save/fs_ISaveFileSystemDriver.h>
#include <nn/fssystem/save/fs_BlockCacheBufferedStorage.h>

class FsIntegrityVerificationStorageTest;

namespace nn { namespace fssystem { namespace save {

/*!
    @brief 完全性検証ファイルレイヤーです。

           完全性検証ファイルシステムでは、このレイヤーを階層化して使用します。
*/
class IntegrityVerificationStorage : public fs::IStorage
{
    NN_DISALLOW_COPY(IntegrityVerificationStorage);

public:
    //!< ハッシュサイズ
    static const int64_t HashSize = nn::crypto::Sha256Generator::HashSize;

    //!< 検証データ
    struct BlockHash
    {
        uint8_t hash[HashSize];
    };
    NN_STATIC_ASSERT(std::is_pod<BlockHash>::value);

public:
    /**
    * @brief        コンストラクタ
    */
    IntegrityVerificationStorage() NN_NOEXCEPT;

    /**
    * @brief        デストラクタ
    */
    virtual ~IntegrityVerificationStorage() NN_NOEXCEPT;

    /**
    * @brief        完全性検証ストレージレイヤーをマウントします。
    *
    * @param[in]    storageHash                     ハッシュデータストレージ
    * @param[in]    storageData                     ハッシュ検証対象ストレージ
    * @param[in]    sizeVerificationBlock           ハッシュブロックサイズ
    * @param[in]    sizeUpperLayerVerificationBlock 上位レイヤーのハッシュブロックサイズ
    * @param[in]    pBuf                            検証済みハッシュを一時的においておくためのバッファ
    * @param[in]    salt                            ハッシュ計算に使用するシソルト
    * @param[in]    isRealData                      実データに対するハッシュかどうか
    * @param[in]    storageType                     使用するストレージのタイプ
    *
    * @return       関数の処理結果を返します。
    */
    void Initialize(
                fs::SubStorage storageHash,
                fs::SubStorage storageData,
                int64_t sizeVerificationBlock,
                int64_t sizeUpperLayerVerificationBlock,
                IBufferManager* pBuf,
                const fs::HashSalt& salt,
                bool isRealData,
                fs::StorageType storageType
           ) NN_NOEXCEPT;

    /**
    * @brief        完全性検証ストレージレイヤーをアンマウントします。
    */
    void Finalize() NN_NOEXCEPT;

    /**
    * @brief        ハッシュ検証対象ストレージのサイズを取得します。
    *
    * @param[out]   outValue    ハッシュ検証対象ストレージのサイズ
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result GetSize(int64_t* outValue) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        ストレージサイズは変更できません。初期化時にのみ指定できます。
    *
    * @return       常に nn::fs::ResultUnsupportedOperation を返します。
    */
    virtual Result SetSize(int64_t) NN_NOEXCEPT NN_OVERRIDE
    {
        return nn::fs::ResultUnsupportedOperation();
    }

    /**
    * @brief        完全性検証ストレージレイヤーからデータを読み込みます。
    *
    * @param[in]    offset  読み込み開始位置
    * @param[out]   buffer  読み込んだ内容をコピーするバッファ
    * @param[in]    size    読み込むデータサイズ
    *
    * @return       関数の処理結果を返します。
    *
    * @details      読み込もうとしたデータに対するハッシュが不正だった場合には、0 フィルされたデータが
    *               読込先バッファに書き込まれ、nn::fs::ResultFailedVerifySign を返します。
    */
    virtual Result Read(
                       int64_t offset,
                       void* buffer,
                       size_t size
                   ) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        完全性検証ストレージレイヤーにデータを書き込みます。
    *
    * @param[in]    offset  書き込み開始位置
    * @param[in]    buffer  書き込むデータ
    * @param[in]    size    書き込むデータサイズ
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result Write(
                       int64_t offset,
                       const void* buffer,
                       size_t size
                   ) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        フラッシュします。
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result Flush() NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        範囲を指定して操作を実行します。
    *
    * @param[out]   outBuffer       範囲指定処理の結果を格納するバッファ
    * @param[in]    outBufferSize   範囲指定処理の結果を格納するバッファのサイズ
    * @param[in]    operationId     実行する操作
    * @param[in]    offset          操作を実行する範囲の先頭オフセット
    * @param[in]    size            操作を実行する範囲のサイズ
    * @param[in]    inBuffer        範囲指定処理に渡すバッファ
    * @param[in]    inBufferSize    範囲指定処理に渡すバッファのサイズ
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result OperateRange(
                       void* outBuffer,
                       size_t outBufferSize,
                       fs::OperationId operationId,
                       int64_t offset,
                       int64_t size,
                       const void* inBuffer,
                       size_t inBufferSize
                   ) NN_NOEXCEPT NN_OVERRIDE;

    using IStorage::OperateRange;

    /**
    * @brief        バッファに対するハッシュを計算します。
    *
    * @param[out]   outHash     ハッシュブロック
    * @param[in]    buffer      ハッシュを計算するデータ
    * @param[in]    sizeBlock   ハッシュを計算するブロックサイズ
    */
    void CalcBlockHash(
             BlockHash* outHash,
             const void* buffer,
             size_t sizeBlock
         ) const NN_NOEXCEPT;

    /**
    * @brief        ブロックサイズを取得します。
    *
    * @return       ブロックサイズを返します。
    */
    int64_t GetBlockSize() const NN_NOEXCEPT
    {
        return m_SizeVerificationBlock;
    }

private:
    /**
    * @brief        指定したオフセットでのハッシュブロックを読み込みます。
    *
    * @param[out]   outBuf  ハッシュブロック
    * @param[in]    bufSize ハッシュサイズ
    * @param[in]    offset  ハッシュ検証対象データのオフセット
    * @param[in]    size    ハッシュ検証対象データサイズ
    *
    * @return       関数の処理結果を返します。
    * @retval       nn::fs::ResultFailedVerifySign  ハッシュの読み込みに失敗した場合に返します。
    */
    Result ReadBlockSignature(
               void* outBuf,
               size_t bufSize,
               int64_t offset,
               size_t size
           ) NN_NOEXCEPT;

    /**
    * @brief        指定したオフセットでのハッシュブロックを更新します。
    *
    * @param[in]    buf     ハッシュブロック
    * @param[in]    bufSize ハッシュサイズ
    * @param[in]    offset  ハッシュ検証対象データのオフセット
    * @param[in]    size    ハッシュ検証対象データサイズ
    *
    * @return       関数の処理結果を返します。
    */
    Result WriteBlockSignature(
               const void* buf,
               size_t bufSize,
               int64_t offset,
               size_t size
           ) NN_NOEXCEPT;

    /**
    * @brief        指定したオフセットでのハッシュとデータから計算したハッシュが等しいかどうか検証します。
    *
    * @param[in]    buf             ハッシュを計算するデータ
    * @param[in]    pHash           ハッシュ
    *
    * @return       関数の処理結果を返します。
    * @retval       nn::fs::ResultIntegrityVerificationStorageCorrupted 検証に失敗した場合に返します。
    */
    Result VerifyHash(
               const void* buf,
               BlockHash* pHash
           ) NN_NOEXCEPT;

    /**
    * @brief        バッファに対するハッシュを計算します。
    *
    * @param[out]   outHash ハッシュブロック
    * @param[in]    buffer  ハッシュを計算するデータ
    */
    void CalcBlockHash(BlockHash* outHash, const void* buffer) const NN_NOEXCEPT
    {
        CalcBlockHash(outHash, buffer, static_cast<size_t>(m_SizeVerificationBlock));
    }

    /**
    * @brief        実データに対するハッシュが有効であることを示すビットを立てます。
    *
    * @param[in]    pHash   有効ビットを立てるハッシュを指定します。
    */
    static void SetValidationBit(BlockHash* pHash) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pHash);
        pHash->hash[31] |= 0x80;
    }

    /**
    * @brief        実データに対するハッシュが有効かどうかを取得します。
    *
    * @return       実データに対するハッシュが有効であるかどうか
    * @retval       true    ハッシュは有効である
    * @retval       false   ハッシュは無効である
    */
    static bool IsValidationBit(const BlockHash* pHash) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pHash);
        return ((pHash->hash[31] & 0x80) != 0);
    }

    /**
    * @brief ハッシュがゼロクリアされているかどうか判定します。
    *
    * @param[out] isCleared ゼロクリアされていれば true、有効ビットが立っていれば false が格納されます。
    * @param[in]  hash 判定するハッシュを指定します。
    *
    * @return 関数の処理結果を返します。
    * @retval ResultSuccess ハッシュはゼロクリアされているか、または有効ビットが立っています。
    * @retval ResultInvalidZeroHash ハッシュは有効ビットが立っていないにもかかわらずゼロクリアされていません。
    */
    Result IsCleared(
               bool *isCleared,
               const BlockHash& hash
           ) NN_NOEXCEPT;


private:
    fs::SubStorage m_StorageHash;
    fs::SubStorage m_StorageData;
    int64_t m_SizeVerificationBlock;               //!< ハッシュブロックサイズ
    int64_t m_OrderVerificationBlock;              //!< ハッシュブロック数を計算するためのシフト値
    int64_t m_SizeUpperLayerVerificationBlock;     //!< 上位のハッシュブロックサイズ
    int64_t m_OrderUpperLayerVerificationBlock;    //!< 上位のハッシュブロック数を計算するためのシフト値
    IBufferManager* m_pBuffer;                     //!< バッファマネージャ
    fs::HashSalt m_Salt;                           //!< ハッシュのソルト
    bool m_IsRealDataSign;                         //!< 実データに対する完全性検証ストレージかどうか
    fs::StorageType m_StorageType;                 //!< 使用するストレージのタイプ

private:
    // テスト用クラスには内部データを公開します。
    friend class ::FsIntegrityVerificationStorageTest;
};

}}}

