﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cstring>

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_StaticAssert.h>
#include <nn/fs/fs_Types.h>
#include <nn/fssystem/dbm/fs_DuplexBitmap.h>
#include <nn/fssystem/save/fs_DuplexStorage.h>
#include <nn/fssystem/save/fs_DuplexBitmapHolder.h>
#include <nn/fssystem/save/fs_BufferedStorage.h>
#include <nn/fssystem/save/fs_BlockCacheBufferedStorage.h>
#include <nn/fssystem/save/fs_TruncatedSubStorage.h>

class FsHierarchicalDuplexStorageTest;

namespace nn { namespace fssystem { namespace save {

//! 二重化選択ビットマップ階層の情報
struct HierarchicalDuplexLevelInformation
{
    nn::fs::Int64 offset;           //!< その層のビットマップがおいてあるオフセット
    nn::fs::Int64 size;             //!< その層のビットマップサイズ
    int32_t orderBlock;             //!< その層のハッシュ計算単位
};
NN_STATIC_ASSERT(sizeof(HierarchicalDuplexLevelInformation) == 20);
NN_STATIC_ASSERT(std::is_pod<HierarchicalDuplexLevelInformation>::value);

//! 二重化選択ビットマップストレージに与える情報
struct HierarchicalDuplexInformation
{
    HierarchicalDuplexLevelInformation info[3];
};
NN_STATIC_ASSERT(std::is_pod<HierarchicalDuplexInformation>::value);

//! メタデータ領域の形式
struct HierarchicalDuplexMetaInformation
{
    //! マジックコード
    uint32_t magic;

    //! バージョン
    uint32_t version;

    //! 階層ハッシュの情報
    HierarchicalDuplexInformation infoDuplex;
};
NN_STATIC_ASSERT(std::is_pod<HierarchicalDuplexMetaInformation>::value);

//! データサイズ
struct HierarchicalDuplexSizeSet
{
    int64_t layeredBitmapSizes[2]; //!< L1, L2 ビットマップのサイズ
    int64_t bodySize;              //!< データのサイズ
};
NN_STATIC_ASSERT(std::is_pod<HierarchicalDuplexSizeSet>::value);

//! メタデータを管理するクラスです。
class HierarchicalDuplexStorageControlArea
{
    NN_DISALLOW_COPY(HierarchicalDuplexStorageControlArea);

public:
    struct InputParam
    {
        size_t sizeBlockLevel[2];      //! ビットマップ用ブロックサイズ
    };
    NN_STATIC_ASSERT(std::is_pod<InputParam>::value);

public:
    /**
    * @brief        コンストラクタ
    */
    HierarchicalDuplexStorageControlArea() NN_NOEXCEPT
    {
    }

    /**
    * @brief        二重化ストレージの各層のデータサイズを取得します。
    *
    * @param[out]   outSizeSet      各層のデータサイズの計算結果
    * @param[in]    inputParam      入力パラメータ
    * @param[in]    sizeData        実データサイズ (二重化元データのサイズ)
    *
    * @return       関数の処理結果を返します。
    *
    * @details      ストレージシステムを構築するために必要なそれぞれのサイズを求めます
    */
    static Result QuerySize(
                      HierarchicalDuplexSizeSet* outSizeSet,
                      const InputParam& inputParam,
                      const int64_t sizeData
                  ) NN_NOEXCEPT;

    /**
    * @brief        二重化ストレージ用メタデータ領域をフォーマットします。
    *
    * @param[out]   storageMeta メタデータストレージ
    * @param[in]    inputParam  入力パラメータ
    *
    * @return       関数の処理結果を返します。
    *
    * @details      この関数では以下の処理を行います。
    *
    *              ・管理領域の生成。
    *              ・ビットマップ領域、実データ領域の、それぞれのオフセット、サイズを計算し管理領域に覚えます。
    *                オフセット情報はフォーマット後は基本的に読み取り専用のデータです。
    */
    static Result Format(
                      fs::SubStorage storageMeta,
                      const InputParam& inputParam,
                      int64_t sizeData
                  ) NN_NOEXCEPT;

    /**
    * @brief        二重化ストレージ用メタデータ領域に拡張したデータの情報を保存します。
    *
    * @param[out]   storageMeta メタデータストレージ
    * @param[in]    sizeData    拡張後の実データサイズ (二重化元データのサイズ)
    *
    * @return       関数の処理結果を返します。
    *
    * @details      メタデータのバージョン、ビットマップサイズ、実データサイズのみを更新します。
    */
    static Result Expand(fs::SubStorage storageMeta, int64_t sizeData) NN_NOEXCEPT;

    /**
    * @brief        二重化ストレージの管理領域をマウントします。
    *
    * @param[in]    storageMeta メタデータストレージ
    *
    * @return       関数の処理結果を返します。
    */
    Result Initialize(fs::SubStorage storageMeta) NN_NOEXCEPT;

    /**
    * @brief        管理領域をアンマウントします。
    */
    void Finalize() NN_NOEXCEPT;

    /**
    * @brief        メタデータを取得します。
    */
    const HierarchicalDuplexMetaInformation& GetMeta() const NN_NOEXCEPT
    {
        return m_Meta;
    }

private:
    /**
    * @brief        二重化ストレージ用メタデータ領域を作成します。
    *
    * @param[out]   pMeta           メタデータ
    * @param[in]    inputparam      ブロックサイズ
    * @param[in]    sizeData        実データサイズ
    *
    * @return       関数の処理結果を返します。
    */
    static Result CreateMetaInformation(
                      HierarchicalDuplexMetaInformation* pMeta,
                      const InputParam& inputParam,
                      int64_t sizeData
                  ) NN_NOEXCEPT;

private:
    fs::SubStorage m_Storage;
    HierarchicalDuplexMetaInformation m_Meta; //! メタデータ

private:
    friend class ::FsHierarchicalDuplexStorageTest;
};

/**
* @brief 二重化ストレージです。
*
*        データ領域を二重化し、ストレージの破損対策をします。
*        ストレージサイズ 4G バイト、ブロックサイズ 4K バイト、
*        としたときのサイズの概算は以下のようになります。
*
*        SIZE=4KB    二重化対象データのサイズ  選択ビットマップのサイズ
*        マスター  <=     2BYTE(x 2)
*        階層1     <=     64KB(x 2)            2BYTE
*        階層2     <=     2GB(x 2)             64KB
*
*        ストレージサイズ 4G バイト、ブロックサイズ 256B バイト
*        としたときのサイズの概算は以下のようになります。
*
*        SIZE=256B   二重化対象データのサイズ  選択ビットマップのサイズ
*        マスター  <=     512BYTE(x 2)
*        階層1     <=     1MB(x 2)                 512BYTE
*        階層2     <=     2GB(x 2)                 1MB
*/
class HierarchicalDuplexStorage : public fs::IStorage
{
    NN_DISALLOW_COPY(HierarchicalDuplexStorage);

public:
    /**
    * @brief        二重化ビットマップ用領域をフォーマットします。
    *
    * @param[in]    infoDuplex              二重化ストレージ情報
    * @param[in]    storageMasterBitmapA    二重化ビットマップストレージ
    * @param[in]    storageMasterBitmapB    二重化ビットマップストレージ
    * @param[in]    storageBitmapL1A        二重化ビットマップストレージ
    * @param[in]    storageBitmapL1B        二重化ビットマップストレージ
    * @param[in]    pBuffer   二重化ビットマップストレージ
    *
    * @return       関数の処理結果を返します。
    */
    static Result Format(
                      const HierarchicalDuplexInformation& infoDuplex,
                      fs::SubStorage storageMasterBitmapA,
                      fs::SubStorage storageMasterBitmapB,
                      fs::SubStorage storageBitmapL1A,
                      fs::SubStorage storageBitmapL1B,
                      IBufferManager* pBuffer
                  ) NN_NOEXCEPT;

    /**
    * @brief        二重化ビットマップ用領域を拡張します。
    *
    * @param[in]    infoDuplexOld           拡張前の二重化ストレージ情報
    * @param[in]    infoDuplexNew           拡張後の二重化ストレージ情報
    * @param[in]    storageMasterBitmapA    二重化ビットマップストレージ (マスター A)
    * @param[in]    storageMasterBitmapB    二重化ビットマップストレージ (マスター B)
    * @param[in]    storageBitmapL1A        二重化ビットマップストレージ (L1A)
    * @param[in]    storageBitmapL1B        二重化ビットマップストレージ (L1B)
    * @param[in]    pBuffer                 バッファマネージャ
    *
    * @return       関数の処理結果を返します。
    */
    static Result Expand(
                      const HierarchicalDuplexInformation& infoDuplexOld,
                      const HierarchicalDuplexInformation& infoDuplexNew,
                      fs::SubStorage storageMasterBitmapA,
                      fs::SubStorage storageMasterBitmapB,
                      fs::SubStorage storageBitmapL1A,
                      fs::SubStorage storageBitmapL1B,
                      IBufferManager* pBuffer
                  ) NN_NOEXCEPT;

    /**
    * @brief        コンストラクタ
    */
    HierarchicalDuplexStorage() NN_NOEXCEPT;

    /**
    * @brief        デストラクタ
    */
    virtual ~HierarchicalDuplexStorage() NN_NOEXCEPT;

    /**
    * @brief        二重化ストレージをマウントします。
    *
    * @param[in]    info            二重ストレージ情報
    * @param[in]    storageBitmap   二重化ビットマップストレージ
    * @param[in]    bSelectBitmap   二重化ブロックのAブロック、Bブロックのどちらを書き換えるか?
    * @param[in]    pBuffer         キャッシュ
    * @param[in]    pLocker         排他制御用のロックオブジェクト
    *
    * @return       関数の処理結果を返します。
    *
    *               "bSelectBitmap" はストレージをコミットする(Close)度に、
    *               "0->1->0..." とトグルしながら呼ばれることを期待しています。
    *               同じ "bSelectBitmap" 値をトグルせず呼ぶことで、前回書き換えた
    *               状態を巻き戻してオープンすることが出来ます。
    */
    Result Initialize(
               const HierarchicalDuplexInformation& info,
               fs::SubStorage storageMasterBitmapA,
               fs::SubStorage storageMasterBitmapB,
               fs::SubStorage storageBitmapL1A,
               fs::SubStorage storageBitmapL1B,
               fs::SubStorage storageDataA,
               fs::SubStorage storageDataB,
               bool bSelectBitmap,
               IBufferManager* pBuffer,
               os::Mutex* pLocker
           ) NN_NOEXCEPT;

    /**
    * @brief        二重化ストレージをアンマウントします。
    */
    void Finalize() NN_NOEXCEPT;

    /**
    * @brief        実データの総バイトサイズを取得します。
    *
    * @param[out]   outValue    実データの総バイトサイズ
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result GetSize(int64_t* outValue) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        サイズは変更できません。初期化時のみ指定できます。
    *
    * @return       常に nn::fs::ResultUnsupportedOperation
    */
    virtual Result SetSize(int64_t) NN_NOEXCEPT NN_OVERRIDE
    {
        return nn::fs::ResultUnsupportedOperation();
    }

    /**
    * @brief        ストレージレイヤーからデータを読み込みます。
    *
    * @param[in]    offset  読み込み開始位置
    * @param[out]   buffer  読み込んだ内容をコピーするバッファ
    * @param[in]    size    読み込むデータサイズ
    *
    * @return       関数の処理結果を返します。
    *
    * @details      読み込みはブロック単位でなくて構いません。
    */
    virtual Result Read(
                       int64_t offset,
                       void* buffer,
                       size_t size
                   ) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        実データ領域にデータを書き込み、二重化選択ビットを更新します。
    *
    * @param[in]    offset  書き込み開始位置
    * @param[in]    buffer  書き込むデータ
    * @param[in]    size    書き込むデータサイズ
    *
    * @return       関数の処理結果を返します。
    *
    * @details      最上位の二重化選択ビットが書き換えられる前に失敗が発生した場合には
    *               次回使用時にロールバックが行われます。
    */
    virtual Result Write(
                       int64_t offset,
                       const void* buffer,
                       size_t size
                   ) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        フラッシュします。
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result Flush() NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief       範囲指定処理を行います。
    *
    * @param[out]  outBuffer        範囲指定処理の結果を格納するバッファ
    * @param[in]   outBufferSize    範囲指定処理の結果を格納するバッファのサイズ
    * @param[in]   operationId      範囲指定処理の種類
    * @param[in]   offset           範囲指定処理開始位置
    * @param[in]   size             範囲指定処理を行うデータサイズ
    * @param[in]   inBuffer         範囲指定処理に渡すバッファ
    * @param[in]   inBufferSize     範囲指定処理に渡すバッファのサイズ
    *
    * @return      関数の処理結果を返します。
    */
    virtual Result OperateRange(
                       void* outBuffer,
                       size_t outBufferSize,
                       fs::OperationId operationId,
                       int64_t offset,
                       int64_t size,
                       const void* inBuffer,
                       size_t inBufferSize
                   ) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        書き込みが発生したかを取得します。
    *
    * @return       書き込みが発生していたらtrue
    *
    * @details      書き込みが発生したら、閉じる際に選択ビットマップを反転してください。
    *               逆に書き込みが発生していない場合は、閉じる際に選択ビットマップを
    *               反転しないでください。
    *
    *              この関数のみ、Close後に呼ぶことができます。
    */
    bool NeedCommit() const NN_NOEXCEPT
    {
        return m_IsWritten;
    }

    /**
    * @brief        書き込みが発生したかをクリアします。
    */
    void ClearCommitFlag() NN_NOEXCEPT
    {
        m_IsWritten = false;
    }

    /**
    * @brief        コミット後、DuplexBitmapを反転します。
    *
    * @return       関数の処理結果を返します。
    */
    Result SwapDuplexBitmap() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(!m_IsWritten);

        // ビットマップの 0<->1 を交換します。
        m_DuplexBitmapStorage[0].SwapDuplexBitmapForHierarchicalDuplexStorage(
                                     &m_DuplexBitmapStorage[1]
                                 );

        NN_RESULT_SUCCESS;
    }

    /**
    * @brief        DuplexBitmap を反転せずに再マウントします。
    */
    void Remount() NN_NOEXCEPT;

    /**
    * @brief        パラメータを取得します。
    *
    * @param[out]   outParam    パラメータの格納先
    */
    void GetParametners(
        HierarchicalDuplexStorageControlArea::InputParam* outParam) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(outParam);
        outParam->sizeBlockLevel[0] = static_cast<size_t>(m_DuplexStorage[0].GetBlockSize());
        outParam->sizeBlockLevel[1] = static_cast<size_t>(m_DuplexStorage[2].GetBlockSize());
    }

private:
    /**
    * @brief        ストレージの内容をコピーします。
    *
    * @param[in]    dst        コピー先のストレージ
    * @param[in]    src        コピー元のストレージ
    * @param[in]    size       コピーするサイズ
    *
    * @return       関数の処理結果を返します。
    */
    static Result CopyStorage(
                      fs::SubStorage dst,
                      fs::SubStorage src,
                      int64_t size
                  ) NN_NOEXCEPT;

private:
    static const int MaxCacheEntries = 4;

    static const auto DuplexStorageCount = 3;
    static const auto DuplexBitmapCount = 4;
    static const auto BaseStorageCount = 4;
    static const auto BufferedBitmapCount = 6;

private:
    DuplexStorage m_DuplexStorage[DuplexStorageCount];                //!< 二重化ストレージの階層
    DuplexBitmapHolder m_DuplexBitmapStorage[DuplexBitmapCount];      //!< 二重化ビットマップの階層
    fs::SubStorage m_DuplexBaseStorage[BaseStorageCount];             //!< 二重化ビットマップストレージ
    TruncatedSubStorage m_BufferedBitmapStorage[BufferedBitmapCount]; //!< 二重化ビットマップ用キャッシュ
    BlockCacheBufferedStorage m_BufferedStorage[BufferedBitmapCount]; //!< 二重化ビットマップ用キャッシュ
    int64_t m_SizeData;                                               //!< 実データサイズ

    IBufferManager* m_pBuffer; //!< バッファマネージャ
    os::Mutex* m_pLocker;      //!< ロックオブジェクト

    bool m_IsWritten; //!< 書き込みが1度でもされた
    char reserved[3];

private:
    // テスト用クラスには内部データを公開します。
    friend class ::FsDuplexBitmapHolderTest;
};

}}}

