﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cstring>

#include <nn/nn_Common.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/fssystem/dbm/fs_DuplexBitmap.h>
#include <nn/fssystem/save/fs_BlockCacheBufferedStorage.h>

class FsDuplexStorageTest;

namespace nn { namespace fssystem { namespace save {

/**
* @brief    二重化ストレージレイヤーです。
*           管理する 2 つの IStorage のどちらが有効かをビットで管理します。
*/
class DuplexStorage : public fs::IStorage
{
    NN_DISALLOW_COPY(DuplexStorage);

public:
    /**
    * @brief        コンストラクタ
    */
    DuplexStorage() NN_NOEXCEPT;

    /**
    * @brief        デストラクタ
    */
    virtual ~DuplexStorage() NN_NOEXCEPT;

    /**
    * @brief        二重化選択ストレージレイヤーを初期化します。
    *
    * @param[in]    pBitmap         二重化選択ビットマップデータ
    * @param[in]    storageData1    実データストレージ 1
    * @param[in]    storageData2    実データストレージ 2
    * @param[in]    sizeBlock       ブロックサイズ
    * @param[in]    pBuffer         バッファマネージャ
    *
    * @return       関数の処理結果を返します。
    */
    void Initialize(
             dbm::DuplexBitmap* pBitmap,
             fs::SubStorage storageData1,
             fs::SubStorage storageData2,
             int64_t sizeBlock,
             IBufferManager* pBuffer
         ) NN_NOEXCEPT;

    /**
    * @brief        二重化ストレージレイヤーを閉じます。
    */
    void Finalize() NN_NOEXCEPT;

    /**
    * @brief        実データ領域からデータを読み込みます。
    *
    * @param[in]    offset  読み込み開始位置
    * @param[out]   buffer  読み込んだ内容をコピーするバッファ
    * @param[in]    size    読み込むデータサイズ
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result Read(
                       int64_t offset,
                       void* buffer,
                       size_t size
                   ) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        実データ領域にデータを書き込みます。
    *
    * @param[in]    offset  書き込み開始位置
    * @param[in]    buffer  書き込むデータ
    * @param[in]    size    書き込むデータサイズ
    *
    * @return       関数の処理結果を返します。
    *
    * @details      書き込みはブロック単位でのアクセスを強制します。
    *               (ブロックキャッシュレイヤーを下に敷きます)
    */
    virtual Result Write(
                       int64_t offset,
                       const void* buffer,
                       size_t size
                   ) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        総バイトサイズを取得します。
    *
    * @param[out]   outValue    総バイトサイズ
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result GetSize(int64_t* outValue) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        ストレージサイズは変更できません。初期化時にのみ指定できます。
    *
    * @return       常に nn::fs::ResultUnsupportedOperation を返します。
    */
    virtual Result SetSize(int64_t) NN_NOEXCEPT NN_OVERRIDE
    {
        return nn::fs::ResultUnsupportedOperation();
    }

    /**
    * @brief        フラッシュします。
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result Flush() NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief       範囲指定処理を行います。
    *
    * @param[out]  outBuffer        範囲指定処理の結果を格納するバッファ
    * @param[in]   outBufferSize    範囲指定処理の結果を格納するバッファのサイズ
    * @param[in]   operationId      範囲指定処理の種類
    * @param[in]   offset           範囲指定処理開始位置
    * @param[in]   size             範囲指定処理を行うデータサイズ
    * @param[in]   inBuffer         範囲指定処理に渡すバッファ
    * @param[in]   inBufferSize     範囲指定処理に渡すバッファのサイズ
    *
    * @return      関数の処理結果を返します。
    */
    virtual Result OperateRange(
                       void* outBuffer,
                       size_t outBufferSize,
                       fs::OperationId operationId,
                       int64_t offset,
                       int64_t size,
                       const void* inBuffer,
                       size_t inBufferSize
                   ) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        ブロックサイズを取得します。
    */
    int64_t GetBlockSize() const NN_NOEXCEPT
    {
        return m_SizeBlock;
    }

    /**
    * @brief        読み込み専用かどうかを設定します。(デバッグ用)
    */
    inline void SetReadOnly(bool isReadOnly) NN_NOEXCEPT
    {
        m_IsReadOnly = isReadOnly;
    }

private:
    /**
    * @brief        指定したブロックでの二重化選択ビットの値を読み込みます。
    *
    * @param[out]   outOriginalBits オリジナルビットデータ
    * @param[out]   outModifiedBits 編集後ビットデータ
    * @param[in]    offset          オフセット
    * @param[in]    bitCount        読み込むビット数
    *
    * @return       関数の処理結果を返します。
    */
    Result ReadDuplexBits(
               uint32_t* outOriginalBits,
               uint32_t* outModifiedBits,
               int64_t offset,
               size_t bitCount
           ) NN_NOEXCEPT;

    /**
    * @brief        上位レイヤーでの二重化選択ビットマップデータを更新します。
    *
    * @param[in]    offset      オフセット
    * @param[in]    offsetEnd   変更状態を示すビット列
    *
    * @return       関数の処理結果を返します。
    */
    Result UpdateModifiedBits(
               int64_t offset,
               int64_t offsetEnd
           ) NN_NOEXCEPT;

private:
    dbm::DuplexBitmap* m_pDuplexBitmap;         //! 二重化選択ビットマップデータ
    fs::SubStorage m_Storage[2];                //! ストレージデータ(二重化のため 2 つ持ちます)
    int64_t m_CountShiftBlock;                  //! ブロック数を求めるためのシフト値
    int64_t m_SizeBlock;                        //! ブロックサイズ
    IBufferManager* m_pBufferManager;           //! バッファマネージャ
    bool m_IsReadOnly;                          //! 読み込み専用かどうか(デバッグ用)
    char reserved[3];

private:
    // テスト用クラスには内部データを公開します。
    friend class ::FsDuplexStorageTest;
};

}}}

