﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/os.h>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/util/util_Optional.h>
#include <nn/fs/fs_Directory.h>
#include <nn/fs/fs_SaveDataManagement.h>
#include <nn/fs/fsa/fs_IDirectory.h>
#include <nn/fs/fsa/fs_IFileSystem.h>

namespace nn { namespace fssystem {

    namespace detail {

        template<typename EnterDirectory, typename ExitDirectory, typename FindFile>
        Result IterateDirectoryRecursiveInternal(
            nn::fs::fsa::IFileSystem* pFileSystem,
            char* path,
            size_t size,
            nn::fs::DirectoryEntry* pDirectoryEntry,
            EnterDirectory enterDirectory,
            ExitDirectory exitDirectory,
            FindFile findFile
        ) NN_NOEXCEPT
        {
            std::unique_ptr<nn::fs::fsa::IDirectory> directory;
            NN_RESULT_DO(pFileSystem->OpenDirectory(&directory, path, nn::fs::OpenDirectoryMode_All));

            while(NN_STATIC_CONDITION(true))
            {
                int64_t readNum = 0;
                NN_RESULT_DO(directory->Read(&readNum, pDirectoryEntry, 1));

                if( readNum == 0 )
                {
                    break;
                }

                auto parentPathSize = strnlen(path, size - 1);
                auto entryNameSize = strnlen(pDirectoryEntry->name, sizeof(pDirectoryEntry->name) - 1);
                auto separatorSize = pDirectoryEntry->directoryEntryType == nn::fs::DirectoryEntryType_Directory ? 1 : 0;
                NN_RESULT_THROW_UNLESS(
                    parentPathSize + entryNameSize + separatorSize < size,
                    nn::fs::ResultTooLongPath());
                strncat(path, pDirectoryEntry->name, size - parentPathSize - 1);
                {
                    if( pDirectoryEntry->directoryEntryType == nn::fs::DirectoryEntryType_Directory )
                    {
                        NN_RESULT_DO(enterDirectory(path, *pDirectoryEntry));
                        strncat(path, "/", size - (parentPathSize + entryNameSize) - 1);
                        NN_RESULT_DO(IterateDirectoryRecursiveInternal(pFileSystem, path, size, pDirectoryEntry, enterDirectory, exitDirectory, findFile));
                        NN_RESULT_DO(exitDirectory(path, *pDirectoryEntry));
                    }
                    else
                    {
                        NN_RESULT_DO(findFile(path, *pDirectoryEntry));
                    }
                }
                path[parentPathSize] = '\0';

            }
            NN_RESULT_SUCCESS;
        }

        template<typename EnterDirectory, typename ExitDirectory, typename DeleteFile>
        Result CleanupDirectoryRecursiveInternal(
            nn::fs::fsa::IFileSystem* pFileSystem,
            char* path,
            size_t size,
            nn::fs::DirectoryEntry* pDirectoryEntry,
            EnterDirectory enterDirectory,
            ExitDirectory exitDirectory,
            DeleteFile deleteFile
        ) NN_NOEXCEPT
        {
            std::unique_ptr<nn::fs::fsa::IDirectory> directory;

            while(NN_STATIC_CONDITION(true))
            {
                int64_t readNum = 0;
                NN_RESULT_DO(pFileSystem->OpenDirectory(&directory, path, nn::fs::OpenDirectoryMode_All));
                NN_RESULT_DO(directory->Read(&readNum, pDirectoryEntry, 1));
                directory.reset();

                if( readNum == 0 )
                {
                    break;
                }

                auto parentPathSize = strnlen(path, size - 1);
                auto entryNameSize = strnlen(pDirectoryEntry->name, sizeof(pDirectoryEntry->name) - 1);
                auto separatorSize = pDirectoryEntry->directoryEntryType == nn::fs::DirectoryEntryType_Directory ? 1 : 0;
                NN_RESULT_THROW_UNLESS(
                    parentPathSize + entryNameSize + separatorSize < size,
                    nn::fs::ResultTooLongPath());
                strncat(path, pDirectoryEntry->name, size - parentPathSize - 1);
                {
                    if( pDirectoryEntry->directoryEntryType == nn::fs::DirectoryEntryType_Directory )
                    {
                        NN_RESULT_DO(enterDirectory(path, *pDirectoryEntry));
                        strncat(path, "/", size - (parentPathSize + entryNameSize) - 1);
                        NN_RESULT_DO(CleanupDirectoryRecursiveInternal(pFileSystem, path, size, pDirectoryEntry, enterDirectory, exitDirectory, deleteFile));
                        NN_RESULT_DO(exitDirectory(path, *pDirectoryEntry));
                    }
                    else
                    {
                        NN_RESULT_DO(deleteFile(path, *pDirectoryEntry));
                    }
                }
                path[parentPathSize] = '\0';

            }
            NN_RESULT_SUCCESS;
        }
    }

    template<typename EnterDirectory, typename ExitDirectory, typename FindFile>
    Result IterateDirectoryRecursive(
        nn::fs::fsa::IFileSystem* pFileSystem,
        const char* rootPath,
        char* pathBuffer,
        size_t sizePathBuffer,
        nn::fs::DirectoryEntry* pDirectoryEntryBuffer,
        EnterDirectory enterDirectory,
        ExitDirectory exitDirectory,
        FindFile findFile
    ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_EQUAL(sizePathBuffer, static_cast<size_t>(nn::fs::EntryNameLengthMax + 1));

        auto lengthRootPath = strnlen(rootPath, nn::fs::EntryNameLengthMax + 1);
        if (lengthRootPath > nn::fs::EntryNameLengthMax ||
            (lengthRootPath == nn::fs::EntryNameLengthMax && rootPath[lengthRootPath - 1] != '/'))
        {
            return nn::fs::ResultTooLongPath();
        }

        std::memcpy(pathBuffer, rootPath, lengthRootPath);
        if (pathBuffer[lengthRootPath - 1] != '/')
        {
            ++lengthRootPath;
            pathBuffer[lengthRootPath - 1] = '/';
        }
        pathBuffer[lengthRootPath] = '\0';
        return detail::IterateDirectoryRecursiveInternal(pFileSystem, pathBuffer, sizePathBuffer, pDirectoryEntryBuffer, enterDirectory, exitDirectory, findFile);
    }

    template<typename EnterDirectory, typename ExitDirectory, typename FindFile>
    Result IterateDirectoryRecursive(
        nn::fs::fsa::IFileSystem* pFileSystem,
        const char* rootPath,
        EnterDirectory enterDirectory,
        ExitDirectory exitDirectory,
        FindFile findFile
    ) NN_NOEXCEPT
    {
        nn::fs::DirectoryEntry directoryEntryBuffer = {{0}};
        char path[nn::fs::EntryNameLengthMax + 1] = {0};
        return IterateDirectoryRecursive<EnterDirectory, ExitDirectory, FindFile>(pFileSystem, rootPath, path, sizeof(path), &directoryEntryBuffer, enterDirectory, exitDirectory, findFile);
    }

    template<typename EnterDirectory, typename ExitDirectory, typename FindFile>
    Result IterateDirectoryRecursive(
        nn::fs::fsa::IFileSystem* pFileSystem,
        EnterDirectory enterDirectory,
        ExitDirectory exitDirectory,
        FindFile findFile
    ) NN_NOEXCEPT
    {
        return IterateDirectoryRecursive<EnterDirectory, ExitDirectory, FindFile>(
            pFileSystem,
            "/",
            enterDirectory,
            exitDirectory,
            findFile
        );
    }

    template<typename EnterDirectory, typename ExitDirectory, typename DeleteFile>
    Result CleanupDirectoryRecursive(
        nn::fs::fsa::IFileSystem* pFileSystem,
        const char* rootPath,
        char* pathBuffer,
        size_t sizePathBuffer,
        nn::fs::DirectoryEntry* pDirectoryEntryBuffer,
        EnterDirectory enterDirectory,
        ExitDirectory exitDirectory,
        DeleteFile deleteFile
    ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_EQUAL(sizePathBuffer, static_cast<size_t>(nn::fs::EntryNameLengthMax + 1));

        auto lengthRootPath = strnlen(rootPath, nn::fs::EntryNameLengthMax + 1);
        if (lengthRootPath > nn::fs::EntryNameLengthMax ||
            (lengthRootPath == nn::fs::EntryNameLengthMax && rootPath[lengthRootPath - 1] != '/'))
        {
            return nn::fs::ResultTooLongPath();
        }

        std::memcpy(pathBuffer, rootPath, lengthRootPath);
        if (pathBuffer[lengthRootPath - 1] != '/')
        {
            ++lengthRootPath;
            pathBuffer[lengthRootPath - 1] = '/';
        }
        pathBuffer[lengthRootPath] = '\0';
        return detail::CleanupDirectoryRecursiveInternal(pFileSystem, pathBuffer, sizePathBuffer, pDirectoryEntryBuffer, enterDirectory, exitDirectory, deleteFile);
    }

    template<typename EnterDirectory, typename ExitDirectory, typename DeleteFile>
    Result CleanupDirectoryRecursive(
        nn::fs::fsa::IFileSystem* pFileSystem,
        const char* rootPath,
        EnterDirectory enterDirectory,
        ExitDirectory exitDirectory,
        DeleteFile deleteFile
    ) NN_NOEXCEPT
    {
        nn::fs::DirectoryEntry directoryEntryBuffer = {{0}};
        char path[nn::fs::EntryNameLengthMax + 1] = {0};
        return CleanupDirectoryRecursive<EnterDirectory, ExitDirectory, DeleteFile>(pFileSystem, rootPath, path, sizeof(path), &directoryEntryBuffer, enterDirectory, exitDirectory, deleteFile);
    }

    template<typename EnterDirectory, typename ExitDirectory, typename DeleteFile>
    Result CleanupDirectoryRecursive(
        nn::fs::fsa::IFileSystem* pFileSystem,
        EnterDirectory enterDirectory,
        ExitDirectory exitDirectory,
        DeleteFile deleteFile
    ) NN_NOEXCEPT
    {
        return CleanupDirectoryRecursive<EnterDirectory, ExitDirectory, DeleteFile>(
            pFileSystem,
            "/",
            enterDirectory,
            exitDirectory,
            deleteFile
        );
    }

    /**
    * @brief pred に最初にマッチするセーブデータを取得します。
    */
    template <typename Pred>
    Result FindSaveData(nn::util::optional<nn::fs::SaveDataInfo>* pOutValue, nn::fs::SaveDataInfo* pWorkSaveDataInfo, int saveDataInfoNum, nn::fs::SaveDataSpaceId spaceId, Pred pred)
    {
        std::unique_ptr<nn::fs::SaveDataIterator> iter;
        NN_RESULT_DO(nn::fs::OpenSaveDataIterator(&iter, spaceId));

        int64_t count;
        while(NN_STATIC_CONDITION(true))
        {
            count = 0;
            memset(pWorkSaveDataInfo, 0x00, sizeof(fs::SaveDataInfo) * saveDataInfoNum);
            NN_RESULT_DO(iter->ReadSaveDataInfo(&count, pWorkSaveDataInfo, saveDataInfoNum));
            if( count == 0 )
            {
                *pOutValue = nn::util::nullopt;
                NN_RESULT_SUCCESS;
            }
            for(int64_t i = 0; i < count; i++)
            {
                if( pred(pWorkSaveDataInfo[i]) )
                {
                    *pOutValue = pWorkSaveDataInfo[i];
                    NN_RESULT_SUCCESS;
                }
            }
        }
    }


    //! @brief デバッグ用途
    Result ListDirectoryRecursive(nn::fs::fsa::IFileSystem* pFileSystem) NN_NOEXCEPT;

    Result CopyFile(nn::fs::fsa::IFileSystem* pDstFileSystem, nn::fs::fsa::IFileSystem* pSrcFileSystem, const char* destinationParentPath, const char* SourcePath, const nn::fs::DirectoryEntry* pEntry, char* workBuffer, size_t workBufferSize);
    Result CopyDirectoryRecursive(nn::fs::fsa::IFileSystem* pDstFileSystem, nn::fs::fsa::IFileSystem* pSrcFileSystem, const char* destinationPath, const char* sourcePath, char* workBuffer, size_t workBufferSize);

    Result VerifyDirectoryRecursive(nn::fs::fsa::IFileSystem* pFileSystem, char* buffer, size_t bufferSize) NN_NOEXCEPT;

    //! @brief pCounter を counterSize バイト幅のビッグエンディアン符号無し整数とみなして additionalValue を加算する
    void AddCounter(void* pCounter, size_t counterSize, uint64_t additionalValue) NN_NOEXCEPT;

    bool IsWindowsDrive(const char* path) NN_NOEXCEPT;

    class SemaphoreAdaptor : public os::Semaphore
    {
    public:
        SemaphoreAdaptor(int initialCount, int maxCount)
            : os::Semaphore(initialCount, maxCount)
        {
        }

        void unlock()
        {
            Release();
        }
        bool try_lock()
        {
            return TryAcquire();
        }
    };


}}
