﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once

#include <memory>
#include <mutex>
#include <shared_mutex>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>

#include <nn/nn_SdkLog.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Abort.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/os/os_ReaderWriterLock.h>

#include <nn/fs/detail/fs_Newable.h>
#include <nn/fs/fs_IStorage.h>

namespace nn { namespace fssystem {

class ProxyStorage : public nn::fs::IStorage, public nn::fs::detail::Newable
{
    NN_DISALLOW_COPY(ProxyStorage);

public:
    ProxyStorage(std::shared_ptr<IStorage> pBaseStorage, Result resultForInvalidatedState) NN_NOEXCEPT
        : m_pBaseStorage(std::move(pBaseStorage))
        , m_ResultForInvalidated(resultForInvalidatedState)
    {
    }

public:
    virtual Result Read(int64_t offset, void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE
    {
        std::shared_lock<os::ReaderWriterLock> scopedReaderLock(m_Lock);
        NN_RESULT_THROW_UNLESS(m_pBaseStorage != nullptr, m_ResultForInvalidated);
        return m_pBaseStorage->Read(offset, buffer, size);
    }

    virtual Result Write(int64_t offset, const void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE
    {
        std::shared_lock<os::ReaderWriterLock> scopedReaderLock(m_Lock);
        NN_RESULT_THROW_UNLESS(m_pBaseStorage != nullptr, m_ResultForInvalidated);
        return m_pBaseStorage->Write(offset, buffer, size);
    }

    virtual Result Flush() NN_NOEXCEPT NN_OVERRIDE
    {
        std::shared_lock<os::ReaderWriterLock> scopedReaderLock(m_Lock);
        NN_RESULT_THROW_UNLESS(m_pBaseStorage != nullptr, m_ResultForInvalidated);
        return m_pBaseStorage->Flush();
    }

    virtual Result SetSize(int64_t size) NN_NOEXCEPT NN_OVERRIDE
    {
        std::shared_lock<os::ReaderWriterLock> scopedReaderLock(m_Lock);
        NN_RESULT_THROW_UNLESS(m_pBaseStorage != nullptr, m_ResultForInvalidated);
        return m_pBaseStorage->SetSize(size);
    }

    virtual Result GetSize(int64_t* outValue) NN_NOEXCEPT NN_OVERRIDE
    {
        std::shared_lock<os::ReaderWriterLock> scopedReaderLock(m_Lock);
        NN_RESULT_THROW_UNLESS(m_pBaseStorage != nullptr, m_ResultForInvalidated);
        return m_pBaseStorage->GetSize(outValue);
    }

    virtual Result OperateRange(
                       void* outBuffer,
                       size_t outBufferSize,
                       nn::fs::OperationId operationId,
                       int64_t offset,
                       int64_t size,
                       const void* inBuffer,
                       size_t inBufferSize
                   ) NN_NOEXCEPT NN_OVERRIDE
    {
        std::shared_lock<os::ReaderWriterLock> scopedReaderLock(m_Lock);
        NN_RESULT_THROW_UNLESS(m_pBaseStorage != nullptr, m_ResultForInvalidated);
        return m_pBaseStorage->OperateRange(
            outBuffer, outBufferSize, operationId, offset, size, inBuffer, inBufferSize);
    }

    void Invalidate() NN_NOEXCEPT
    {
        std::lock_guard<os::ReaderWriterLock> scopedWriterLock(m_Lock);
        m_pBaseStorage.reset();
    }

private:
    std::shared_ptr<IStorage> m_pBaseStorage;
    Result m_ResultForInvalidated;
    os::ReaderWriterLock m_Lock;
};


}}

