﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cstring>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkLog.h>

#include <nn/fssystem/fs_DbmRomTypes.h>

namespace nn { namespace fssystem {

/*!
    :private

    @brief ROM ファイルシステム用パス操作関数群
*/
namespace RomPathTool
{
    using namespace StringTraitsRom;

    // パスの最大長
    static const uint32_t MAX_PATH_LENGTH = 768;

    static const RomPathChar DOT_CHAR = '.';

    // エントリ名
    struct RomEntryName
    {
        size_t length;
        const RomPathChar* path;
    };
    NN_STATIC_ASSERT(std::is_pod<RomEntryName>::value);

    /*!
        :private

        @brief 指定した文字が区切り文字かどうかを取得します。

        @param[in] ch 文字

        @return 区切り文字かどうか
    */
    inline bool IsSeparator(RomPathChar ch)
    {
        return (ch == DIR_SEPARATOR);
    }

    /*!
        :private

        @brief カレントディレクトリを示す文字かどうかを取得します。

        @param[in] pDirName ディレクトリ名

        @return カレントディレクトリかどうか
    */
    inline bool IsCurrentDirectory(const RomEntryName& dirName)
    {
        return ((dirName.length == 1) && (dirName.path[0] == DOT_CHAR));
    }
    inline bool IsCurrentDirectory(const RomPathChar* pDirName)
    {
        return ((pDirName[0] == DOT_CHAR) && (pDirName[1] == NUL));
    }
    inline bool IsCurrentDirectory(const RomPathChar* pDirName, size_t length)
    {
        return ((length == 1) && (pDirName[0] == DOT_CHAR));
    }

    /*!
        :private

        @brief 親ディレクトリを示す文字かどうかを取得します。

        @param[in] pDirName ディレクトリ名

        @return カレントディレクトリかどうか
    */
    inline bool IsParentDirectory(const RomEntryName& dirName)
    {
        return ((dirName.length == 2) && (dirName.path[0] == DOT_CHAR) && (dirName.path[1] == DOT_CHAR));
    }
    inline bool IsParentDirectory(const RomPathChar* pDirName)
    {
        return ((pDirName[0] == DOT_CHAR) && (pDirName[1] == DOT_CHAR) && (pDirName[2] == NUL));
    }
    inline bool IsParentDirectory(const RomPathChar* pDirName, size_t length)
    {
        return ((length == 2) && (pDirName[0] == DOT_CHAR) && (pDirName[1] == DOT_CHAR));
    }

    /*!
        :private

        @brief パスが等しいかどうかを取得します。
    */
    inline bool IsEqualPath(const RomPathChar* pExtraKey1, const RomPathChar* pExtraKey2, size_t length)
    {
//        return (wcsncmp(pExtraKey1, pExtraKey2, length) == 0);
        return (strncmp(pExtraKey1, pExtraKey2, length) == 0);
    }

    /*!
        :private

        @brief ROM エントリー名とファイルパスが等しいかどうかを取得します。

        @param[in] name1 ROM エントリー名
        @param[in] name2 ファイルパス

        @return ROM エントリー名が等しければ true、そうでなければ false
    */
    inline bool IsEqualName(const RomEntryName& name1, const RomPathChar* pName2)
    {
        size_t len = strnlen(pName2, MAX_PATH_LENGTH);
        if (name1.length != len)
        {
            return false;
        }
        return IsEqualPath(name1.path, pName2, len);
    }

    /*!
        :private

        @brief ROM エントリー名が等しいかどうかを取得します。

        @param[in] name1 ROM エントリー名
        @param[in] name2 ROM エントリー名

        @return ROM エントリー名が等しければ true、そうでなければ false
    */
    inline bool IsEqualName(const RomEntryName& name1, const RomEntryName& name2)
    {
        if (name1.length != name2.length)
        {
            return false;
        }
        return IsEqualPath(name1.path, name2.path, name1.length);
    }

    /*!
        :private

        @brief ROM エントリー名を初期化します。

        @param[in] pName ROM エントリー名
    */
    inline void InitEntryName(RomEntryName* pName)
    {
        pName->length = 0;
    }

    /*!
        :private

        @brief フルパスを解析し、イテレーションするためのクラスです。
    */
    class PathParser
    {
    private:
        const RomPathChar* m_pPrevStartPath;   // 解析済みのパス始点へのポインタ
        const RomPathChar* m_pPrevEndPath;     // 解析済みのパス終端へのポインタ
        const RomPathChar* m_pNextPath;        // 次回解析するパスへのポインタ
        bool m_bParseFinished;                 // NUL 文字に到着したかどうか

    public:
        /*!
            @brief コンストラクタです。
        */
        PathParser();

        /*!
            @brief パーサを初期化します。

            @param[in] pFullPath フルパス

            @return 関数の処理結果を返します。
        */
        Result Initialize(const RomPathChar* pFullPath);

        /*!
            @brief パーサを破棄します。
        */
        void Finalize();

        /*!
            @brief パスのパースが完了したかどうかを取得します。
        */
        bool IsParseFinished() const;

        /*!
            @brief パスがディレクトリだったかどうかを取得します。

                   IsParseFinished が true の時のみ使用できます。
        */
        bool IsDirectoryPath() const;

        /*!
            @brief 最上位ディレクトリ部分を取得します。

            @param[out] pDirName ディレクトリ名

            @return 関数の処理結果を返します。
        */
        Result GetNextDirectoryName(RomEntryName* pDirName);

        /*!
            @brief 文字列の先頭から次の１節をディレクトリ名として取得します。

            @param[out] pName ディレクトリ名

            @return 関数の処理結果を返します。
        */
        Result GetAsDirectoryName(RomEntryName* pName) const;

        /*!
            @brief 文字列の先頭から次の１節をファイル名として取得します。

            @param[out] pName ファイル名

            @return 関数の処理結果を返します。
        */
        Result GetAsFileName(RomEntryName* pName) const;
    };

    /*!
        :private

        @brief 親ディレクトリ名を抽出します。

               フルパス:"/AAA/BBB/CCC" 開始:"AAA" => 親ディレクトリ:"" (ルートディレクトリ)
               フルパス:"/AAA/BBB/CCC" 開始:"BBB" => 親ディレクトリ:"AAA"
               フルパス:"/AAA/BBB/CCC/." 開始:"." => 親ディレクトリ:"BBB"
               フルパス:"/AAA/BBB/CCC/.." 開始:".." => 親ディレクトリ:"AAA"

        @param[out] pOut 親ディレクトリ名
        @param[in] base 開始ディレクトリ名
                        フルパス内の文字列を指している必要があります。
        @param[in] pFullPath フルパス

        @return 関数の処理結果を返します。
    */
    Result GetParentDirectoryName(RomEntryName* pOut, const RomEntryName& base, const RomPathChar* pHead);

#ifndef NN_SWITCH_DISABLE_DEBUG_PRINT_FOR_SDK
    inline void Print(const RomPathChar* pStr)
    {
        NN_UNUSED(pStr);
        NN_SDK_LOG(reinterpret_cast<const char *>(pStr));
    }
#endif //NN_SWITCH_DISABLE_DEBUG_PRINT_FOR_SDK
}

}}
