﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/fs/fsa/fs_IFileSystem.h>
#include <nn/fs/detail/fs_Newable.h>
#include <nn/fssystem/fs_AesXtsFile.h>

#include <nn/crypto/crypto_AesEncryptor.h>

namespace nn { namespace fssystem {

class AesXtsFileSystem : public fs::fsa::IFileSystem, public fs::detail::Newable
{
    NN_DISALLOW_COPY(AesXtsFileSystem);

public:
    typedef AesXtsFileHeader::GenerateRandomFunction GenerateRandomFunction;

public:
    static const size_t AesBlockSize = AesXtsStorage::AesBlockSize;
    static const size_t EncryptionKeyGenerationKeySize = AesXtsFileHeader::EncryptionKeyGenerationKeySize;
    static const size_t MacKeySize = AesXtsFileHeader::MacKeySize;
    static const size_t IvSize  = AesBlockSize;

public:
    AesXtsFileSystem() NN_NOEXCEPT;
    virtual ~AesXtsFileSystem() NN_NOEXCEPT NN_OVERRIDE;

    /**
       @pre
       - encryptionKeyGenerationKeySize == EncryptionKeyGenerationKeySize
       - macKeySize == MacKeySize
       - xtsBlockSize は AesBockSizeの倍数
    */
    Result Initialize(
        std::shared_ptr<fs::fsa::IFileSystem> pBaseFileSystem,
        const void* pEncryptionKeyGenerationKey, size_t encryptionKeyGenerationKeySize,
        const void* pMacKey, size_t macKeySize,
        GenerateRandomFunction pGenerateRandom,
        size_t xtsBlockSize
    ) NN_NOEXCEPT;

    /**
    * @brief        セットされた乱数列生成関数を実行して乱数を生成します。
    *
    * @return       なし
    */
    void GenerateRandom(
        void* pOutData,
        size_t dataSize
    ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutData);
        NN_ABORT_UNLESS_NOT_NULL(m_pGenerateRandom);
        m_pGenerateRandom(pOutData, dataSize);
    }

private:
    virtual Result DoCreateFile(const char* path, int64_t size, int option) NN_NOEXCEPT NN_OVERRIDE;
    virtual Result DoDeleteFile(const char* path) NN_NOEXCEPT NN_OVERRIDE;
    virtual Result DoCreateDirectory(const char* path) NN_NOEXCEPT NN_OVERRIDE;
    virtual Result DoDeleteDirectory(const char* path) NN_NOEXCEPT NN_OVERRIDE;
    virtual Result DoDeleteDirectoryRecursively(const char* path) NN_NOEXCEPT NN_OVERRIDE;
    virtual Result DoCleanDirectoryRecursively(const char* path) NN_NOEXCEPT NN_OVERRIDE;
    virtual Result DoRenameFile(const char* currentPath, const char* newPath) NN_NOEXCEPT NN_OVERRIDE;
    virtual Result DoRenameDirectory(const char* currentPath, const char* newPath) NN_NOEXCEPT NN_OVERRIDE;
    virtual Result DoGetEntryType(fs::DirectoryEntryType* outValue, const char* path) NN_NOEXCEPT NN_OVERRIDE;
    virtual Result DoGetFreeSpaceSize(int64_t* outValue, const char* path) NN_NOEXCEPT NN_OVERRIDE;
    virtual Result DoGetTotalSpaceSize(int64_t* outValue, const char* path) NN_NOEXCEPT NN_OVERRIDE;
    virtual Result DoOpenFile(std::unique_ptr<fs::fsa::IFile>* outValue, const char* path, fs::OpenMode mode) NN_NOEXCEPT NN_OVERRIDE;
    virtual Result DoOpenDirectory(std::unique_ptr<fs::fsa::IDirectory>* outValue, const char* path, fs::OpenDirectoryMode mode) NN_NOEXCEPT NN_OVERRIDE;
    virtual Result DoCommit() NN_NOEXCEPT NN_OVERRIDE;
    virtual Result DoCommitProvisionally(int64_t counter) NN_NOEXCEPT NN_OVERRIDE;
    virtual Result DoRollback() NN_NOEXCEPT NN_OVERRIDE;
    virtual Result DoQueryEntry(char* outBuffer, size_t outBufferSize, const char* inBuffer, size_t inBufferSize, fs::fsa::QueryId queryId, const char* path) NN_NOEXCEPT NN_OVERRIDE;

    Result DecryptHeader(AesXtsFileHeader* pOutHeader, const char* path, const char* decryptPath) NN_NOEXCEPT;
    Result EncryptHeader(AesXtsFileHeader* pHeader, const char* path, const char* encryptPath) NN_NOEXCEPT;

private:
    std::shared_ptr<fs::fsa::IFileSystem> m_pBaseFileSystem;
    char m_EncryptionKeyGenerationKey[EncryptionKeyGenerationKeySize];
    char m_MacKey[MacKeySize];
    size_t m_XtsBlockSize;
    GenerateRandomFunction m_pGenerateRandom;
};

}}
