﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_StaticAssert.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_ResultPrivate.h>
#include <nn/fssystem/dbm/fs_BufferedAllocationTableStorage.h>

class KeyValueListTemplateTest;

namespace nn { namespace fssystem { namespace dbm {

/**
* @brief    キーバリューストレージテンプレートクラスです。
*
* @tparam   Key
* @tparam   Value
*
* @details  キーバリューストレージテンプレートクラスです。
*           キーとバリューの組合わせを記憶します。
*/
template <typename TKey, typename TValue>
class KeyValueListTemplate
{
    NN_DISALLOW_COPY(KeyValueListTemplate);

public:
    typedef uint32_t Index;         //! キーバリューストレージのインデックス
    typedef TKey Key;
    typedef TValue Value;

    //! キーバリューストレージのエントリーです。
    struct StorageElement
    {
        Key     key;                //! キー
        Value   value;              //! 実データへのインデックス
        Index   next;               //! 同じハッシュキーを共有する次のインデクス
    };
    NN_STATIC_ASSERT(std::is_pod<StorageElement>::value);

    //! キーと値の列挙に使用する探索位置指示構造体です。
    struct FindIndex
    {
        Index index;
    };
    NN_STATIC_ASSERT(std::is_pod<FindIndex>::value);

public:
    /**
    * @brief        格納可能なキー数からキーバリューストレージのサイズを求めます。
    *
    * @param[in]    countKeyValue   キーバリュー数
    *
    * @return       必要なストレージのバイトサイズ。
    *
    * @details      格納可能なキー数からキーバリューストレージのサイズを求めます。
    */
    static inline uint32_t QueryStorageSize(uint32_t countKeyValue) NN_NOEXCEPT
    {
        return (countKeyValue + AllocationTableStorageReservedCount) * sizeof(StorageElement);
    }

    /**
    * @brief        キーバリューストレージのサイズから格納可能なキー数を求めます。
    *
    * @param[in]    size        キーバリューストレージのサイズ
    *
    * @return       格納可能なキー数。
    *
    * @details      QueryStorageSize の逆です。
    */
    static inline uint32_t QueryEntryCount(int64_t size) NN_NOEXCEPT
    {
        uint32_t countAll = static_cast<uint32_t>(size / sizeof(StorageElement));
        return countAll - AllocationTableStorageReservedCount;
    }

    /**
    * @brief        ストレージを初期化します。
    *
    * @param[in]    pStorage            アロケーションテーブルストレージ
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess                   正常に取得できました。
    * @retval       ResultNotInitialized            初期化されていません。
    * @retval       ResultInvalidOffset             pStorage が小さすぎます。
    * @retval       ResultDatabaseCorrupted         内部データに問題があります。
    * @retval       上記以外                        ストレージからの読み込みに失敗しました。
    *
    * @pre          pStorage != nullptr
    * @pre          (1 ブロックのサイズ) > (1 エントリーのサイズ)
    *
    * @details      ストレージを初期化します。
    *               ストレージをマウントする前に一度呼び出す必要があります。
    */
    static Result Format(AllocationTableStorage* pStorage) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pStorage);

        // サイズチェック
        {
            // 1 ブロックのサイズは要素のサイズ以上である必要があります。
            int64_t blockSize = pStorage->GetBlockSize(AllocationTableStorageAllocateBlockCount);
            NN_SDK_ASSERT_LESS_EQUAL(sizeof(StorageElement), static_cast<size_t>(blockSize));
            NN_UNUSED(blockSize);
        }

        // 割り当て領域分のフリーリストを作ります。
        StorageElement storageElement;
        std::memset(&storageElement, 0, sizeof(StorageElement));
        storageElement.next = AllocationTableStorageFreeEntry;

        // フォーマット済みの領域を控えておきます。
        FormatInformation* pFormatInfo = FormatInformation::GetFromStorageElement(&storageElement);
        pFormatInfo->indexNotFormatted = AllocationTableStorageReservedCount;
        int64_t allocSize;
        NN_RESULT_DO(pStorage->GetSize(&allocSize));
        uint32_t newLimitation = QueryEntryCount(allocSize) + AllocationTableStorageReservedCount;
        pFormatInfo->indexLimitation = newLimitation;

        // インデックスは 2 つ以上が必要
        NN_SDK_ASSERT_LESS_EQUAL(AllocationTableStorageReservedCount, newLimitation);

        // 管理情報部分のみ書き込みます
        NN_RESULT_DO(
            pStorage->Write(
                sizeof(StorageElement) * AllocationTableStorageFreeEntry,
                &storageElement,
                sizeof(StorageElement)
            )
        );

        // リストの先頭部分を書き込みます
        StorageElement storageElementHead;
        std::memset(&storageElementHead, 0, sizeof(StorageElement));
        storageElementHead.next = AllocationTableStorageFreeEntry;
        NN_RESULT_DO(
            pStorage->Write(
                sizeof(StorageElement) * AllocationTableStorageHeadEntry,
                &storageElementHead,
                sizeof(StorageElement)
            )
        );
        NN_RESULT_SUCCESS;
    }

    /**
    * @brief        コンストラクタです。
    *
    * @details      コンストラクタです。
    */
    KeyValueListTemplate() NN_NOEXCEPT
    : m_pKeyValueStorage(nullptr)
    {
    }

    /**
    * @brief        ストレージをマウントします。
    *
    * @param[in]    pKeyValueStorage    キーバリューストレージ
    *
    * @pre          Format をしたことがある
    * @pre          pKeyValueStorage != nullptr
    *
    * @details      一度も Format せずに呼んだ場合の動作は未定義です。
    */
    void Initialize(BufferedAllocationTableStorage* pKeyValueStorage) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pKeyValueStorage);
        m_pKeyValueStorage = pKeyValueStorage;
    }

    /**
    * @brief        ストレージをアンマウントします。
    *
    * @details      ストレージをアンマウントします。
    */
    void Finalize() NN_NOEXCEPT
    {
        m_pKeyValueStorage = nullptr;
    }

    /**
    * @brief        要素数を求めます。
    *
    * @param[out]   outValue            要素数
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess                   処理が正常に終了しました。
    * @retval       ResultNotInitialized            初期化されていません。
    * @retval       ResultInvalidOffset             内部で不明な問題が発生しました。
    * @retval       ResultDatabaseCorrupted         内部データに問題があります。
    * @retval       上記以外                        ストレージからのデータ読み込みに失敗しました。
    *
    * @pre          outValue != nullptr
    * @pre          マウントしている。
    *
    * @details      要素数を求めます。
    */
    Result GetMaxCount(Index* outValue) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(outValue);
        StorageElement element;
        NN_RESULT_DO(ReadKeyValueImpl(&element, AllocationTableStorageFreeEntry));
        FormatInformation* pFormatInfo = FormatInformation::GetFromStorageElement(&element);
        *outValue = pFormatInfo->indexLimitation;
        NN_RESULT_SUCCESS;
    }

    /**
    * @brief        キーと値を追加、または更新します。
    *
    * @param[in]    key                  キー
    * @param[in]    value                値
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess                   キーと値を正常に追加、または更新できました。
    * @retval       ResultNotInitialized            初期化されていません。
    * @retval       ResultAllocationTableFull       アロケーションテーブルに空きがありません。
    * @retval       ResultInvalidOffset             内部で不明な問題が発生しました。
    * @retval       ResultInvalidOffset             内部データに問題があります。
    * @retval       ResultDatabaseCorrupted         内部データに問題があります。
    * @retval       上記以外                        ストレージでのデータ読み書きに失敗しました。
    *
    * @pre          マウントしている。
    *
    * @details     同名のキーが既に存在する場合、値を更新します。
    */
    Result Add(const Key& key, const Value& value) NN_NOEXCEPT
    {
        Index indexAdded;
        return AddInternal(&indexAdded, key, value);
    }

    /**
    * @brief        キーに対する値を取得します。
    *
    * @param[out]   outValue                値
    * @param[in]    key                     キー
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess                   処理が正常に終了しました。
    * @retval       ResultNotInitialized            初期化されていません。
    * @retval       ResultDatabaseKeyNotFound       key に一致するエントリーがありません。
    * @retval       ResultInvalidOffset             内部データに問題があります。
    * @retval       ResultDatabaseCorrupted         内部データに問題があります。
    * @retval       上記以外                        ストレージからのデータ読み込みに失敗しました。
    *
    * @pre          マウントしている。
    * @pre          outValue != nullptr
    *
    * @details      キーに対する値を取得します。
    */
    Result Get(Value* outValue, const Key& key) const NN_NOEXCEPT
    {
        Index index;
        return GetInternal(&index, outValue, key);
    }

    /**
    * @brief        キーを削除します。
    *
    * @param[in]    key                     キー
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess                   処理が正常に終了しました。
    * @retval       ResultNotInitialized            初期化されていません。
    * @retval       ResultDatabaseKeyNotFound       key に一致するエントリーがありません。
    * @retval       ResultInvalidOffset             内部で不明な問題が発生しました。
    * @retval       ResultInvalidOffset             内部データに問題があります。
    * @retval       ResultDatabaseCorrupted         内部データに問題があります。
    * @retval       上記以外                        ストレージでのデータ読み書きに失敗しました。
    *
    * @pre          マウントしている。
    *
    * @details      キーを削除します。
    */
    Result Remove(const Key& key) NN_NOEXCEPT
    {
        Index index;
        Index indexPrevious;
        StorageElement storageElement;

        // 見つからない、もしくは I/O エラーが発生するとエラーリザルト。
        NN_RESULT_DO(FindInternal(&index, &indexPrevious, &storageElement, key));

        // 前後のエントリーを接続させて自身を切り離します。
        NN_RESULT_DO(LinkEntry(nullptr, indexPrevious, storageElement.next));

        // この要素をフリーリストに接続します。
        return FreeEntry(index);
    }

    /**
    * @brief        キーと値の列挙を開始します。
    *
    * @param[out]   outIter                         イテレータ
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess                   処理が正常に終了しました。
    * @retval       ResultNotInitialized            初期化されていません。
    * @retval       ResultInvalidOffset             内部で不明な問題が発生しました。
    * @retval       ResultInvalidOffset             内部データに問題があります。
    * @retval       ResultDatabaseCorrupted         内部データに問題があります。
    * @retval       上記以外                        ストレージからのデータ読み込みに失敗しました。
    *
    * @pre          outIter != nullptr
    * @pre          マウントしている。
    *
    * @details      キーと値の列挙を開始します。
    */
    Result FindOpen(FindIndex* outIter) const NN_NOEXCEPT
    {
        StorageElement storageElement;

        NN_SDK_REQUIRES_NOT_NULL(outIter);

        // 検索インデックスを初期化します。
        NN_RESULT_DO(ReadKeyValue(&storageElement, AllocationTableStorageHeadEntry));

        outIter->index = storageElement.next;

        NN_RESULT_SUCCESS;
    }

    /**
    * @brief        キーと値をイテレートします。
    *
    * @param[out]   outKey              キー
    * @param[out]   outValue            値
    * @param[out]   outFinished         イテレーションが終了したか
    * @param[in]    iter                イテレータ
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess                   正常に処理が終了しました。
    * @retval       ResultNotInitialized            初期化されていません。
    * @retval       ResultInvalidOffset             iter が範囲外を指しています。
    * @retval       ResultInvalidOffset             内部で不明な問題が発生しました。
    * @retval       ResultDatabaseCorrupted         内部データに問題があります。
    * @retval       上記以外                        ストレージからのデータ読み込みに失敗しました。
    *
    * @pre          outKey != nullptr
    * @pre          outValue != nullptr
    * @pre          iter != nullptr
    *
    * @details      キーと値をイテレートします。
    */
    Result FindNext(
               Key* outKey,
               Value* outValue,
               bool* outFinished,
               FindIndex* iter
           ) const NN_NOEXCEPT
    {
        StorageElement storageElement;

        NN_SDK_REQUIRES_NOT_NULL(outKey);
        NN_SDK_REQUIRES_NOT_NULL(outValue);
        NN_SDK_REQUIRES_NOT_NULL(iter);

        // KeyValueリストを探索します。
        if( iter->index == AllocationTableStorageFreeEntry )
        {
            *outFinished = true;
            NN_RESULT_SUCCESS;
        }

        // このエントリーを返します。
        NN_RESULT_DO(ReadKeyValue(&storageElement, iter->index));

        // 読み込んだ内容を検証します。
        Index count;
        NN_RESULT_DO(GetMaxCount(&count));
        if( storageElement.next >= count )
        {
            NN_SDK_ASSERT(false);
            return nn::fs::ResultInvalidKeyValueListElementIndex();
        }

        // 次に進めておきます。
        // リスト終端時は AllocationTableStorageFreeEntry が入ります。
        iter->index = storageElement.next;

        // キーと値を返します。
        *outKey = storageElement.key;
        *outValue = storageElement.value;
        *outFinished = false;

        NN_RESULT_SUCCESS;
    }

    /**
    * @brief        キーのみを書き換えます。
    *
    * @param[in]    keyTo               新しいキー
    * @param[in]    keyFrom             以前のキー
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess                   正常にキーを書き換えました。
    * @retval       ResultNotInitialized            初期化されていません。
    * @retval       ResultDatabaseKeyNotFound       keyFrom に一致するエントリーがありません。
    * @retval       ResultInvalidOffset             内部で不明な問題が発生しました。
    * @retval       ResultInvalidOffset             内部データに問題があります。
    * @retval       ResultDatabaseCorrupted         内部データに問題があります。
    * @retval       上記以外                        ストレージでのデータ読み書きに失敗しました。
    *
    * @pre          keyFrom != nullptr
    *
    * @details      キーのみを書き換えます。
    */
    Result ModifyKey(const Key& keyTo, const Key& keyFrom) NN_NOEXCEPT
    {
        Index index;
        Index indexPrevious;
        StorageElement storageElement;

        // from 側のキーを求めます。
        NN_RESULT_DO(FindInternal(&index, &indexPrevious, &storageElement, keyFrom));

        // キーを修正します。
        storageElement.key = keyTo;

        // エントリーに書き直します。
        return WriteKeyValue(&storageElement, index);
    }

    /**
    * @brief        ストレージのバッファリングを有効化します。
    *
    * @return       関数の処理結果を返します。
    *
    * @pre          初期化済み
    */
    Result EnableCacheBuffer() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(m_pKeyValueStorage);
        static const auto CacheCount = 32;
        return m_pKeyValueStorage->EnableCacheBuffer(CacheCount);
    }

    /**
    * @brief        ストレージのバッファリングを無効化します。
    *
    * @pre          初期化済み
    */
    void DisableCacheBuffer() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(m_pKeyValueStorage);
        m_pKeyValueStorage->DisableCacheBuffer();
    }

protected:
    /**
    * @brief        キーと値を追加、または更新します。
    *
    * @param[out]   outIndex            追加されたインデックス
    * @param[in]    key                 キー
    * @param[in]    value               値
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess                   キーと値を正常に追加、または更新できました。
    * @retval       ResultNotInitialized            初期化されていません。
    * @retval       ResultAllocationTableFull       アロケーションテーブルに空きがありません。
    * @retval       ResultInvalidOffset             内部で不明な問題が発生しました。
    * @retval       ResultInvalidOffset             内部データに問題があります。
    * @retval       ResultDatabaseCorrupted         内部データに問題があります。
    * @retval       上記以外                        ストレージでのデータ読み書きに失敗しました。
    *
    * @pre          outIndex != nullptr
    *
    * @details      同名のキーが既に存在する場合、値を更新します。
    */
    Result AddInternal(
               Index* outIndex,
               const Key& key,
               const Value& value
           ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(outIndex);
        Index index;
        Index indexPrevious;
        StorageElement storageElement;

        Result result = FindInternal(
                            &index,
                            &indexPrevious,
                            &storageElement,
                            key
                        );
        if( result.IsFailure() )
        {
            // 見つかりませんでした。
            if( ! nn::fs::ResultDatabaseKeyNotFound::Includes(result) )
            {
                // キーが見つからない、以外のコードは
                // ファイルシステムで発生しています。
                return result;
            }

            // キーが見つからない場合
            // フリーリストから要素を取得し
            // 新規にキーを追加します。
            NN_RESULT_DO(AllocateEntry(&index));

            Index indexNext;
            NN_RESULT_DO(LinkEntry(&indexNext, AllocationTableStorageHeadEntry, index));

            storageElement.key = key;
            storageElement.next = indexNext;
        }

        // キーが登録された or されているので、値部を上書きします。
        *outIndex = index;
        storageElement.value = value;
        return WriteKeyValue(&storageElement, index);
    }

    /**
    * @brief        キーに対する値を取得します。
    *
    * @param[out]   outIndex                格納されているストレージインデックス
    * @param[out]   outValue                値
    * @param[in]    key                     キー
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess                   処理が正常に終了しました。
    * @retval       ResultNotInitialized            初期化されていません。
    * @retval       ResultDatabaseKeyNotFound       key に一致するエントリーがありません。
    * @retval       ResultInvalidOffset             内部データに問題があります。
    * @retval       ResultDatabaseCorrupted         内部データに問題があります。
    * @retval       上記以外                        ストレージからのデータ読み込みに失敗しました。
    *
    * @pre          マウントしている。
    * @pre          outIndex != nullptr
    * @pre          outValue != nullptr
    *
    * @details      キーに対する値を取得します。
    */
    Result GetInternal(
               Index* outIndex,
               Value* outValue,
               const Key& key
           ) const NN_NOEXCEPT
    {
        Index index;
        Index indexPrevious;
        StorageElement storageElement;

        NN_SDK_REQUIRES_NOT_NULL(outIndex);
        NN_SDK_REQUIRES_NOT_NULL(outValue);

        NN_RESULT_DO(FindInternal(&index, &indexPrevious, &storageElement, key));

        *outIndex = index;
        *outValue = storageElement.value;

        NN_RESULT_SUCCESS;
    }

    /**
    * @brief        インデックスからエントリー(キーと値)を取得します。
    *
    * @param[out]   outKey              キー
    * @param[out]   outValue            値
    * @param[in]    index               インデックス
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess                   正常にエントリーが取得できました。
    * @retval       ResultNotInitialized            初期化されていません。
    * @retval       ResultInvalidOffset             index が範囲外です。
    * @retval       ResultInvalidOffset             内部で不明な問題が発生しました。
    * @retval       ResultDatabaseCorrupted         内部データに問題があります。
    * @retval       上記以外                        ストレージからのデータ読み込みに失敗しました。
    *
    * @pre          マウントしている。
    * @pre          outKey != nullptr
    * @pre          outValue != nullptr
    *
    * @details      インデックスからエントリー(キーと値)を取得します。
    */
    Result GetByIndex(Key* outKey, Value* outValue, Index index) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(outKey);
        NN_SDK_REQUIRES_NOT_NULL(outValue);

        StorageElement storageElement;
        NN_RESULT_DO(ReadKeyValue(&storageElement, index));

        // 読み込んだ内容を検証します。
        Index count;
        NN_RESULT_DO(GetMaxCount(&count));
        if( storageElement.next >= count )
        {
            NN_SDK_ASSERT(false);
            return nn::fs::ResultInvalidKeyValueListElementIndex();
        }

        *outKey = storageElement.key;
        *outValue = storageElement.value;

        NN_RESULT_SUCCESS;
    }

    /**
    * @brief        インデックスを用いてバリューを更新します。
    *
    * @param[in]    index               インデックス
    * @param[in]    value               値
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess                   正常に更新できました。
    * @retval       ResultNotInitialized            初期化されていません。
    * @retval       ResultInvalidOffset             index が範囲外です。
    * @retval       ResultInvalidOffset             内部で不明な問題が発生しました。
    * @retval       ResultDatabaseCorrupted         内部データに問題があります。
    * @retval       上記以外                        ストレージでの読み書きに失敗しました。
    *
    * @details      インデックスを用いてバリューを更新します。
    */
    Result SetByIndex(Index index, const Value &value) const NN_NOEXCEPT
    {
        StorageElement storageElement;
        NN_RESULT_DO(ReadKeyValue(&storageElement, index));

        // 読み込んだ内容を検証します。
        Index count;
        NN_RESULT_DO(GetMaxCount(&count));
        if( storageElement.next >= count )
        {
            NN_SDK_ASSERT(false);
            return nn::fs::ResultInvalidKeyValueListElementIndex();
        }

        storageElement.value = value;
        return WriteKeyValue(&storageElement, index);
    }

private:
    /**
    * @brief        フォーマットの管理情報です。
    *
    * @details      先頭の StorageElement を管理領域として扱うためのクラスです。
    */
    struct FormatInformation
    {
        Index indexNotFormatted;    //! フォーマットされていない領域の先頭
        Index indexLimitation;      //! 全ての領域の末端

        /**
        * @brief        FormatInformation を取得します。
        *
        * @details      強制的にキャストします。
        */
        static FormatInformation* GetFromStorageElement(
                                      StorageElement* storageElement
                                  ) NN_NOEXCEPT
        {
            return reinterpret_cast<FormatInformation*>(storageElement);
        }
    };
    NN_STATIC_ASSERT(sizeof(FormatInformation) <= sizeof(Key) + sizeof(Value));
    NN_STATIC_ASSERT(std::is_pod<FormatInformation>::value);

private:
    //! キーバリューストレージの先頭はフリーリストの先頭として扱います。
    static const Index AllocationTableStorageFreeEntry = 0;

    //! キーバリューストレージの2番目はリストの先頭として扱います。
    static const Index AllocationTableStorageHeadEntry = 1;

    //! キーバリューストレージの予約領域
    //! (現在はフリーリストと先頭用に 2 確保)
    static const size_t AllocationTableStorageReservedCount = 2;

    //! エントリーが確保できなくなった時に確保するアロケーションテーブル上のブロック数
    static const size_t AllocationTableStorageAllocateBlockCount = 1;

private:
    /**
    * @brief        キーストレージから読み込みます。
    *
    * @param[out]   outElement          キーバリューストレージに書かれていた値
    * @param[in]    index               インデックス
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess                   処理が正常に終了しました。
    * @retval       ResultNotInitialized            初期化されていません。
    * @retval       ResultInvalidOffset             index が範囲外です。
    * @retval       ResultInvalidOffset             内部で不明な問題が発生しました。
    * @retval       ResultDatabaseCorrupted         内部データに問題があります。
    * @retval       上記以外                        ストレージからのデータ読み込みに失敗しました。
    *
    * @pre          マウントしている。
    * @pre          outElement != nullptr
    *
    * @details      キーストレージから読み込みます。
    */
    inline Result ReadKeyValue(
                      StorageElement* outElement,
                      Index index
                  ) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(outElement);
        NN_SDK_REQUIRES_NOT_NULL(m_pKeyValueStorage);

        // 入力内容を検証します。
        Index count;
        NN_RESULT_DO(GetMaxCount(&count));

        if( (index >= count) && (count != AllocationTableStorageHeadEntry) )
        {
            NN_SDK_ASSERT(false);
            return nn::fs::ResultInvalidOffset();
        }
        return ReadKeyValueImpl(outElement, index);
    }

    /**
    * @brief        ReadKeyValue 内で ReadKeyValue を呼ぶための関数。
    *
    * @param[out]   outElement          読み込み先
    * @param[in]    index               読み込みたいエントリのインデックス
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess                   処理が正常に終了しました。
    * @retval       ResultNotInitialized            初期化されていません。
    * @retval       ResultInvalidOffset             index が範囲外です。
    * @retval       ResultDatabaseCorrupted         内部データに問題があります。
    * @retval       上記以外                        ストレージからのデータ読み込みに失敗しました。
    *
    * @pre          マウントしている。
    * @pre          outElement != nullptr
    *
    * @details      ReadKeyValue 内で ReadKeyValue を呼ぶための関数。
    */
    inline Result ReadKeyValueImpl(
                      StorageElement* outElement,
                      Index index
                  ) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(outElement);
        NN_SDK_REQUIRES_NOT_NULL(m_pKeyValueStorage);
        int64_t offset = index * sizeof(StorageElement);
        return m_pKeyValueStorage->Read(offset, outElement, sizeof(StorageElement));
    }

    /**
    * @brief        キーストレージに書き込みます。
    *
    * @param[out]   outElement          キーバリューストレージに書き込む値
    * @param[in]    index               インデックス
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess                   正常に書き込めました。
    * @retval       ResultNotInitialized            初期化されていません。
    * @retval       ResultInvalidOffset             index が範囲外です。
    * @retval       ResultInvalidOffset             内部で不明な問題が発生しました。
    * @retval       ResultDatabaseCorrupted         内部データに問題があります。
    * @retval       上記以外                        ストレージでのデータ読み書きに失敗しました。
    *
    * @pre          マウントしている。
    * @pre          outElement != nullptr
    *
    * @details      キーストレージに書き込みます。
    */
    inline Result WriteKeyValue(
                      const StorageElement* outElement,
                      Index index
                  ) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(outElement);
        NN_SDK_REQUIRES_NOT_NULL(m_pKeyValueStorage);

        // 入力内容を検証します。
        Index count;
        NN_RESULT_DO(GetMaxCount(&count));

        if( index >= count )
        {
            return nn::fs::ResultInvalidOffset();
        }
        int64_t offset = index * sizeof(StorageElement);
        return m_pKeyValueStorage->Write(offset, outElement, sizeof(StorageElement));
    }

    /**
    * @brief        キーを指定し、対応する値セットとインデックス番号を取得します。(内部用)
    *
    * @param[out]   outIndex                得られたインデックス
    * @param[out]   outPreviousIndex        得られたインデックスを指し示すインデックス
    * @param[out]   outElement              得られた値
    * @param[in]    key                     キー
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess                   処理が正常に終了しました。
    * @retval       ResultNotInitialized            初期化されていません。
    * @retval       ResultDatabaseKeyNotFound       key に一致するエントリーがありません。
    * @retval       ResultInvalidOffset             内部データに問題があります。
    * @retval       ResultDatabaseCorrupted         内部データに問題があります。
    * @retval       上記以外                        ストレージからのデータ読み込みに失敗しました。
    *
    * @pre          マウントしている。
    * @pre          outIndex != nullptr
    * @pre          outPreviousIndex != nullptr
    * @pre          outElement != nullptr
    *
    * @details      キーを指定し、対応する値セットとインデックス番号を取得します。(内部用)
    */
    Result FindInternal(
               Index* outIndex,
               Index* outPreviousIndex,
               StorageElement* outElement,
               const Key& key
           ) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(outIndex);
        NN_SDK_REQUIRES_NOT_NULL(outPreviousIndex);
        NN_SDK_REQUIRES_NOT_NULL(outElement);

        *outIndex = 0;
        *outPreviousIndex = AllocationTableStorageHeadEntry;

        Index indexTop;
        StorageElement storageElementHead;
        NN_RESULT_DO(ReadKeyValue(&storageElementHead, AllocationTableStorageHeadEntry));

        indexTop = storageElementHead.next;

        // 読み込んだ内容を検証します。
        Index count;
        NN_RESULT_DO(GetMaxCount(&count));
        if( indexTop >= count )
        {
            NN_SDK_ASSERT(false);
            return nn::fs::ResultInvalidKeyValueListElementIndex();
        }

        // 空だった場合
        if( indexTop == AllocationTableStorageFreeEntry )
        {
            return nn::fs::ResultDatabaseKeyNotFound();
        }

        // アローケーションリンクを辿り、エントリーを探します。
        Index index = indexTop;
        for( ; ; )
        {
            // 一つエントリーを得て比較します。
            NN_RESULT_DO(ReadKeyValue(outElement, index));

            // キー同士が一致すれば検索終了です。
            if( key == outElement->key )
            {
                *outIndex = index;
                break;
            }
            // 次のインデクスを探します。
            *outPreviousIndex = index;
            index = outElement->next;
            if( AllocationTableStorageFreeEntry == index )
            {
                // 一致したキーが見つかりません。
                return nn::fs::ResultDatabaseKeyNotFound();
            }
        }
        NN_RESULT_SUCCESS;
    }

    /**
    * @brief        フリーエントリーからエントリーを確保します。
    *
    * @param[out]   outEntry            得られたエントリー位置
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess                   正常に確保できました。
    * @retval       ResultNotInitialized            初期化されていません。
    * @retval       ResultAllocationTableFull       アロケーションテーブルに空きがありません。
    * @retval       ResultInvalidOffset             内部で不明な問題が発生しました。
    * @retval       ResultInvalidOffset             内部データに問題があります。
    * @retval       ResultDatabaseCorrupted         内部データに問題があります。
    * @retval       上記以外                        ストレージでのデータ読み書きに失敗しました。
    *
    * @pre          マウントしている。
    * @pre          outEntry != nullptr
    *
    * @details      フリーエントリーに空きが無ければストレージを拡張します。
    */
    Result AllocateEntry(Index* outEntry) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(outEntry);

        // フリーエントリーを取得します。
        StorageElement storageElement;
        NN_RESULT_DO(ReadKeyValue(&storageElement, AllocationTableStorageFreeEntry));

        // 既にフリー領域は全て割り当てされています。
        if( storageElement.next == AllocationTableStorageFreeEntry )
        {
            // クイックフォーマット情報からフリーエントリーを取得します
            FormatInformation* pFormatInfo = FormatInformation::GetFromStorageElement(&storageElement);
            bool isFormated = pFormatInfo->indexLimitation != AllocationTableStorageFreeEntry;
            bool isEntryFull = pFormatInfo->indexLimitation <= pFormatInfo->indexNotFormatted;
            if( (! isFormated) || isEntryFull )
            {
                // 割り当て可能領域が無いなら、AllocationTableStorage を拡張する
                NN_RESULT_DO(
                    m_pKeyValueStorage->AllocateBlock(
                        AllocationTableStorageAllocateBlockCount
                    )
                );

                // 拡張できたら情報を更新
                int64_t allocSize;
                NN_RESULT_DO(m_pKeyValueStorage->GetSize(&allocSize));
                uint32_t entryCount = QueryEntryCount(allocSize);
                uint32_t newLimitation = entryCount + AllocationTableStorageReservedCount;
                pFormatInfo->indexLimitation = newLimitation;
            }
            NN_SDK_ASSERT_LESS(pFormatInfo->indexNotFormatted, pFormatInfo->indexLimitation);

            // 割り当て可能領域を見つけました。
            // この領域を切り出して使用します。
            Index indexFree = pFormatInfo->indexNotFormatted;
            pFormatInfo->indexNotFormatted++;

            NN_RESULT_DO(WriteKeyValue(&storageElement, AllocationTableStorageFreeEntry));

            // 得られたエントリーを返します。
            *outEntry = indexFree;
        }
        else
        {
            StorageElement nextElement;
            // 次の領域へのインデックスを取得します。
            Index indexFree = storageElement.next;
            NN_RESULT_DO(ReadKeyValue(&nextElement, indexFree));

            // フリー領域のリンクを繋ぎなおします。
            storageElement.next = nextElement.next;
            NN_RESULT_DO(WriteKeyValue(&storageElement, AllocationTableStorageFreeEntry));

            // 得られたエントリーを返します。
            *outEntry = indexFree;
        }
        NN_RESULT_SUCCESS;
    }

    /**
    * @brief        リストの接続情報を更新します。
    *
    * @param[out]   oldNext             もともとの次ブロックのインデックス
    * @param[in]    indexPrevious       更新後の前ブロックのインデックス
    * @param[in]    indexNext           更新後の次ブロックのインデックス
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess                   処理が正常に終了しました。
    * @retval       ResultNotInitialized            初期化されていません。
    * @retval       ResultInvalidOffset             indexPrevious が範囲外です。
    * @retval       ResultInvalidOffset             内部で不明な問題が発生しました。
    * @retval       ResultDatabaseCorrupted         内部データに問題があります。
    * @retval       上記以外                        ストレージでのデータ読み書きに失敗しました。
    *
    * @pre          マウントしている。
    * @pre          oldNext は nullptr でも良い。
    *
    * @details      リストの接続情報を更新します。
    */
    Result LinkEntry(Index* oldNext, Index indexPrevious, Index indexNext) NN_NOEXCEPT
    {
        StorageElement storageElement;

        NN_RESULT_DO(ReadKeyValue(&storageElement, indexPrevious));

        // 自分を指し示していた要素(自分の一つ手前)を更新します。
        if( oldNext != nullptr )
        {
            *oldNext = storageElement.next;
        }
        storageElement.next = indexNext;

        return WriteKeyValue(&storageElement, indexPrevious);
    }

    /**
    * @brief        割り当てたブロックを解放します。
    *
    * @param[in]    entry               フリーエントリーに追加するエントリー位置
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess                   正常にフリーエントリーに戻しました。
    * @retval       ResultNotInitialized            初期化されていません。
    * @retval       ResultInvalidOffset             entry が範囲外です。
    * @retval       ResultInvalidOffset             内部で不明な問題が発生しました。
    * @retval       ResultDatabaseCorrupted         内部データに問題があります。
    * @retval       上記以外                        ストレージでのデータ読み書きに失敗しました。
    *
    * @pre          マウントしている。
    *
    * @details      割り当てたブロックを解放します。
    *               解放したブロックはフリーエントリーへ戻されます。
    */
    Result FreeEntry(Index entry) NN_NOEXCEPT
    {
        StorageElement storageElementFree;
        StorageElement storageElement;

        // フリーエントリーを取得します。
        NN_RESULT_DO(ReadKeyValue(&storageElementFree, AllocationTableStorageFreeEntry));

        // フリー領域のリンクを繋ぎなおします。
        NN_RESULT_DO(ReadKeyValue(&storageElement, entry));
        storageElement.next = storageElementFree.next;
        NN_RESULT_DO(WriteKeyValue(&storageElement, entry));

        // フリーエントリーを更新します。
        storageElementFree.next = entry;
        NN_RESULT_DO(WriteKeyValue(&storageElementFree, AllocationTableStorageFreeEntry));

        NN_RESULT_SUCCESS;
    }

private:
    BufferedAllocationTableStorage* m_pKeyValueStorage;     //! キーバリューストレージ

private:
    // テスト用に private を公開します。
    friend class KeyValueListTemplateTest;
};

template <typename TKey, typename TValue>
const typename KeyValueListTemplate<TKey, TValue>::Index KeyValueListTemplate<TKey, TValue>::AllocationTableStorageFreeEntry;

template <typename TKey, typename TValue>
const typename KeyValueListTemplate<TKey, TValue>::Index KeyValueListTemplate<TKey, TValue>::AllocationTableStorageHeadEntry;

template <typename TKey, typename TValue>
const size_t KeyValueListTemplate<TKey, TValue>::AllocationTableStorageReservedCount;

template <typename TKey, typename TValue>
const size_t KeyValueListTemplate<TKey, TValue>::AllocationTableStorageAllocateBlockCount;


}}}

