﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_BitTypes.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/fssystem/dbm/fs_DbmParameters.h>
#include <nn/fssystem/dbm/fs_DbmPathTool.h>
#include <nn/fssystem/dbm/fs_HierarchicalFileTableTemplate.h>
#include <nn/fssystem/dbm/fs_KeyValueListTemplate.h>

namespace nn { namespace fssystem { namespace dbm {

template <
    typename TDirectoryName,
    typename TFileName,
    typename TDirectoryInfo,
    typename TFileInfo>
const typename HierarchicalFileTableTemplate<
    TDirectoryName,
    TFileName,
    TDirectoryInfo,
    TFileInfo>::StorageIndex HierarchicalFileTableTemplate<
        TDirectoryName,
        TFileName,
        TDirectoryInfo,
        TFileInfo>::IndexNone;

template <
    typename TDirectoryName,
    typename TFileName,
    typename TDirectoryInfo,
    typename TFileInfo>
const uint32_t HierarchicalFileTableTemplate<
    TDirectoryName,
    TFileName,
    TDirectoryInfo,
    TFileInfo>::DirectoryCountSystemReserved;

template <
    typename TDirectoryName,
    typename TFileName,
    typename TDirectoryInfo,
    typename TFileInfo>
const uint32_t HierarchicalFileTableTemplate<
    TDirectoryName,
    TFileName,
    TDirectoryInfo,
    TFileInfo>::FileCountSystemReserved;

/**
* @brief        エントリー数から必要なストレージのサイズを求めます。
*
* @param[in]    countDirectoryEntry ディレクトリエントリー数
*
* @return       必要なストレージのサイズ。
*
* @details      エントリー数から必要なストレージのサイズを求めます。
*               - "エントリー数"は、作成可能なディレクトリ数、ファイル数です。
*               - エントリー数以上にディレクトリやファイルを作成しようとすると、
*                 ストレージの割り当て領域が増やされ、最大エントリー数が増えます。
*/
template <
    typename TDirectoryName,
    typename TFileName,
    typename TDirectoryInfo,
    typename TFileInfo
>
uint32_t HierarchicalFileTableTemplate<
             TDirectoryName,
             TFileName,
             TDirectoryInfo,
             TFileInfo
         >::QueryDirectoryEntryStorageSize(
             uint32_t countDirectoryEntry
         ) NN_NOEXCEPT
{
    uint32_t countAllEntry = countDirectoryEntry + DirectoryCountSystemReserved;
    return DirectoryEntryList::QueryStorageSize(countAllEntry);
}

/**
* @brief        エントリー数から必要なストレージのサイズを求めます。
*
* @param[in]    countFileEntry      ファイルエントリー数
*
* @return       必要なストレージのサイズ。
*
* @details      エントリー数から必要なストレージのサイズを求めます。
*               - "エントリー数"は、作成可能なディレクトリ数、ファイル数です。
*               - エントリー数以上にディレクトリやファイルを作成しようとすると、
*                 ストレージの割り当て領域が増やされ、最大エントリー数が増えます。
*/
template <
    typename TDirectoryName,
    typename TFileName,
    typename TDirectoryInfo,
    typename TFileInfo
>
uint32_t HierarchicalFileTableTemplate<
             TDirectoryName,
             TFileName,
             TDirectoryInfo,
             TFileInfo
         >::QueryFileEntryStorageSize(
             uint32_t countFileEntry
         ) NN_NOEXCEPT
{
    uint32_t countAllEntry = countFileEntry + FileCountSystemReserved;
    return FileEntryList::QueryStorageSize(countAllEntry);
}

/**
* @brief        ファイル管理テーブルをフォーマットします。
*
* @param[in]    pDirectoryEntry     ディレクトリエントリー用ストレージ
* @param[in]    pFileEntry          ファイルエントリー用ストレージ
*
* @return       関数の処理結果を返します。
* @retval       ResultSuccess                   正常に初期化できました。
* @retval       ResultNotInitialized            pDirectoryEntry が初期化されていません。
* @retval       ResultNotInitialized            pFileEntry が初期化されていません。
* @retval       ResultInvalidOffset             pDirectoryEntry が小さすぎます。
* @retval       ResultInvalidOffset             pFileEntry が小さすぎます。
* @retval       ResultAllocationTableFull       アロケーションテーブルに空きがありません。
* @retval       ResultInvalidOffset             内部で不明な問題が発生しました。
* @retval       ResultInvalidOffset             内部データに問題があります。
* @retval       ResultDatabaseCorrupted         内部データに問題があります。
*
* @pre          pDirectoryEntry が NULL ではない。
* @pre          pFileEntry が NULL ではない。
*
* @details      ファイル管理テーブルをフォーマットします。
*               IStorage を初期化します。使用する前に一度呼び出します。
*/
template <
    typename TDirectoryName,
    typename TFileName,
    typename TDirectoryInfo,
    typename TFileInfo
>
Result HierarchicalFileTableTemplate<
           TDirectoryName,
           TFileName,
           TDirectoryInfo,
           TFileInfo
       >::Format(
           BufferedAllocationTableStorage* pDirectoryEntry,
           BufferedAllocationTableStorage* pFileEntry
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDirectoryEntry);
    NN_SDK_REQUIRES_NOT_NULL(pFileEntry);

    // ディレクトリエントリー用テーブルをフォーマットします。
    NN_RESULT_DO(DirectoryEntryList::Format(pDirectoryEntry));

    // ファイルエントリー用テーブルをフォーマットします。
    NN_RESULT_DO(FileEntryList::Format(pFileEntry));

    // ルートディレクトリ作成のために一時的にマウントします。
    DirectoryEntryList tableDirectory;
    tableDirectory.Initialize(pDirectoryEntry);

    // ルートディレクトリエントリーを作成します。
    StorageIndex rootIndex;
    DirectoryKey rootKey;
    rootKey.parentDir = IndexNone;
    std::memset(&rootKey.name, 0, sizeof(TDirectoryName));
    DirectoryEntry rootEntry;
    rootEntry.indexNext = IndexNone;
    rootEntry.indexDirectory = IndexNone;
    rootEntry.indexFile = IndexNone;
    std::memset(&rootEntry.info, 0, sizeof(TDirectoryInfo));
    NN_RESULT_DO(tableDirectory.Add(&rootIndex, rootKey, rootEntry));

    // アンマウントします。
    tableDirectory.Finalize();

    NN_RESULT_SUCCESS;
}

/**
* @brief        コンストラクタです。
*
* @details      コンストラクタです。
*/
template <
    typename TDirectoryName,
    typename TFileName,
    typename TDirectoryInfo,
    typename TFileInfo
>
HierarchicalFileTableTemplate<
    TDirectoryName,
    TFileName,
    TDirectoryInfo,
    TFileInfo
>::HierarchicalFileTableTemplate() NN_NOEXCEPT
{
}

/**
* @brief        ストレージをマウントします。
*
* @param[in]    pDirectoryEntry     ディレクトリエントリー用ストレージ
* @param[in]    pFileEntry          ファイルエントリー用ストレージ
*
* @pre          pDirectoryEntry が NULL ではない。
* @pre          pFileEntry が NULL ではない。
*
* @details      ストレージをマウントします。
*/
template <
    typename TDirectoryName,
    typename TFileName,
    typename TDirectoryInfo,
    typename TFileInfo
>
void HierarchicalFileTableTemplate<
         TDirectoryName,
         TFileName,
         TDirectoryInfo,
         TFileInfo
     >::Initialize(
            BufferedAllocationTableStorage* pDirectoryEntry,
            BufferedAllocationTableStorage* pFileEntry
        ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDirectoryEntry);
    NN_SDK_REQUIRES_NOT_NULL(pFileEntry);

    // ディレクトリ管理用ストレージをマウントします。
    m_TableDirectory.Initialize(pDirectoryEntry);

    // ファイル管理用ストレージをマウントします。
    m_TableFile.Initialize(pFileEntry);
}

/**
* @brief        ストレージをアンマウントします。
*
* @details      ストレージをアンマウントします。
*/
template <
    typename TDirectoryName,
    typename TFileName,
    typename TDirectoryInfo,
    typename TFileInfo
>
void HierarchicalFileTableTemplate<
         TDirectoryName,
         TFileName,
         TDirectoryInfo,
         TFileInfo
     >::Finalize() NN_NOEXCEPT
{
    // ディレクトリ管理用ストレージをアンマウントします。
    m_TableDirectory.Finalize();

    // ファイル管理用ストレージをアンマウントします。
    m_TableFile.Finalize();
}

/**
* @brief        ディレクトリを作成します。
*
* @param[in]    pFullPath                   ディレクトリパス
* @param[in]    dirInfo                     ディレクトリ情報
*
* @return       関数の処理結果を返します。
* @retval       ResultSuccess                   正常に作成できました。
* @retval       ResultNotInitialized            初期化されていません。
* @retval       ResultInvalidPathFormat         pFullPath がルートディレクトリから始まっていません。
* @retval       ResultTooLongPath               ディレクトリ名が長すぎます。
* @retval       ResultDirectoryUnobtainable     ルートディレクトリの親ディレクトリは取得できません。
* @retval       ResultIncompatiblePath          pFullPath がファイルでした。
* @retval       ResultAlreadyExists             同名のファイルまたはディレクトリが存在します。
* @retval       ResultAllocationTableFull       アロケーションテーブルに空きがありません。
* @retval       ResultInvalidOffset             内部データに問題があります。
* @retval       ResultDatabaseCorrupted         内部データに問題があります。
* @retval       上記以外                        ストレージでの読み書きに失敗しました。
*
* @pre          pFullPath が NULL ではない。
*
* @details      ディレクトリを作成します。
*/
template <
    typename TDirectoryName,
    typename TFileName,
    typename TDirectoryInfo,
    typename TFileInfo
>
Result HierarchicalFileTableTemplate<
           TDirectoryName,
           TFileName,
           TDirectoryInfo,
           TFileInfo
       >::CreateDirectory(
           const PathChar* pFullPath,
           const TDirectoryInfo& dirInfo
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pFullPath);

    DirectoryKey parentKey;
    DirectoryEntry parentEntry;
    DirectoryKey newKey;

    CacheBufferEnabler enableCacheBuffer(this);

    // 親ディレクトリのエントリーを探索し、新規ディレクトリ用のキーを生成します。
    NN_RESULT_DO(
        FindDirectoryRecursive(
            &parentKey,
            &parentEntry,
            &newKey,
            pFullPath
        )
    );

    {
        // 同名のファイル、ディレクトリがすでに存在しているかどうかチェックします。
        // 存在した場合は、すでにエントリーが存在していることを結果として返します。
        bool isFileTmp;
        NN_RESULT_DO(
            CheckSameEntryExists(
                &isFileTmp,
                newKey.parentDir,
                reinterpret_cast<PathChar*>(&newKey.name),
                PathTool::GetDirectoryNameLength(&newKey.name),
                nn::fs::ResultAlreadyExists()
            )
        );
    }

    // 新規ディレクトリエントリーを初期化します。
    DirectoryEntry newEntry;
    newEntry.indexNext = IndexNone;
    newEntry.indexDirectory = IndexNone;
    newEntry.indexFile = IndexNone;
    newEntry.info = dirInfo;

    // 新規ディレクトリエントリーをテーブルに登録します。
    StorageIndex newIndex = IndexNone;
    NN_RESULT_DO(m_TableDirectory.Add(&newIndex, newKey, newEntry));

    // 新規ディレクトリエントリーを親ディレクトリにおける子ディレクトリリストの先頭として追加します。
    newEntry.indexNext = parentEntry.indexDirectory;
    parentEntry.indexDirectory = newIndex;

    // 新規ディレクトリエントリーのリンク情報を更新します。
    if( newEntry.indexNext != IndexNone )
    {
        NN_RESULT_DO(m_TableDirectory.SetByIndex(newIndex, newEntry));
    }

    // 親ディレクトリのリンク情報を更新します。
    return m_TableDirectory.SetByIndex(newKey.parentDir, parentEntry);
}

/**
* @brief        ファイルを作成します。
*
* @param[out]   outIndex                    識別 ID
* @param[in]    pFullPath                   ファイルパス
* @param[in]    fileInfo                    ファイル情報
*
* @retval       ResultSuccess                   正常に作成できました。
* @retval       ResultNotInitialized            初期化されていません。
* @retval       ResultAlreadyExists             同名のファイルまたはディレクトリが存在します。
* @retval       ResultInvalidPathFormat         pFullPath がルートディレクトリから始まっていません。
* @retval       ResultTooLongPath               ディレクトリ名が長すぎます。
* @retval       ResultDirectoryUnobtainable     ルートディレクトリの親ディレクトリは取得できません。
* @retval       ResultIncompatiblePath          pFullPath がディレクトリでした。
* @retval       ResultAllocationTableFull       アロケーションテーブルに空きがありません。
* @retval       ResultInvalidOffset             内部データに問題があります。
* @retval       ResultDatabaseCorrupted         内部データに問題があります。
* @retval       上記以外                        ストレージでの読み書きに失敗しました。
*
* @pre          マウントしている。
* @pre          outIndex が NULL ではない。
* @pre          pFullPath が NULL ではない。
*
* @details      ファイルを作成します。
*/
template <
    typename TDirectoryName,
    typename TFileName,
    typename TDirectoryInfo,
    typename TFileInfo
>
Result HierarchicalFileTableTemplate<
           TDirectoryName,
           TFileName,
           TDirectoryInfo,
           TFileInfo
       >::CreateFile(
           StorageIndex* outIndex,
           const PathChar* pFullPath,
           const TFileInfo& fileInfo
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outIndex);
    NN_SDK_REQUIRES_NOT_NULL(pFullPath);

    DirectoryKey parentKey;
    DirectoryEntry parentEntry;
    FileKey newKey;

    CacheBufferEnabler enableCacheBuffer(this);

    // 親ディレクトリのエントリーを探索し、新規ファイル用のキーを生成します。
    NN_RESULT_DO(
        FindFileRecursive(
            &parentKey,
            &parentEntry,
            &newKey,
            pFullPath
        )
    );

    {
        // 同名のファイル、ディレクトリがすでに存在しているかどうかチェックします。
        // 存在した場合は、すでにエントリーが存在していることを結果として返します。
        bool isFileTmp;
        NN_RESULT_DO(
            CheckSameEntryExists(
                &isFileTmp,
                newKey.parentDir,
                reinterpret_cast<PathChar*>(&newKey.name),
                PathTool::GetFileNameLength(&newKey.name),
                nn::fs::ResultAlreadyExists()
            )
        );
    }

    // 新規ファイルエントリーを初期化します。
    FileEntry newEntry;
    newEntry.indexNext = IndexNone;
    newEntry.info = fileInfo;

    // 新規ファイルエントリーをテーブルに登録します。
    StorageIndex newIndex = IndexNone;
    NN_RESULT_DO(m_TableFile.Add(&newIndex, newKey, newEntry));

    // 新規ファイルエントリーを親ディレクトリにおける子ファイルリストの先頭に追加します。
    newEntry.indexNext = parentEntry.indexFile;
    parentEntry.indexFile = newIndex;

    // 新規ファイルエントリーのリンク情報を更新します。
    if( newEntry.indexNext != IndexNone )
    {
        NN_RESULT_DO(m_TableFile.SetByIndex(newIndex, newEntry));
    }

    // 親ディレクトリのリンク情報を更新します。
    NN_RESULT_DO(m_TableDirectory.SetByIndex(newKey.parentDir, parentEntry));

    *outIndex = newIndex;

    NN_RESULT_SUCCESS;
}

/**
* @brief        指定したディレクトリを削除します。
*
* @param[in]    pFullpath                   ディレクトリパス
*
* @return       関数の処理結果を返します。
* @retval       ResultSuccess                   正常に取得できました。
* @retval       ResultNotInitialized            初期化されていません。
* @retval       ResultInvalidPathFormat         pFullPath がルートディレクトリから始まっていません。
* @retval       ResultTooLongPath               ディレクトリ名が長すぎます。
* @retval       ResultDirectoryUnobtainable     ルートディレクトリの親ディレクトリは取得できません。
* @retval       ResultDirectoryUndeletable      ルートディレクトリを削除することはできません。
* @retval       ResultIncompatiblePath          pFullPath がファイルでした。
* @retval       ResultIncompatiblePath          内部データに問題があります。
* @retval       ResultDirectoryNotFound         内部データに問題があります。
* @retval       ResultDatabaseKeyNotFound       内部データに問題があります。
* @retval       ResultInvalidOffset             内部データに問題があります。
* @retval       ResultDatabaseCorrupted         内部データに問題があります。
* @retval       上記以外                        ストレージでのデータ読み書きに失敗しました。
*
* @pre          pFullPath が NULL ではない。
*
* @details      指定したディレクトリを削除します。
*/
template <
    typename TDirectoryName,
    typename TFileName,
    typename TDirectoryInfo,
    typename TFileInfo
>
Result HierarchicalFileTableTemplate<
           TDirectoryName,
           TFileName,
           TDirectoryInfo,
           TFileInfo
       >::DeleteDirectory(const PathChar* pFullPath) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pFullPath);

    // 削除するディレクトリの親ディレクトリエントリーを探索し、
    // 削除するディレクトリ用キーを生成します。
    DirectoryKey parentKey;
    DirectoryEntry parentEntry;
    DirectoryKey deleteKey;

    CacheBufferEnabler enableCacheBuffer(this);

    NN_RESULT_DO(
        FindDirectoryRecursive(
            &parentKey,
            &parentEntry,
            &deleteKey,
            pFullPath
        )
    );

    // ルートディレクトリを削除することはできません。
    if( deleteKey.parentDir == parentKey.parentDir )
    {
        // ファイル操作は拒否されました。
        return nn::fs::ResultDirectoryUndeletable();
    }

    // 削除するディレクトリエントリーを取得します。
    StorageIndex deleteIndex;
    DirectoryEntry deleteEntry;
    NN_RESULT_DO(GetDirectoryEntry(&deleteIndex, &deleteEntry, deleteKey));

    // 削除するディレクトリ以下にファイルが残っていないかどうかチェックします。
    // 残っていた場合には削除できません。
    if( (deleteEntry.indexDirectory != IndexNone) || (deleteEntry.indexFile != IndexNone) )
    {
        // ファイル操作は拒否されました。
        return nn::fs::ResultDirectoryNotEmpty();
    }

    // 親ディレクトリ、兄弟ディレクトリエントリーからのリンクを解除します。
    NN_RESULT_DO(
        RemoveDirectoryLink(
            &parentEntry,
            deleteKey.parentDir,
            &deleteEntry,
            deleteIndex
        )
    );

    // テーブルから削除します。
    NN_RESULT_DO(m_TableDirectory.Remove(deleteKey));

    NN_RESULT_SUCCESS;
}

/**
* @brief        指定したファイルを削除します。
*
* @param[in]    pFullpath                   ファイルパス
*
* @return       関数の処理結果を返します。
* @retval       ResultSuccess                   正常に取得できました。
* @retval       ResultNotInitialized            初期化されていません。
* @retval       ResultTooLongPath               ディレクトリ名が長すぎます。
* @retval       ResultDirectoryUnobtainable     ルートディレクトリの親ディレクトリは取得できません。
* @retval       ResultInvalidPathFormat         pFullPath がルートディレクトリから始まっていません。
* @retval       ResultIncompatiblePath          pFullPath がディレクトリでした。
* @retval       ResultIncompatiblePath          内部データに問題があります。
* @retval       ResultFileNotFound              内部データに問題があります。
* @retval       ResultDatabaseKeyNotFound       内部データに問題があります。
* @retval       ResultInvalidOffset             内部データに問題があります。
* @retval       ResultDatabaseCorrupted         内部データに問題があります。
* @retval       上記以外                        ストレージでのデータ読み書きに失敗しました。
*
* @pre          pFullPath は NULL ではない。
*
* @details      指定したファイルを削除します。
*/
template <
    typename TDirectoryName,
    typename TFileName,
    typename TDirectoryInfo,
    typename TFileInfo
>
Result HierarchicalFileTableTemplate<
           TDirectoryName,
           TFileName,
           TDirectoryInfo,
           TFileInfo
       >::DeleteFile(const PathChar* pFullPath) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pFullPath);

    // 削除するファイルの親ディレクトリエントリーを探索し、
    // 削除するファイル用キーを生成します。
    DirectoryKey parentKey;
    DirectoryEntry parentEntry;
    FileKey deleteKey;

    CacheBufferEnabler enableCacheBuffer(this);

    NN_RESULT_DO(
        FindFileRecursive(
            &parentKey,
            &parentEntry,
            &deleteKey,
            pFullPath
        )
    );

    // 削除するファイルエントリーを取得します。
    FileEntry deleteEntry;
    StorageIndex deleteIndex;
    NN_RESULT_DO(GetFileEntry(&deleteIndex, &deleteEntry, deleteKey));

    // 親ディレクトリ、兄弟ファイルエントリーからのリンクを解除します。
    NN_RESULT_DO(
        RemoveFileLink(
            &parentEntry,
            deleteKey.parentDir,
            &deleteEntry,
            deleteIndex
        )
    );

    // テーブルから削除します。
    NN_RESULT_DO(m_TableFile.Remove(deleteKey));

    NN_RESULT_SUCCESS;
}

/**
* @brief        指定したディレクトリの名前を変更します。
*
* @param[out]   outIsFile                       リネーム先に同名のエントリーが存在したときにファイルであれば true、ディレクトリであれば false
* @param[out]   outIsSameEntry                  変更前と変更後のパスが同じエントリーを指しているかどうか
* @param[out]   outIsParentEntry                変更後のパスが変更前のパスの親ディレクトリかどうか
* @param[in]    pNewFullPath                    変更後のファイルパス
* @param[in]    pOldFullPath                    変更前のファイルパス
*
* @return       関数の処理結果を返します。
* @retval       ResultSuccess                   正常に変更できました。
* @retval       ResultAlreadyExists             変更前後で名前が同じです。
* @retval       ResultNotInitialized            初期化されていません。
* @retval       ResultInvalidPathFormat         パスがルートディレクトリから始まっていません。
* @retval       ResultTooLongPath               ディレクトリ名が長すぎます。
* @retval       ResultDirectoryUnobtainable     ルートディレクトリの親ディレクトリは取得できません。
* @retval       ResultIncompatiblePath          パスがファイルへのパスでした。
* @retval       ResultDirectoryUnrenamable      ルートディレクトリはリネームはできません。
* @retval       ResultNotInitialized            初期化されていません。
* @retval       ResultDirectoryUnrenamable      pNewFullPath と pOldFullPath は親子関係にあります。
* @retval       ResultAlreadyExists             同名のファイルまたはディレクトリが存在します。
* @retval       ResultIncompatiblePath          内部データに問題があります。
* @retval       ResultDirectoryNotFound         内部データに問題があります。
* @retval       ResultDatabaseKeyNotFound       内部データに問題があります。
* @retval       ResultInvalidOffset             内部データに問題があります。
* @retval       ResultDatabaseCorrupted         内部データに問題があります。
* @retval       上記以外                        ストレージでの読み書きに失敗しました。
*
* @pre          引数がすべて NULL ではない。
*
* @details      指定したディレクトリの名前を変更します。
*/
template <
    typename TDirectoryName,
    typename TFileName,
    typename TDirectoryInfo,
    typename TFileInfo
>
Result HierarchicalFileTableTemplate<
           TDirectoryName,
           TFileName,
           TDirectoryInfo,
           TFileInfo
       >::RenameDirectory(
           bool* outIsFile,
           bool* outIsSameEntry,
           bool* outIsParentEntry,
           const PathChar* pNewFullPath,
           const PathChar* pOldFullPath
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outIsFile);
    NN_SDK_REQUIRES_NOT_NULL(outIsSameEntry);
    NN_SDK_REQUIRES_NOT_NULL(outIsParentEntry);
    NN_SDK_REQUIRES_NOT_NULL(pOldFullPath);
    NN_SDK_REQUIRES_NOT_NULL(pNewFullPath);

    *outIsFile = false;
    *outIsSameEntry = false;
    *outIsParentEntry = false;

    DirectoryKey oldParentKey;
    DirectoryEntry oldParentEntry;
    DirectoryKey oldKey;

    StorageIndex oldIndex;
    DirectoryEntry oldEntry;

    DirectoryKey newParentKey;
    DirectoryEntry newParentEntry;
    DirectoryKey newKey;

    CacheBufferEnabler enableCacheBuffer(this);

    {
        // リネーム前の親ディレクトリのエントリーを探索します。
        NN_RESULT_DO(
            FindDirectoryRecursive(
                &oldParentKey,
                &oldParentEntry,
                &oldKey,
                pOldFullPath
            )
        );

        // ルートディレクトリはリネームはできません。
        if( oldKey.parentDir == oldParentKey.parentDir )
        {
            // ファイル操作は拒否されました。
            return nn::fs::ResultDirectoryUnrenamable();
        }

        // リネーム前のエントリー情報を取得します。
        NN_RESULT_DO(GetDirectoryEntry(&oldIndex, &oldEntry, oldKey));

        // リネーム後の親ディレクトリのエントリーを探索します。
        NN_RESULT_DO(
            FindDirectoryRecursive(
                &newParentKey,
                &newParentEntry,
                &newKey,
                pNewFullPath
            )
        );

        // ルートディレクトリへはリネームはできません。
        if( newKey.parentDir == newParentKey.parentDir )
        {
            *outIsParentEntry = true;
            return nn::fs::ResultDirectoryUnrenamable();
        }

        // 自身以下へのリネームはできません。
        NN_RESULT_DO(
            CheckSubDirectory(
                oldKey,
                newParentKey,
                nn::fs::ResultDirectoryUnrenamable()
            )
        );

        // 親ディレクトリへのリネームはできません。
        Result result = CheckSubDirectory(
            newKey,
            oldParentKey,
            nn::fs::ResultDirectoryUnrenamable()
        );
        if( result.IsFailure() )
        {
            *outIsParentEntry = true;
            return result;
        }

        bool isEqualDirectoryName = PathTool::IsEqualDirectoryName(&oldKey.name, &newKey.name);

        // 同名ディレクトリであれば成功を返します。
        if( (oldKey.parentDir == newKey.parentDir) && isEqualDirectoryName )
        {
            *outIsSameEntry = true;
            return nn::fs::ResultAlreadyExists();
        }

        // 同名のファイル、ディレクトリがすでに存在しているかどうかチェックします。
        NN_RESULT_DO(
            CheckSameEntryExists(
                outIsFile,
                newKey.parentDir,
                reinterpret_cast<PathChar*>(&newKey.name),
                PathTool::GetDirectoryNameLength(&newKey.name),
                nn::fs::ResultAlreadyExists()
            )
        );
    }

    // 親ディレクトリが変わる場合はリンク情報を更新します。
    if( oldKey.parentDir != newKey.parentDir )
    {
        // リネーム前の親ディレクトリとのリンクを切ります。
        NN_RESULT_DO(
            RemoveDirectoryLink(
                &oldParentEntry,
                oldKey.parentDir,
                &oldEntry,
                oldIndex
            )
        );

        // リネーム後ディレクトリの兄弟リストの先頭に追加します。
        oldEntry.indexNext = newParentEntry.indexDirectory;
        newParentEntry.indexDirectory = oldIndex;
        NN_RESULT_DO(m_TableDirectory.SetByIndex(newKey.parentDir, newParentEntry));
        NN_RESULT_DO(m_TableDirectory.SetByIndex(oldIndex, oldEntry));
    }

    // キーをリネームします。
    return m_TableDirectory.Rename(newKey, oldKey);
}

/**
* @brief        指定したファイル名を変更します。
*
* @param[out]   outIsFile                   リネーム先に同名のエントリーが存在したときにファイルであれば true、ディレクトリであれば false
* @param[out]   outIsSameEntry              変更前と変更後のパスが同じエントリーを挿しているかどうか
* @param[in]    pNewFullPath                変更後のファイルパス
* @param[in]    pOldFullPath                変更前のファイルパス
*
* @return       関数の処理結果を返します。
* @retval       ResultSuccess                   正常に取得できました。
* @retval       ResultAlreadyExists             変更前後で同じファイル名です。
* @retval       ResultNotInitialized            初期化されていません。
* @retval       ResultInvalidPathFormat         パスがルートディレクトリから始まっていません。
* @retval       ResultTooLongPath               ディレクトリ名が長すぎます。
* @retval       ResultDirectoryUnobtainable     ルートディレクトリの親ディレクトリは取得できません。
* @retval       ResultIncompatiblePath          パスがディレクトリへのパスでした。
* @retval       ResultAlreadyExists             同名のファイルまたはディレクトリが存在します。
* @retval       ResultIncompatiblePath          内部データに問題があります。
* @retval       ResultFileNotFound              内部データに問題があります。
* @retval       ResultDatabaseKeyNotFound       内部データに問題があります。
* @retval       ResultInvalidOffset             内部データに問題があります。
* @retval       ResultDatabaseCorrupted         内部データに問題があります。
* @retval       上記以外                        ストレージでの読み書きに失敗しました。
*
* @pre          引数がすべて NULL ではない。
*
* @details      指定したファイル名を変更します。
*/
template <
    typename TDirectoryName,
    typename TFileName,
    typename TDirectoryInfo,
    typename TFileInfo
>
Result HierarchicalFileTableTemplate<
           TDirectoryName,
           TFileName,
           TDirectoryInfo,
           TFileInfo
       >::RenameFile(
           bool* outIsFile,
           bool* outIsSameEntry,
           const PathChar* pNewFullPath,
           const PathChar* pOldFullPath
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outIsFile);
    NN_SDK_REQUIRES_NOT_NULL(outIsSameEntry);
    NN_SDK_REQUIRES_NOT_NULL(pOldFullPath);
    NN_SDK_REQUIRES_NOT_NULL(pNewFullPath);

    *outIsFile = false;
    *outIsSameEntry = false;

    DirectoryKey oldParentKey;
    DirectoryEntry oldParentEntry;
    FileKey oldKey;

    StorageIndex oldIndex;
    FileEntry oldEntry;

    DirectoryKey newParentKey;
    DirectoryEntry newParentEntry;
    FileKey newKey;

    CacheBufferEnabler enableCacheBuffer(this);

    {
        // リネーム前の親ディレクトリのエントリーを探索します。
        NN_RESULT_DO(
            FindFileRecursive(
                &oldParentKey,
                &oldParentEntry,
                &oldKey,
                pOldFullPath
            )
        );

        // リネーム前のエントリー情報を取得します。
        NN_RESULT_DO(GetFileEntry(&oldIndex, &oldEntry, oldKey));

        // リネーム後の親ディレクトリのエントリーを探索します。
        NN_RESULT_DO(
            FindFileRecursive(
                &newParentKey,
                &newParentEntry,
                &newKey,
                pNewFullPath
            )
        );

        bool isFileNameSame = PathTool::IsEqualFileName(&oldKey.name, &newKey.name);

        // 同名ファイル名です。
        if( (oldKey.parentDir == newKey.parentDir) && isFileNameSame )
        {
            *outIsFile = true;
            *outIsSameEntry = true;
            return nn::fs::ResultAlreadyExists();
        }

        // 同名のファイル、ディレクトリがすでに存在しているかどうかチェックします。
        NN_RESULT_DO(
            CheckSameEntryExists(
                outIsFile,
                newKey.parentDir,
                reinterpret_cast<PathChar*>(&newKey.name),
                PathTool::GetFileNameLength(&newKey.name),
                nn::fs::ResultAlreadyExists()
            )
        );
    }

    // 親ディレクトリが変わる場合はリンク情報を更新します。
    if( oldKey.parentDir != newKey.parentDir )
    {
        // リネーム前の親ディレクトリとのリンクを切ります。
        NN_RESULT_DO(
            RemoveFileLink(
                &oldParentEntry,
                oldKey.parentDir,
                &oldEntry,
                oldIndex
            )
        );

        // リネーム後ディレクトリを兄弟リストの先頭に追加します。
        oldEntry.indexNext = newParentEntry.indexFile;
        newParentEntry.indexFile = oldIndex;
        NN_RESULT_DO(m_TableDirectory.SetByIndex(newKey.parentDir, newParentEntry));
        NN_RESULT_DO(m_TableFile.SetByIndex(oldIndex, oldEntry));
    }

    // キーをリネームします。
    NN_RESULT_DO(m_TableFile.Rename(newKey, oldKey));

    NN_RESULT_SUCCESS;
}

/**
* @brief        指定したディレクトリ情報を取得します。
*
* @param[out]   outIndex                        インデックス
* @param[out]   outDirectoryInfo                ディレクトリ情報
* @param[in]    pFullPath                       ディレクトリのフルパス
*
* @return       関数の処理結果を返します。
* @retval       ResultSuccess                   正常に取得できました。
* @retval       ResultNotInitialized            初期化されていません。
* @retval       ResultInvalidPathFormat         pFullPath がルートディレクトリから始まっていません。
* @retval       ResultTooLongPath               ディレクトリ名が長すぎます。
* @retval       ResultDirectoryUnobtainable     ルートディレクトリの親ディレクトリは取得できません。
* @retval       ResultIncompatiblePath          pFullPath がファイルでした。
* @retval       ResultNotInitialized            初期化されていません。
* @retval       ResultIncompatiblePath          内部データに問題があります。
* @retval       ResultDirectoryNotFound         内部データに問題があります。
* @retval       ResultInvalidOffset             内部データに問題があります。
* @retval       ResultDatabaseCorrupted         内部データに問題があります。
* @retval       上記以外                        ストレージからのデータ読み込みに失敗しました。
*
* @pre          outDirectoryInfo が NULL ではない。
* @pre          pFullPath が NULL ではない。
*
* @details      指定したディレクトリ情報を取得します。
*/
template <
    typename TDirectoryName,
    typename TFileName,
    typename TDirectoryInfo,
    typename TFileInfo
>
Result HierarchicalFileTableTemplate<
           TDirectoryName,
           TFileName,
           TDirectoryInfo,
           TFileInfo
       >::GetDirectoryInformation(
           StorageIndex* outIndex,
           TDirectoryInfo* outDirectoryInfo,
           const PathChar* pFullPath
       ) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outDirectoryInfo);
    NN_SDK_REQUIRES_NOT_NULL(pFullPath);

    // 親ディレクトリのエントリーを探索し、対象ディレクトリ用キ－を生成します。
    DirectoryKey parentKey;
    DirectoryEntry parentEntry;
    DirectoryKey key;
    NN_RESULT_DO(
        FindDirectoryRecursive(
            &parentKey,
            &parentEntry,
            &key,
            pFullPath
        )
    );
    return GetDirectoryInformationFromKey(outIndex, outDirectoryInfo, key);
}

/**
* @brief        ファイルを開きます。
*
* @param[out]   outIndex                    インデックス
* @param[out]   outFileInfo                 ファイル情報
* @param[in]    pFullPath                   ファイルのフルパス
*
* @return       関数の処理結果を返します。
* @retval       ResultSuccess                   正常に取得できました。
* @retval       ResultNotInitialized            初期化されていません。
* @retval       ResultInvalidPathFormat         pFullPath がルートディレクトリから始まっていません。
* @retval       ResultTooLongPath               ディレクトリ名が長すぎます。
* @retval       ResultDirectoryUnobtainable     ルートディレクトリの親ディレクトリは取得できません。
* @retval       ResultIncompatiblePath          pFullPath がディレクトリでした。
* @retval       ResultIncompatiblePath          内部データに問題があります。
* @retval       ResultFileNotFound              内部データに問題があります。
* @retval       ResultInvalidOffset             内部データに問題があります。
* @retval       ResultDatabaseCorrupted         内部データに問題があります。

* @retval       上記以外                        ストレージからのデータ読み込みに失敗しました。
*
* @pre          引数がすべて NULL ではない。
*
* @details      ファイルを開きます。
*/
template <
    typename TDirectoryName,
    typename TFileName,
    typename TDirectoryInfo,
    typename TFileInfo
>
Result HierarchicalFileTableTemplate<
           TDirectoryName,
           TFileName,
           TDirectoryInfo,
           TFileInfo
       >::OpenFile(
           StorageIndex* outIndex,
           TFileInfo* outFileInfo,
           const PathChar* pFullPath
       ) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outIndex);
    NN_SDK_REQUIRES_NOT_NULL(outFileInfo);
    NN_SDK_REQUIRES_NOT_NULL(pFullPath);

    DirectoryKey parentKey;
    DirectoryEntry parentEntry;
    FileKey fileKey;

    CacheBufferEnabler enableCacheBuffer(this);

    // 親ディレクトリのエントリーを探索し、対象ファイル用キ－を生成します。
    NN_RESULT_DO(
        FindFileRecursive(
            &parentKey,
            &parentEntry,
            &fileKey,
            pFullPath
        )
    );

    return OpenFileFromKey(outIndex, outFileInfo, fileKey);
}

/**
* @brief        指定したディレクトリ情報を更新します。
*
* @param[in]    pFullPath                       ディレクトリパス
* @param[in]    dirInfo                         ディレクトリ情報
*
* @return       関数の処理結果を返します。
* @retval       ResultSuccess                   正常に更新できました。
* @retval       ResultNotInitialized            初期化されていません。
* @retval       ResultInvalidPathFormat         pFullPath がルートディレクトリから始まっていません。
* @retval       ResultTooLongPath               ディレクトリ名が長すぎます。
* @retval       ResultDirectoryUnobtainable     ルートディレクトリの親ディレクトリは取得できません。
* @retval       ResultIncompatiblePath          pFullPath がファイルでした。
* @retval       ResultIncompatiblePath          内部データに問題があります。
* @retval       ResultDirectoryNotFound         内部データに問題があります。
* @retval       ResultInvalidOffset             内部データに問題があります。
* @retval       ResultDatabaseCorrupted         内部データに問題があります。
* @retval       上記以外                        ストレージでの読み書きに失敗しました。
*
* @pre          pFullPath が NULL ではない。
*
* @details      指定したディレクトリ情報を更新します。
*/
template <
    typename TDirectoryName,
    typename TFileName,
    typename TDirectoryInfo,
    typename TFileInfo
>
Result HierarchicalFileTableTemplate<
           TDirectoryName,
           TFileName,
           TDirectoryInfo,
           TFileInfo
       >::UpdateDirectoryInformation(
           const PathChar* pFullPath,
           const TDirectoryInfo& dirInfo
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pFullPath);

    // 親ディレクトリのエントリーを探索し、対象ディレクトリ用キ－を生成します。
    DirectoryKey parentKey;
    DirectoryEntry parentEntry;
    DirectoryKey key;
    NN_RESULT_DO(
        FindDirectoryRecursive(
            &parentKey,
            &parentEntry,
            &key,
            pFullPath
        )
    );

    return UpdateDirectoryInformationFromKey(key, dirInfo);
}

/**
* @brief        指定したファイル情報を更新します。
*
* @param[in]    pFullPath                       ファイルパス
* @param[in]    fileInfo                        ファイル情報
*
* @return       関数の処理結果を返します。
* @retval       ResultSuccess                   正常に更新できました。
* @retval       ResultNotInitialized            初期化されていません。
* @retval       ResultInvalidPathFormat         pFullPath がルートディレクトリから始まっていません。
* @retval       ResultTooLongPath               ディレクトリ名が長すぎます。
* @retval       ResultDirectoryUnobtainable     ルートディレクトリの親ディレクトリは取得できません。
* @retval       ResultIncompatiblePath          pFullPath がディレクトリでした。
* @retval       ResultIncompatiblePath          内部データに問題があります。
* @retval       ResultFileNotFound              内部データに問題があります。
* @retval       ResultInvalidOffset             内部データに問題があります。
* @retval       ResultDatabaseCorrupted         内部データに問題があります。
* @retval       上記以外                        ストレージでのデータ読み書きに失敗しました。
*
* @pre          pFullPath が NULL ではない。
*
* @details      指定したファイル情報を更新します。
*/
template <
    typename TDirectoryName,
    typename TFileName,
    typename TDirectoryInfo,
    typename TFileInfo
>
Result HierarchicalFileTableTemplate<
           TDirectoryName,
           TFileName,
           TDirectoryInfo,
           TFileInfo
       >::UpdateFileInformation(
           const PathChar* pFullPath,
           const TFileInfo& fileInfo
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pFullPath);

    // 親ディレクトリのエントリーを探索し、対象ファイル用キ－を生成します。
    DirectoryKey parentKey;
    DirectoryEntry parentEntry;
    FileKey key;
    NN_RESULT_DO(
        FindFileRecursive(
            &parentKey,
            &parentEntry,
            &key,
            pFullPath
        )
    );
    return UpdateFileInformationFromKey(key, fileInfo);
}

/**
* @brief        指定したディレクトリ以下のディレクトリとファイル列挙を開始します。
*
* @param[out]   iter                            イテレータ
* @param[in]    pFullPath                       ディレクトリパス
*
* @return       関数の処理結果を返します。
* @retval       ResultSuccess                   正常に取得できました。
* @retval       ResultNotInitialized            初期化されていません。
* @retval       ResultInvalidPathFormat         pFullPath がルートディレクトリから始まっていません。
* @retval       ResultTooLongPath               ディレクトリ名が長すぎます。
* @retval       ResultDirectoryUnobtainable     ルートディレクトリの親ディレクトリは取得できません。
* @retval       ResultIncompatiblePath          pFullPath がファイルでした。
* @retval       ResultIncompatiblePath          内部データに問題があります。
* @retval       ResultDirectoryNotFound         内部データに問題があります。
* @retval       ResultInvalidOffset             内部データに問題があります。
* @retval       ResultDatabaseCorrupted         内部データに問題があります。
* @retval       上記以外                        ストレージからのデータ読み込みに失敗しました。
*
* @pre          iter が NULL ではない。
* @pre          pFullPath が NULL ではない。
*
* @details      指定したディレクトリ以下のディレクトリとファイル列挙を開始します。
*               イテレータを @see FindNextDirectory、@see FindNextFile に
*               渡すことでディレクトリ名、ファイル名をイテレーションできます。
*               また、イテレーションはファイル、ディレクトリ各々に対して行うことができます。
*/
template <
    typename TDirectoryName,
    typename TFileName,
    typename TDirectoryInfo,
    typename TFileInfo
>
Result HierarchicalFileTableTemplate<
           TDirectoryName,
           TFileName,
           TDirectoryInfo,
           TFileInfo
       >::FindOpen(FindIndex* iter, const PathChar* pFullPath) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(iter);
    NN_SDK_REQUIRES_NOT_NULL(pFullPath);

    // 親ディレクトリのエントリーを探索し、
    // イテレーション対象ディレクトリ用キ－を生成します。
    DirectoryKey parentKey;
    DirectoryEntry parentEntry;
    DirectoryKey dirKey;
    NN_RESULT_DO(
        FindDirectoryRecursive(
            &parentKey,
            &parentEntry,
            &dirKey,
            pFullPath
        )
    );
    return FindOpenWithKey(iter, dirKey);
}

/**
* @brief        ディレクトリ以下の次のディレクトリを取得します。
*
* @param[out]   outDirectoryName        ディレクトリ名
* @param[out]   outFinished             ディレクトリ名
* @param[in]    iter                    イテレータ
*
* @return       関数の処理結果を返します。
* @retval       ResultSuccess                   正常にディレクトリが取得できました。
* @retval       ResultNotInitialized            初期化されていません。
* @retval       ResultInvalidOffset             iter が範囲外です。
* @retval       ResultInvalidOffset             内部で不明な問題が発生しました。
* @retval       ResultDatabaseCorrupted         内部データに問題があります。
* @retval       上記以外                        ストレージからのデータ読み込みに失敗しました。
*
* @pre          引数がすべて NULL ではない。
* @pre          @see FindOpen でイテレータを初期化している。
*
* @details      ディレクトリ以下の次のディレクトリを取得します。
*/
template <
    typename TDirectoryName,
    typename TFileName,
    typename TDirectoryInfo,
    typename TFileInfo
>
Result HierarchicalFileTableTemplate<
           TDirectoryName,
           TFileName,
           TDirectoryInfo,
           TFileInfo
       >::FindNextDirectory(
           TDirectoryName* outDirectoryName,
           bool* outFinished,
           FindIndex* iter
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outDirectoryName);
    NN_SDK_REQUIRES_NOT_NULL(outFinished);
    NN_SDK_REQUIRES_NOT_NULL(iter);

    // 子ディレクトリリストの終端チェックを行います。
    if( iter->nextIndexDirectory == IndexNone )
    {
        // 探索は終了しました
        *outFinished = true;
        NN_RESULT_SUCCESS;
    }

    // 現在のディレクトリエントリーを取得します。
    DirectoryKey key;
    DirectoryEntry entry;
    NN_RESULT_DO(m_TableDirectory.GetByIndex(&key, &entry, iter->nextIndexDirectory));

    // 発見したディレクトリ名を設定します。
    *outDirectoryName = key.name;

    // 次のディレクトリへのインデックスに更新します。
    iter->nextIndexDirectory = entry.indexNext;

    *outFinished = false;

    NN_RESULT_SUCCESS;
}

/**
* @brief        ディレクトリ以下の次のファイルを取得します。
*
* @param[out]   outFileSize             ファイルのサイズ
* @param[out]   outFileName             ファイル名
* @param[out]   outFinished             イテレーションが終了したか(次ファイルが有るか)
* @param[in]    iter                    イテレータ
*
* @return       関数の処理結果を返します。
* @retval       ResultSuccess                   正常にファイルが取得できました。
* @retval       ResultNotInitialized            初期化されていません。
* @retval       ResultInvalidOffset             iter が範囲外です。
* @retval       ResultInvalidOffset             内部で不明な問題が発生しました。
* @retval       ResultDatabaseCorrupted         内部データに問題があります。
* @retval       上記以外                        ストレージからのデータ読み込みに失敗しました。
*
* @pre          @see FindOpen でイテレータを初期化している。
* @pre          引数がすべて NULL ではない。
*
* @details      ディレクトリ以下の次のファイルを取得します。
*/
template <
    typename TDirectoryName,
    typename TFileName,
    typename TDirectoryInfo,
    typename TFileInfo
>
Result HierarchicalFileTableTemplate<
           TDirectoryName,
           TFileName,
           TDirectoryInfo,
           TFileInfo
       >::FindNextFile(
           int64_t* outFileSize,
           TFileName* outFileName,
           bool* outFinished,
           FindIndex* iter
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(iter);
    NN_SDK_REQUIRES_NOT_NULL(outFileSize);
    NN_SDK_REQUIRES_NOT_NULL(outFileName);

    // 子ファイルリストの終端チェックを行います。
    if( iter->nextIndexFile == IndexNone )
    {
        // 探索は終了しました
        *outFinished = true;
        NN_RESULT_SUCCESS;
    }

    // 現在のファイルエントリーを取得します。
    FileKey key;
    FileEntry entry;
    NN_RESULT_DO(m_TableFile.GetByIndex(&key, &entry, iter->nextIndexFile));

    // 発見したファイル情報を設定します。
    *outFileName = key.name;
    *outFileSize = entry.info.fs.size.Get();

    // 次のファイルへのインデックスに更新します。
    iter->nextIndexFile = entry.indexNext;

    *outFinished = false;

    NN_RESULT_SUCCESS;
}

/**
* @brief        指定したパスから親ディレクトリとディレクトリキーを取得します。
*
* @param[out]   outParentDirectoryKey       親ディレクトリキー
* @param[out]   outParentDirectoryEntry     親ディレクトリエントリー
* @param[out]   outDirectoryKey             ディレクトリキー
* @param[in]    pFullPath                   フルパス
*
* @return       関数の処理結果を返します。
* @retval       ResultSuccess                   正常に取得できました。
* @retval       ResultNotInitialized            初期化されていません。
* @retval       ResultInvalidPathFormat         pFullPath がルートディレクトリから始まっていません。
* @retval       ResultTooLongPath               ディレクトリ名が長すぎます。
* @retval       ResultDirectoryUnobtainable     ルートディレクトリの親ディレクトリは取得できません。
* @retval       ResultIncompatiblePath          pFullPath がファイルでした。
* @retval       ResultInvalidOffset             内部データに問題があります。
* @retval       ResultDatabaseCorrupted         内部データに問題があります。
* @retval       上記以外                        ストレージからのデータ読み込みに失敗しました。
*
* @pre          引数はすべて NULL ではない。
*
* @details      指定したパスから親ディレクトリとディレクトリキーを取得します。
*/
template <
    typename TDirectoryName,
    typename TFileName,
    typename TDirectoryInfo,
    typename TFileInfo
>
Result HierarchicalFileTableTemplate<
           TDirectoryName,
           TFileName,
           TDirectoryInfo,
           TFileInfo
       >::FindDirectoryRecursive(
           DirectoryKey* outParentDirectoryKey,
           DirectoryEntry* outParentDirectoryEntry,
           DirectoryKey* outDirectoryKey,
           const PathChar* pFullPath
       ) const NN_NOEXCEPT
{
    CacheBufferEnabler enableCacheBuffer(this);

    NN_SDK_REQUIRES_NOT_NULL(outParentDirectoryKey);
    NN_SDK_REQUIRES_NOT_NULL(outParentDirectoryEntry);
    NN_SDK_REQUIRES_NOT_NULL(outDirectoryKey);
    NN_SDK_REQUIRES_NOT_NULL(pFullPath);

    // フルパスを解析するためのパーサを初期化します。
    PathTool::PathParser parser;
    NN_RESULT_DO(parser.Initialize(pFullPath));

    // 親ディレクトリを探索します。
    StorageIndex parentIndex = IndexNone;
    NN_RESULT_DO(
        FindParentDirectoryRecursive(
            &parentIndex,
            outParentDirectoryKey,
            outParentDirectoryEntry,
            parser
        )
    );

    // パスの最後の部分をディレクトリ名として取得します。
    size_t nameLength;
    NN_RESULT_DO(parser.GetAsDirectoryName(&outDirectoryKey->name, &nameLength));

    // 最後のパスが "." であった場合
    if( PathTool::IsCurrentDirectory(&outDirectoryKey->name, nameLength) )
    {
        // 親ディレクトリと指定したキーをカレントディレクトリとします。
        *outDirectoryKey = *outParentDirectoryKey;

        // 親ディレクトリを更新します。
        if( outDirectoryKey->parentDir != IndexNone )
        {
            NN_RESULT_DO(
                m_TableDirectory.GetByIndex(
                    outParentDirectoryKey,
                    outParentDirectoryEntry,
                    outDirectoryKey->parentDir
                )
            );
        }
    }
    // 最後のパスが".."であった場合
    else if( PathTool::IsParentDirectory(&outDirectoryKey->name, nameLength) )
    {
        if( outParentDirectoryKey->parentDir == IndexNone )
        {
            // ルートディレクトリの親ディレクトリは取得できません。
            return nn::fs::ResultDirectoryUnobtainable();
        }
        // 親ディレクトリの親ディレクトリをカレントディレクトリとします。
        DirectoryEntry currDirectoryEntry;
        NN_RESULT_DO(
            m_TableDirectory.GetByIndex(
                outDirectoryKey,
                &currDirectoryEntry,
                outParentDirectoryKey->parentDir
            )
        );
        // 親ディレクトリを更新します。
        if( outDirectoryKey->parentDir != IndexNone )
        {
            NN_RESULT_DO(
                m_TableDirectory.GetByIndex(
                    outParentDirectoryKey,
                    outParentDirectoryEntry,
                    outDirectoryKey->parentDir
                )
            );
        }
        else
        {
            // "/XXX/.."のようなパスです。
            outParentDirectoryKey->parentDir = IndexNone;
        }
    }
    else
    {
        // 親ディレクトリ ID を設定します。
        // (nameLength == 0 であればルートディレクトリを示しています)
        outDirectoryKey->parentDir = (nameLength > 0) ? parentIndex : IndexNone;
    }
    NN_RESULT_SUCCESS;
}

/**
* @brief        指定したパスから親ディレクトリとファイルキーを取得します。
*
* @param[out]   outParentDirectoryKey       親ディレクトリキー
* @param[out]   outParentDirectoryEntry     親ディレクトリエントリー
* @param[out]   outFileKey                  ファイルキー
* @param[in]    pFullPath                   フルパス
*
* @return       関数の処理結果を返します。
* @retval       ResultSuccess                   正常に取得できました。
* @retval       ResultNotInitialized            初期化されていません。
* @retval       ResultInvalidPathFormat         pFullPath がルートディレクトリから始まっていません。
* @retval       ResultTooLongPath               ディレクトリ名が長すぎます。
* @retval       ResultDirectoryUnobtainable     ルートディレクトリの親ディレクトリは取得できません。
* @retval       ResultIncompatiblePath          pFullPath がディレクトリでした。
* @retval       ResultInvalidOffset             内部データに問題があります。
* @retval       ResultDatabaseCorrupted         内部データに問題があります。
* @retval       上記以外                        ストレージからのデータ読み込みに失敗しました。
*
* @pre          引数がすべて NULL ではない。
* @pre          マウントしている。
*
* @details      指定したパスから親ディレクトリとファイルキーを取得します。
*/
template <
    typename TDirectoryName,
    typename TFileName,
    typename TDirectoryInfo,
    typename TFileInfo
>
Result HierarchicalFileTableTemplate<
           TDirectoryName,
           TFileName,
           TDirectoryInfo,
           TFileInfo
       >::FindFileRecursive(
           DirectoryKey* outParentDirectoryKey,
           DirectoryEntry* outParentDirectoryEntry,
           FileKey* outFileKey,
           const PathChar* pFullPath
       ) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outParentDirectoryKey);
    NN_SDK_REQUIRES_NOT_NULL(outParentDirectoryEntry);
    NN_SDK_REQUIRES_NOT_NULL(outFileKey);
    NN_SDK_REQUIRES_NOT_NULL(pFullPath);

    // フルパスを解析するためのパーサを初期化します。
    PathTool::PathParser parser;
    NN_RESULT_DO(parser.Initialize(pFullPath));

    CacheBufferEnabler enableCacheBuffer(this);

    // 親ディレクトリを探索します。
    StorageIndex parentIndex = IndexNone;
    NN_RESULT_DO(
        FindParentDirectoryRecursive(
            &parentIndex,
            outParentDirectoryKey,
            outParentDirectoryEntry,
            parser
        )
    );

    // パスがファイルを示していたかどうかチェックします。
    if( parser.IsDirectoryPath() )
    {
        // 不正な操作です。
        return nn::fs::ResultIncompatiblePath();
    }

    // パスの最後の部分をファイル名として取得します。
    outFileKey->parentDir = parentIndex;
    size_t nameLength;
    NN_RESULT_DO(parser.GetAsFileName(&outFileKey->name, &nameLength));
    NN_RESULT_SUCCESS;
}

/**
* @brief        指定したディレクトリ情報を取得します。
*
* @param[out]   outIndex                        ディレクトリエントリーのインデックス
* @param[out]   outDirectoryInfo                ディレクトリ情報
* @param[in]    dirKey                          ディレクトリキー
*
* @return       関数の処理結果を返します。
* @retval       ResultSuccess                   ディレクトリ情報が取得できました。
* @retval       ResultNotInitialized            初期化されていません。
* @retval       ResultIncompatiblePath          dirKey はディレクトリではなくファイルです。
* @retval       ResultDirectoryNotFound         dirKey に紐づくディレクトリエントリーがありません。
* @retval       ResultInvalidOffset             内部データに問題があります。
* @retval       ResultDatabaseCorrupted         内部データに問題があります。
* @retval       上記以外                        ストレージからのデータ読み込みに失敗しました。
*
* @pre          outIndex が NULL ではない。
* @pre          outDirectoryInfo が NULL ではない。
*
* @details      指定したディレクトリ情報を取得します。
*/
template <
    typename TDirectoryName,
    typename TFileName,
    typename TDirectoryInfo,
    typename TFileInfo
>
Result HierarchicalFileTableTemplate<
           TDirectoryName,
           TFileName,
           TDirectoryInfo,
           TFileInfo
       >::GetDirectoryInformationFromKey(
           StorageIndex* outIndex,
           TDirectoryInfo* outDirectoryInfo,
           const DirectoryKey& dirKey
       ) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outIndex);
    NN_SDK_REQUIRES_NOT_NULL(outDirectoryInfo);
    // ディレクトリエントリーを取得します。
    StorageIndex index;
    DirectoryEntry entry;
    NN_RESULT_DO(GetDirectoryEntry(&index, &entry, dirKey));

    // ディレクトリの付加情報をコピーします。
    *outIndex = index;
    *outDirectoryInfo = entry.info;

    NN_RESULT_SUCCESS;
}

/**
* @brief        ファイルを開きます。
*
* @param[out]   outIndex                        ファイルエントリーのインデックス
* @param[out]   outFileInfo                     ファイル情報
* @param[in]    fileKey                         ファイルキー
*
* @return       関数の処理結果を返します。
* @retval       ResultSuccess                   正常にファイル情報を取得しました。
* @retval       ResultNotInitialized            初期化されていません。
* @retval       ResultIncompatiblePath          fileKey はファイルではなくディレクトリです。
* @retval       ResultFileNotFound              fileKey に紐づくファイルエントリーがありません。
* @retval       ResultInvalidOffset             内部データに問題があります。
* @retval       ResultDatabaseCorrupted         内部データに問題があります。
* @retval       上記以外                        ストレージからのデータ読み込みに失敗しました。
*
* @pre          outIndex が NULL ではない。
* @pre          outFileInfo が NULL ではない。
*
* @details      ファイルを開きます。
*/
template <
    typename TDirectoryName,
    typename TFileName,
    typename TDirectoryInfo,
    typename TFileInfo
>
Result HierarchicalFileTableTemplate<
           TDirectoryName,
           TFileName,
           TDirectoryInfo,
           TFileInfo
       >::OpenFileFromKey(
           StorageIndex* outIndex,
           TFileInfo* outFileInfo,
           const FileKey& fileKey
       ) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outIndex);
    NN_SDK_REQUIRES_NOT_NULL(outFileInfo);

    CacheBufferEnabler enableCacheBuffer(this);

    // ファイルエントリーを取得します。
    FileEntry fileEntry;
    StorageIndex index;
    NN_RESULT_DO(GetFileEntry(&index, &fileEntry, fileKey));

    // ファイルの付加情報をコピーします。
    *outFileInfo = fileEntry.info;
    *outIndex = index;

    NN_RESULT_SUCCESS;
}

/**
* @brief        指定したディレクトリ情報を更新します。
*
* @param[in]    dirKey                      ディレクトリキー
* @param[in]    dirInfo                     ディレクトリ情報
*
* @return       関数の処理結果を返します。
* @retval       ResultSuccess                   ディレクトリエントリーが更新できました。
* @retval       ResultNotInitialized            初期化されていません。
* @retval       ResultIncompatiblePath          dirKey はディレクトリではなくファイルです。
* @retval       ResultDirectoryNotFound         dirKey に紐づくディレクトリエントリーがありません。
* @retval       ResultInvalidOffset             内部データに問題があります。
* @retval       ResultDatabaseCorrupted         内部データに問題があります。
* @retval       上記以外                        ストレージでの読み書きに失敗しました。
*
* @details      指定したディレクトリ情報を更新します。
*/
template <
    typename TDirectoryName,
    typename TFileName,
    typename TDirectoryInfo,
    typename TFileInfo
>
Result HierarchicalFileTableTemplate<
           TDirectoryName,
           TFileName,
           TDirectoryInfo,
           TFileInfo
       >::UpdateDirectoryInformationFromKey(
           const DirectoryKey& dirKey,
           const TDirectoryInfo& dirInfo
       ) NN_NOEXCEPT
{
    CacheBufferEnabler enableCacheBuffer(this);

    // 情報を更新するディレクトリエントリーを取得します。
    StorageIndex index;
    DirectoryEntry entry;
    NN_RESULT_DO(GetDirectoryEntry(&index, &entry, dirKey));

    // ディレクトリ情報を更新します。
    entry.info = dirInfo;
    NN_RESULT_DO(m_TableDirectory.SetByIndex(index, entry));

    NN_RESULT_SUCCESS;
}

/**
* @brief        指定したファイル情報を更新します。
*
* @param[in]    fileKey                     ファイルキー
* @param[in]    fullInfo                    ファイル情報
*
* @return       関数の処理結果を返します。
* @retval       ResultSuccess                   正常に更新できました。
* @retval       ResultNotInitialized            初期化されていません。
* @retval       ResultIncompatiblePath          fileKey はファイルではなくディレクトリです。
* @retval       ResultFileNotFound              fileKey に紐づくファイルエントリーがありません。
* @retval       ResultInvalidOffset             内部データに問題があります。
* @retval       ResultDatabaseCorrupted         内部データに問題があります。
* @retval       上記以外                        ストレージでのデータ読み書きに失敗しました。
*
* @details      指定したファイル情報を更新します。
*/
template <
    typename TDirectoryName,
    typename TFileName,
    typename TDirectoryInfo,
    typename TFileInfo
>
Result HierarchicalFileTableTemplate<
           TDirectoryName,
           TFileName,
           TDirectoryInfo,
           TFileInfo
       >::UpdateFileInformationFromKey(
           const FileKey& fileKey,
           const TFileInfo& fileInfo
       ) NN_NOEXCEPT
{
    CacheBufferEnabler enableCacheBuffer(this);

    // 情報を更新するファイルエントリーを取得します。
    StorageIndex index;
    FileEntry entry;
    NN_RESULT_DO(GetFileEntry(&index, &entry, fileKey));

    // ファイル情報を更新します。
    entry.info = fileInfo;
    NN_RESULT_DO(m_TableFile.SetByIndex(index, entry));

    NN_RESULT_SUCCESS;
}

/**
* @brief        指定したディレクトリ以下のディレクトリとファイル列挙を開始します。
*
* @param[out]   pIter                           イテレータ
* @param[in]    dirKey                          ディレクトリキー
*
* @return       関数の処理結果を返します。
* @retval       ResultSuccess                   正常にイテレータを取得できました。
* @retval       ResultNotInitialized            初期化されていません。
* @retval       ResultIncompatiblePath          dirKey はディレクトリではなくファイルです。
* @retval       ResultDirectoryNotFound         dirKey に紐づくディレクトリエントリーがありません。
* @retval       ResultInvalidOffset             内部データに問題があります。
* @retval       ResultDatabaseCorrupted         内部データに問題があります。
* @retval       上記以外                        ストレージからのデータ読み込みに失敗しました。
*
* @pre          pIter が NULL ではない。
*
* @details      指定したディレクトリ以下のディレクトリとファイル列挙を開始します。
*               イテレータを @see FindNextDirectory、@see FindNextFile に
*               渡すことでディレクトリ名、ファイル名をイテレーションできます。
*               また、イテレーションはファイル、ディレクトリ各々に対して行うことができます。
*/
template <
    typename TDirectoryName,
    typename TFileName,
    typename TDirectoryInfo,
    typename TFileInfo
>
Result HierarchicalFileTableTemplate<
           TDirectoryName,
           TFileName,
           TDirectoryInfo,
           TFileInfo
       >::FindOpenWithKey(
           FindIndex* pIter,
           const DirectoryKey& dirKey
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pIter);

    pIter->nextIndexDirectory = IndexNone;
    pIter->nextIndexFile = IndexNone;

    // イテレーション対象のディレクトリエントリーを取得します。
    StorageIndex index;
    DirectoryEntry dirEntry;
    NN_RESULT_DO(GetDirectoryEntry(&index, &dirEntry, dirKey));

    // 子ディレクトリ、ファイルリストへのインデックスを取得します。
    pIter->nextIndexDirectory = dirEntry.indexDirectory;
    pIter->nextIndexFile = dirEntry.indexFile;

    NN_RESULT_SUCCESS;
}

/**
* @brief        ID からエントリとディレクトリが親子関係にあるか調べる
*
* @param[out]   outIsParent         checkEntryId が baseDirectoryId の子か否か
* @param[in]    baseDirectoryId     ディレクトリの ID
* @param[in]    checkEntryId        エントリの ID
* @param[in]    isFile              ファイルであれば true、ディレクトリであれば false
*
* @return       関数の処理結果を返します。
*
* @details      ID からエントリとディレクトリが親子関係にあるか調べる
*/
template <
    typename TDirectoryName,
    typename TFileName,
    typename TDirectoryInfo,
    typename TFileInfo
>
Result HierarchicalFileTableTemplate<
        TDirectoryName,
        TFileName,
        TDirectoryInfo,
        TFileInfo
    >::CheckSubEntry(
        bool* outIsSubEntry,
        StorageIndex baseDirectoryId,
        StorageIndex checkEntryId,
        bool isFile
    ) const NN_NOEXCEPT
{
    *outIsSubEntry = false;
    // ファイルなら、親ディレクトリを求める
    StorageIndex parentDirectory = checkEntryId;
    if( isFile )
    {
        FileKey keyFile;
        FileEntry keyEntry;
        NN_RESULT_DO(
            m_TableFile.GetByIndex(&keyFile, &keyEntry, checkEntryId)
        );
        parentDirectory = keyFile.parentDir;
    }

    // 各ディレクトリのキーを取得
    DirectoryKey keyBase;
    DirectoryEntry entryBase;
    NN_RESULT_DO(
        m_TableDirectory.GetByIndex(&keyBase, &entryBase, baseDirectoryId)
    );
    DirectoryKey keyCheck;
    DirectoryEntry entryCheck;
    NN_RESULT_DO(
        m_TableDirectory.GetByIndex(&keyCheck, &entryCheck, parentDirectory)
    );

    // 親子関係にあるかチェックする
    // CheckSubDirectory と同じコードです
    while( keyCheck.parentDir != IndexNone )
    {
        bool isEqualDirectoryName
            = PathTool::IsEqualDirectoryName(&keyBase.name, &keyCheck.name);
        if( (keyBase.parentDir == keyCheck.parentDir) && isEqualDirectoryName )
        {
            *outIsSubEntry = true;
            break;
        }
        NN_RESULT_DO(
            m_TableDirectory.GetByIndex(&keyCheck, &entryCheck, keyCheck.parentDir)
        );
    }
    NN_RESULT_SUCCESS;
}

/**
* @brief        メタデータの更新を通知します。
*
* @param[out]   outFinished             イテレーションが終了したか
* @param[in]    iter                    イテレータ
* @param[in]    index                   インデックス
* @param[in]    isFile                  ファイルなら true, ディレクトリなら false
*
* @return       関数の処理結果を返します。
* @retval       ResultSuccess                   正常にエントリーが取得できました。
* @retval       ResultNotInitialized            初期化されていません。
* @retval       ResultInvalidOffset             iter が範囲外です。
* @retval       ResultInvalidOffset             内部で不明な問題が発生しました。
* @retval       ResultDatabaseCorrupted         内部データに問題があります。
* @retval       上記以外                        ストレージからのデータ読み込みに失敗しました。
*
*
* @pre          iter が NULL ではない。
*
* @details      メタデータの更新を通知します。
*/
template <
    typename TDirectoryName,
    typename TFileName,
    typename TDirectoryInfo,
    typename TFileInfo
>
Result HierarchicalFileTableTemplate<
           TDirectoryName,
           TFileName,
           TDirectoryInfo,
           TFileInfo
       >::Notify(
           bool* outFinished,
           FindIndex* iter,
           int64_t index,
           bool isFile
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(iter);

    // 削除対象がディレクトリの場合
    if( !isFile )
    {
        if( index == iter->nextIndexDirectory )
        {
            // 列挙対象のエントリーが操作されようとしているので、列挙を一つ進めておきます。
            DirectoryName dummyName;
            return FindNextDirectory(&dummyName, outFinished, iter);
        }
    }
    // 削除対象がファイルの場合
    else
    {
        if( index == iter->nextIndexFile )
        {
            // 列挙対象のエントリーが操作されようとしているので、列挙を一つ進めておきます。
            FileName dummyName;
            int64_t dummySize;
            return FindNextFile(&dummySize, &dummyName, outFinished, iter);
        }
    }
    NN_RESULT_SUCCESS;
}

/**
* @brief        指定した親ディレクトを取得します。
*
* @param[out]   outParentIndex              親ディレクトリインデックス
* @param[out]   outParentDirectoryKey       親ディレクトリキー
* @param[out]   outParentDirectoryEntry     親ディレクトリエントリー
* @param[in]    parser                      パスパーサー
*
* @return       関数の処理結果を返します。
* @retval       ResultSuccess                   正常に親ディレクトリが取得できました。
* @retval       ResultNotInitialized            初期化されていません。
* @retval       ResultTooLongPath               parser のディレクトリ名が長すぎます。
* @retval       ResultIncompatiblePath          parser はディレクトリではなくファイルです。
* @retval       ResultDirectoryNotFound         parser に紐づくディレクトリエントリーがありません。
* @retval       ResultDirectoryUnobtainable     ルートディレクトリからは親ディレクトリを取得できません。
* @retval       ResultInvalidOffset             内部データに問題があります。
* @retval       ResultDatabaseCorrupted         内部データに問題があります。
* @retval       上記以外                        ストレージからのデータ読み込みに失敗しました。
*
* @pre          マウントしている。
* @pre          pParentIndex が NULL ではない。
* @pre          pParentDirectoryKey が NULL ではない。
* @pre          pParentDirectoryEntry が NULL ではない。
*
* @details      指定した親ディレクトを取得します。
*/
template <
    typename TDirectoryName,
    typename TFileName,
    typename TDirectoryInfo,
    typename TFileInfo
>
Result HierarchicalFileTableTemplate<
           TDirectoryName,
           TFileName,
           TDirectoryInfo,
           TFileInfo
       >::FindParentDirectoryRecursive(
           StorageIndex* outParentIndex,
           DirectoryKey* outParentDirectoryKey,
           DirectoryEntry* outParentDirectoryEntry,
           PathTool::PathParser& parser
       ) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outParentIndex);
    NN_SDK_REQUIRES_NOT_NULL(outParentDirectoryKey);
    NN_SDK_REQUIRES_NOT_NULL(outParentDirectoryEntry);

    StorageIndex dirIndex = IndexNone;
    size_t dirNameLength;
    DirectoryKey dirKey;
    DirectoryEntry dirEntry;

    // ルートディレクトリを取得します。
    dirKey.parentDir = IndexNone;
    NN_RESULT_DO(parser.GetNextDirectoryName(&dirKey.name, &dirNameLength));
    NN_RESULT_DO(GetDirectoryEntry(&dirIndex, &dirEntry, dirKey));

    StorageIndex parentIndex = dirIndex;
    while( ! parser.IsParseFinished() )
    {
        DirectoryKey oldKey = dirKey;

        // 次のディレクトリ名を取得します。
        NN_RESULT_DO(parser.GetNextDirectoryName(&dirKey.name, &dirNameLength));

        if( PathTool::IsCurrentDirectory(&dirKey.name, dirNameLength) )
        {
            // "."の場合何もしない。
            dirKey = oldKey;
            continue;
        }
        else if( PathTool::IsParentDirectory(&dirKey.name, dirNameLength) )
        {
            if( dirKey.parentDir == IndexNone )
            {
                // ルートディレクトリの親ディレクトリは取得できません。
                return nn::fs::ResultDirectoryUnobtainable();
            }

            parentIndex = dirKey.parentDir;

            // ".."の場合は親ディレクトリを取得します。
            NN_RESULT_DO(m_TableDirectory.GetByIndex(&dirKey, &dirEntry, parentIndex));
        }
        else
        {
            // 親ディレクトリ ID + ディレクトリ名から、ディレクトリエントリーを取得します。
            dirKey.parentDir = parentIndex;
            NN_RESULT_DO(GetDirectoryEntry(&dirIndex, &dirEntry, dirKey));

            parentIndex = dirIndex;
        }
    }

    // 親ディレクトリ情報を設定します。
    *outParentIndex = parentIndex;
    *outParentDirectoryKey = dirKey;
    *outParentDirectoryEntry = dirEntry;

    NN_RESULT_SUCCESS;
}

/**
* @brief        指定したエントリーと同名のファイル、ディレクトリが存在するかどうかを取得します。
*
* @param[out]   outIsFile                   同名のエントリーが存在したときに
*                                           ファイルであれば true、ディレクトリであれば false
* @param[in]    parentDir                   親ディレクトリインデックス
* @param[in]    pName                       エントリー名
* @param[in]    nameChars                   エントリー名の長さ
* @param[in]    resultIfExist                   同名のエントリーが存在したときに返すリザルト
*
* @return       関数の処理結果を返します。
* @retval       ResultSuccess                   同名のエントリーは検出されませんでした。
* @retval       ResultNotInitialized            初期化されていません。
* @retval       ResultInvalidOffset             内部データに問題があります。
* @retval       ResultDatabaseCorrupted         内部データに問題があります。
* @retval       resultIfExist に渡した値        同名のファイルまたはディレクトリが存在します。
* @retval       上記以外                        ストレージからのデータ読み込みに失敗しました。
*
* @pre          outIsFile が NULL ではない。
*
* @details      指定したエントリーと同名のファイル、ディレクトリが存在するかどうかを取得します。
*/
template <
    typename TDirectoryName,
    typename TFileName,
    typename TDirectoryInfo,
    typename TFileInfo
>
Result HierarchicalFileTableTemplate<
           TDirectoryName,
           TFileName,
           TDirectoryInfo,
           TFileInfo
       >::CheckSameEntryExists(
           bool* outIsFile,
           StorageIndex parentDir,
           const PathChar* pName,
           size_t nameChars,
           Result resultIfExist
       ) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outIsFile);

    // 同名ディレクトリの存在をチェックします。
    DirectoryKey dirKey;
    dirKey.parentDir = parentDir;
    if( PathTool::GetDirectoryName(&dirKey.name, pName, nameChars) )
    {
        StorageIndex index;
        DirectoryEntry dirEntry;
        Result result = m_TableDirectory.Get(&index, &dirEntry, dirKey);
        if( ! nn::fs::ResultDatabaseKeyNotFound::Includes(result) )
        {
            if( result.IsFailure() )
            {
                // ストレージによるエラーです。
                return result;
            }
            else
            {
                // 同名のエントリーが存在します。
                *outIsFile = false;
                return resultIfExist;
            }
        }
    }

    // 同名ファイルの存在をチェックします。
    FileKey fileKey;
    fileKey.parentDir = parentDir;
    if( PathTool::GetFileName(&fileKey.name, pName, nameChars) )
    {
        StorageIndex index;
        FileEntry fileEntry;
        Result result = m_TableFile.Get(&index, &fileEntry, fileKey);
        if( ! nn::fs::ResultDatabaseKeyNotFound::Includes(result) )
        {
            if( result.IsFailure() )
            {
                // ストレージによるエラーです。
                return result;
            }
            else
            {
                // 同名のエントリーが存在します。
                *outIsFile = true;
                return resultIfExist;
            }
        }
    }
    // 同名のエントリーは存在しません。
    NN_RESULT_SUCCESS;
}

/**
* @brief        ディレクトリエントリーを取得します。
*
* @param[out]   outIndex                    ディレクトリインデックス
* @param[out]   outEntry                    ディレクトリエントリー
* @param[in]    key                         ディレクトリキー
*
* @return       関数の処理結果を返します。
* @retval       ResultSuccess                   ディレクトリエントリーが取得できました。
* @retval       ResultNotInitialized            初期化されていません。
* @retval       ResultIncompatiblePath          key はディレクトリではなくファイルです。
* @retval       ResultDirectoryNotFound         key に紐づくディレクトリエントリーがありません。
* @retval       ResultInvalidOffset             内部データに問題があります。
* @retval       ResultDatabaseCorrupted         内部データに問題があります。
* @retval       上記以外                        ストレージからのデータ読み込みに失敗しました。
*
* @pre          pIndex が NULL ではない。
* @pre          pEntry が NULL ではない。
*
* @details      ディレクトリエントリーを取得します。
*/
template <
    typename TDirectoryName,
    typename TFileName,
    typename TDirectoryInfo,
    typename TFileInfo
>
Result HierarchicalFileTableTemplate<
           TDirectoryName,
           TFileName,
           TDirectoryInfo,
           TFileInfo
       >::GetDirectoryEntry(
           StorageIndex* outIndex,
           DirectoryEntry* outEntry,
           const DirectoryKey& dirKey
       ) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outIndex);
    NN_SDK_REQUIRES_NOT_NULL(outEntry);

    Result result = m_TableDirectory.Get(outIndex, outEntry, dirKey);
    if( nn::fs::ResultDatabaseKeyNotFound::Includes(result) )
    {
        // ファイルをディレクトリとして使用していないかどうかチェックします。
        FileKey fileKey;
        fileKey.parentDir = dirKey.parentDir;
        if( PathTool::ConvertDirectoryNameToFileName(&fileKey.name, &dirKey.name) )
        {
            StorageIndex fileIndex;
            FileEntry fileEntry;
            result = m_TableFile.Get(&fileIndex, &fileEntry, fileKey);
            if( result.IsFailure() )
            {
                if( ! nn::fs::ResultDatabaseKeyNotFound::Includes(result) )
                {
                    return result;
                }
            }
            else
            {
                // 不正な操作です。
                return nn::fs::ResultIncompatiblePath();
            }
        }
        // ディレクトリが見つかりませんでした。
        return nn::fs::ResultDirectoryNotFound();
    }

    return result;
}

/**
* @brief        ファイルエントリーを取得します。
*
* @param[out]   outIndex                    ファイルインデックス
* @param[out]   outEntry                    ファイルエントリー
* @param[in]    key                         ファイルキー
*
* @return       関数の処理結果を返します。
* @retval       ResultSuccess                   ファイルエントリーが取得できました。
* @retval       ResultNotInitialized            初期化されていません。
* @retval       ResultIncompatiblePath          key はファイルではなくディレクトリです。
* @retval       ResultFileNotFound              key に紐づくファイルエントリーがありません。
* @retval       ResultInvalidOffset             内部データに問題があります。
* @retval       ResultDatabaseCorrupted         内部データに問題があります。
* @retval       上記以外                        ストレージからのデータ読み込みに失敗しました。
*
* @pre          outIndex が NULL ではない。
* @pre          outEntry が NULL ではない。
*
* @details      ファイルエントリーを取得します。
*/
template <
    typename TDirectoryName,
    typename TFileName,
    typename TDirectoryInfo,
    typename TFileInfo
>
Result HierarchicalFileTableTemplate<
           TDirectoryName,
           TFileName,
           TDirectoryInfo,
           TFileInfo
       >::GetFileEntry(
           StorageIndex* outIndex,
           FileEntry* outEntry,
           const FileKey& key
       ) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outIndex);
    NN_SDK_REQUIRES_NOT_NULL(outEntry);

    Result result = m_TableFile.Get(outIndex, outEntry, key);
    if( nn::fs::ResultDatabaseKeyNotFound::Includes(result) )
    {
        // ディレクトリをファイルとして使用していないかどうかチェックします。
        DirectoryKey dirKey;
        dirKey.parentDir = key.parentDir;
        if( PathTool::ConvertFileNameToDirectoryName(&dirKey.name, &key.name) )
        {
            StorageIndex dirIndex;
            DirectoryEntry dirEntry;
            result = m_TableDirectory.Get(&dirIndex, &dirEntry, dirKey);
            if( result.IsFailure() )
            {
                if( ! nn::fs::ResultDatabaseKeyNotFound::Includes(result) )
                {
                    return result;
                }
            }
            else
            {
                // 不正な操作です。
                return nn::fs::ResultIncompatiblePath();
            }
        }
        // ファイルが見つかりませんでした。
        return nn::fs::ResultFileNotFound();
    }

    return result;
}

/**
* @brief        指定したディレクトリエントリーへのリンクを削除します。
*
* @param[in]    outParentEntry      削除するエントリーの親ディレクトリエントリー
* @param[in]    parentIndex         削除するエントリーの親ディレクトリインデックス
* @param[in]    pDeleteEntry        削除するディレクトリエントリー
* @param[in]    deleteIndex         削除するディレクトリインデックス
*
* @return       関数の処理結果を返します。
* @retval       ResultSuccess                   正常に削除しました
* @retval       ResultNotInitialized            初期化されていません。
* @retval       ResultInvalidOffset             parentIndex が範囲外です。
* @retval       ResultInvalidOffset             pParentEntry が範囲外です。
* @retval       ResultInvalidOffset             内部データに問題があります。
* @retval       ResultDatabaseCorrupted         内部データに問題があります。
* @retval       上記以外                        ストレージでの読み書きに失敗しました。
*
* @pre          outParentEntry は NULL ではない。
* @pre          pDeleteEntry は NULL ではない。
* @pre          outParentEntry->indexDirectory は IndexNone ではない。
*
* @details      指定したディレクトリエントリーへのリンクを削除します。
*/
template <
    typename TDirectoryName,
    typename TFileName,
    typename TDirectoryInfo,
    typename TFileInfo
>
Result HierarchicalFileTableTemplate<
           TDirectoryName,
           TFileName,
           TDirectoryInfo,
           TFileInfo
       >::RemoveDirectoryLink(
           DirectoryEntry* pParentEntry,
           StorageIndex parentIndex,
           DirectoryEntry* pDeleteEntry,
           StorageIndex deleteIndex
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pParentEntry);
    NN_SDK_REQUIRES_NOT_NULL(pDeleteEntry);

    NN_SDK_REQUIRES_NOT_EQUAL(IndexNone, pParentEntry->indexDirectory);

    if( pParentEntry->indexDirectory == deleteIndex )
    {
        // 先頭のリンクを更新します。
        pParentEntry->indexDirectory = pDeleteEntry->indexNext;
        pDeleteEntry->indexNext = IndexNone;
        NN_RESULT_DO(m_TableDirectory.SetByIndex(parentIndex, *pParentEntry));
    }
    else
    {
        DirectoryKey prevKey;
        DirectoryEntry prevEntry;
        NN_RESULT_DO(m_TableDirectory.GetByIndex(&prevKey, &prevEntry, pParentEntry->indexDirectory));

        StorageIndex prevIndex = pParentEntry->indexDirectory;
        StorageIndex index = prevEntry.indexNext;
        DirectoryKey key;
        DirectoryEntry entry;
        while( index != IndexNone )
        {
            NN_RESULT_DO(m_TableDirectory.GetByIndex(&key, &entry, index));

            // 削除するエントリーを見つけた場合はリンクを更新します。
            if( index == deleteIndex )
            {
                prevEntry.indexNext = entry.indexNext;
                NN_RESULT_DO(m_TableDirectory.SetByIndex(prevIndex, prevEntry));

                entry.indexNext = IndexNone;
            }
            prevIndex = index;
            index = entry.indexNext;
            prevEntry = entry;
        }
    }
    NN_RESULT_SUCCESS;
}

/**
* @brief        指定したファイルエントリーへのリンクを削除します。
*
* @param[in]    pParentEntry        削除するエントリーの親ディレクトリエントリー
* @param[in]    parentIndex         削除するエントリーの親ディレクトリインデックス
* @param[in]    pDeleteEntry        削除するファイルエントリー
* @param[in]    deleteIndex         削除するファイルインデックス
*
* @return       関数の処理結果を返します。
* @retval       ResultSuccess                   正常に削除しました。
* @retval       ResultNotInitialized            初期化されていません。
* @retval       ResultInvalidOffset             parentIndex が範囲外です。
* @retval       ResultInvalidOffset             pParentEntry が範囲外です。
* @retval       ResultInvalidOffset             内部データに問題があります。
* @retval       ResultDatabaseCorrupted         内部データに問題があります。
* @retval       上記以外                        ストレージでの読み書きに失敗しました。
*
* @pre          pParentEntry は NULL ではない。
* @pre          pDeleteEntry は NULL ではない。
* @pre          pParentEntry->indexFile は IndexNone ではない。
*
* @details      指定したファイルエントリーへのリンクを削除します。
*/
template <
    typename TDirectoryName,
    typename TFileName,
    typename TDirectoryInfo,
    typename TFileInfo
>
Result HierarchicalFileTableTemplate<
           TDirectoryName,
           TFileName,
           TDirectoryInfo,
           TFileInfo
       >::RemoveFileLink(
           DirectoryEntry* pParentEntry,
           StorageIndex parentIndex,
           FileEntry* pDeleteEntry,
           StorageIndex deleteIndex
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pParentEntry);
    NN_SDK_REQUIRES_NOT_NULL(pDeleteEntry);

    NN_SDK_REQUIRES_NOT_EQUAL(IndexNone, pParentEntry->indexFile);

    if( pParentEntry->indexFile == deleteIndex )
    {
        // 先頭のリンクを更新します。
        pParentEntry->indexFile = pDeleteEntry->indexNext;
        pDeleteEntry->indexNext = IndexNone;
        NN_RESULT_DO(m_TableDirectory.SetByIndex(parentIndex, *pParentEntry));
    }
    else
    {
        FileKey prevKey;
        FileEntry prevEntry;
        NN_RESULT_DO(m_TableFile.GetByIndex(&prevKey, &prevEntry, pParentEntry->indexFile));

        StorageIndex prevIndex = pParentEntry->indexFile;
        StorageIndex index = prevEntry.indexNext;
        FileKey key;
        FileEntry entry;
        while( index != IndexNone )
        {
            NN_RESULT_DO(m_TableFile.GetByIndex(&key, &entry, index));

            // 削除するエントリーを見つけた場合はリンクを更新します。
            if( index == deleteIndex )
            {
                prevEntry.indexNext = entry.indexNext;
                NN_RESULT_DO(m_TableFile.SetByIndex(prevIndex, prevEntry));

                pDeleteEntry->indexNext = IndexNone;
                break;
            }

            prevIndex = index;
            prevEntry = entry;
            index = entry.indexNext;
        }
    }

    NN_RESULT_SUCCESS;
}

/**
* @brief        子ディレクトリかどうかを判定します。
*
* @param[in]    baseDirectoryKey        基準となるディレクトリ
* @param[in]    dirKey                  探索するディレクトリ
* @param[in]    resultIfSubDirectory    各ディレクトリが親子関係だった時に返すリザルト。
*
* @return       関数の処理結果を返します。
* @retval       ResultSuccess                   baseDirectoryKey と dirKey の間に親子関係はありませんでした。
* @retval       ResultNotInitialized            初期化されていません。
* @retval       ResultInvalidOffset             dirKey が範囲外です。
* @retval       ResultInvalidOffset             内部で不明な問題が発生しました。
* @retval       ResultDatabaseCorrupted         内部データに問題があります。
* @retval       resultIfSubDirectory            baseDirectoryKey と dirKey は親子関係にあります。
* @retval       上記以外                        ストレージからのデータ読み込みに失敗しました。
*
* @details      子ディレクトリかどうかを判定します。
*               直接の親子だけではなく、子孫と先祖の関係にあるかを調べます。
*               また、baseDirectoryKey と dirKey が同一であるときも、resultIfSubDirectoryを返します。
*/
template <
    typename TDirectoryName,
    typename TFileName,
    typename TDirectoryInfo,
    typename TFileInfo
>
Result HierarchicalFileTableTemplate<
           TDirectoryName,
           TFileName,
           TDirectoryInfo,
           TFileInfo
       >::CheckSubDirectory(
           const DirectoryKey& baseDirectoryKey,
           DirectoryKey dirKey,
           Result resultIfSubDirectory
       ) NN_NOEXCEPT
{
    while( dirKey.parentDir != IndexNone )
    {
        bool isEqualDirectoryName = PathTool::IsEqualDirectoryName(&baseDirectoryKey.name, &dirKey.name);
        if( (baseDirectoryKey.parentDir == dirKey.parentDir) && isEqualDirectoryName )
        {
            return resultIfSubDirectory;
        }
        DirectoryEntry dirEntry;
        NN_RESULT_DO(m_TableDirectory.GetByIndex(&dirKey, &dirEntry, dirKey.parentDir));
    }

    // ルートディレクトリまで辿った場合は Success を返します。
    NN_RESULT_SUCCESS;
}

}}}

