﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_BitTypes.h>

#include <nn/fs/fs_SubStorage.h>
#include <nn/fssystem/dbm/fs_DbmUtils.h>

class DuplexBitmapTest;

namespace nn { namespace fssystem { namespace dbm {

/**
* @brief        二重化ビットマップクラスです。
*
* @details      データ破損対策用ストレージの二重化に使用します。
*/
class DuplexBitmap
{
    NN_DISALLOW_COPY(DuplexBitmap);

public:
    //! 二重化ビットマップ用イテレータです。
    struct Iterator
    {
        uint32_t index;             //!< 次回探索インデックス
        uint32_t indexEnd;          //!< 探索終了インデックス
    };
    NN_STATIC_ASSERT(std::is_pod<Iterator>::value);

public:
    /**
    * @brief        指定したビット数(ビット数)の二重化ビットマップに必要なデータサイズを取得します。
    *
    * @param[in]    blockCount      ビット数
    *
    * @return       指定したビット数の二重化ビットマップに必要なデータサイズ。
    *
    * @details      指定したビット数(ビット数)の二重化ビットマップに必要なデータサイズを取得します。
    *               64 バイト単位で切り上げを行います。
    */
    static int64_t QuerySize(uint32_t bitCount) NN_NOEXCEPT;

    /**
    * @brief        メタデータストレージをフォーマットします。
    *
    * @param[in]    bitCount            総ビット数
    * @param[in]    table               メタデータ用テーブル
    * @param[in]    originalTable       オリジナルメタデータ用テーブル
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess       正常にフォーマットできました。
    * @retval       上記以外            ストレージへの書き込みに失敗しました。
    *
    * @pre          pTable != nullptr
    * @pre          pOriginalTable != nullptr
    *
    * @details      メタデータストレージをフォーマットします。
    */
    static Result Format(
                      uint32_t bitCount,
                      fs::SubStorage table,
                      fs::SubStorage originalTable
                  ) NN_NOEXCEPT;

    /**
    * @brief        メタデータストレージを拡張します。
    *
    * @param[in]    bitCountOld         拡張前の総ビット数
    * @param[in]    bitCountNew         拡張後の総ビット数
    * @param[in]    table               メタデータ用テーブル
    * @param[in]    originalTable       オリジナルメタデータ用テーブル
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess   正常に拡張できました。
    * @retval       上記以外        ストレージへの書き込みに失敗しました。
    *
    * @pre          bitCountNew >= bitCountOld
    *
    * @details      メタデータストレージを拡張します。
    */
    static Result Expand(
                      uint32_t bitCountOld,
                      uint32_t bitCountNew,
                      fs::SubStorage table,
                      fs::SubStorage originalTable
                  ) NN_NOEXCEPT;

    /**
    * @brief        コンストラクタです。
    *
    * @details      コンストラクタです。
    */
    DuplexBitmap() NN_NOEXCEPT;

    /**
    * @brief        メタデータストレージをマウントします。
    *
    * @param[in]    blockCount          総ビット数
    * @param[in]    table               メタデータ用ストレージ
    * @param[in]    originalTable       オリジナルメタデータ用ストレージ
    *
    * @pre          pTable != nullptr
    * @pre          pOriginalTable != nullptr
    *
    * @details      メタデータストレージをマウントします。
    */
    void Initialize(
             uint32_t bitCount,
             fs::SubStorage table,
             fs::SubStorage originalTable
         ) NN_NOEXCEPT;

    /**
    * @brief        メタデータストレージをアンマウントします。
    *
    * @details      メタデータストレージをアンマウントします。
    */
    void Finalize() NN_NOEXCEPT;

    /**
    * @brief        オリジナルメタデータビットマップから、32 ビット分のビットマップを取得します。
    *
    * @param[out]   outIter         イテレータ
    * @param[in]    index           インデックス
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess   正常に処理が終了しました。
    * @retval       上記以外        ストレージからの読み込みに失敗しました。
    *
    * @pre          マウントしている。
    *
    * @details      オリジナルメタデータビットマップから、32 ビット分のビットマップを取得します。
    */
    Result ReadOriginalBitmap32(Bit32* outIter, uint32_t index) NN_NOEXCEPT
    {
        return ReadBitmap32Impl(outIter, index, &m_OriginalTable);
    }

    /**
    * @brief        オリジナルメタデータビットマップに対してイテレーションを開始します。
    *
    * @param[out]   outIter         イテレータ
    * @param[in]    index           インデックス
    * @param[in]    count           ストレージ
    *
    * @pre          マウントしている。
    * @pre          index + count <= (総ビット数)
    *
    * @details      オリジナルメタデータビットマップに対してイテレーションを開始します。
    */
    void IterateOriginalBegin(
             Iterator* outIter,
             uint32_t index,
             size_t count
         ) const NN_NOEXCEPT;

    /**
    * @brief        オリジナルメタデータビットマップに対してイテレーションを行います。
    *
    * @param[out]   outCountZero        0 が連続している回数
    * @param[out]   outCountOne         1 が連続している回数
    * @param[in]    outIter             イテレータ
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess       正常に処理が終了しました。
    * @retval       上記以外            ストレージからの読み込みに失敗しました。
    *
    * @pre          マウントしている。
    * @pre          outIter->index <= (総ビット数)
    * @post         イテレーションが終了したら outCountZero = 0、outCountOne = 0。
    *
    * @details      オリジナルメタデータビットマップに対してイテレーションを行います。
    */
    Result IterateOriginalNext(
               size_t* outCountZero,
               size_t* outCountOne,
               Iterator* outIter
           ) NN_NOEXCEPT
    {
        return IterateNextImpl(
                   outCountZero,
                   outCountOne,
                   outIter,
                   &m_OriginalTable
               );
    }

    /**
    * @brief        更新側メタデータビットマップから、32 ビット分のビットマップを取得します。
    *
    * @param[out]   outValue            ビットマップ
    * @param[in]    index               インデックス
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess       正常に取得しました。
    * @retval       上記以外            ストレージからの読み込みに失敗しました。
    *
    * @pre          マウントしている。
    *
    * @details      更新側メタデータビットマップから、32 ビット分のビットマップを取得します。
    */
    Result ReadBitmap32(Bit32* outValue, uint32_t index) NN_NOEXCEPT
    {
        return ReadBitmap32Impl(outValue, index, &m_Table);
    }

    /**
    * @brief        メタデータビットマップに対してイテレーションを開始します。
    *
    * @param[out]   outIter             イテレータ
    * @param[in]    index               開始インデックス
    * @param[in]    count               イテレートするビット数
    *
    * @pre          マウントしている。
    * @pre          index + count <= (総ビット数)
    *
    * @details      メタデータビットマップに対してイテレーションを開始します。
    */
    void IterateBegin(
             Iterator* outIter,
             uint32_t index,
             size_t count
         ) const NN_NOEXCEPT
    {
        return IterateOriginalBegin(outIter, index, count);
    }

    /**
    * @brief        続けてイテレーションを行います。
    *
    * @param[out]   outCountZero        0 が連続している回数
    * @param[out]   outCountOne         1 が連続している回数
    * @param[in]    outIter             イテレータ
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess       正常に処理が終了しました。
    * @retval       上記以外            ストレージからの読み込みに失敗しました。
    *
    * @pre          マウントしている。
    * @pre          outIter->index <= (総ビット数)
    * @post         イテレーションが終了したら outCountZero = 0、outCountOne = 0。
    *
    * @details      続けてイテレーションを行います。
    */
    Result IterateNext(
               size_t* outCountZero,
               size_t* outCountOne,
               Iterator* outIter
           ) NN_NOEXCEPT
    {
        return IterateNextImpl(
                   outCountZero,
                   outCountOne,
                   outIter,
                   &m_Table
               );
    }

    /**
    * @brief        指定範囲に対してビットデータを反転します。
    *
    * @param[in]    index           開始インデックス
    * @param[in]    count           反転するビットの数
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess   正常に処理が終了しました。
    * @retval       上記以外        ストレージでの読み書きに失敗しました。
    *
    * @pre          マウントしている。
    * @pre          index + count <= (総ビット数)
    *
    * @details      指定範囲に対してビットデータを反転します。
    */
    Result MarkModified(uint32_t index, size_t count) NN_NOEXCEPT;

    /**
    * @brief        メタデータビットデータをフラッシュします。
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess   正常にフラッシュできました。
    * @retval       上記以外        ストレージへの書き込みに失敗しました。
    *
    * @pre          マウントしている。
    *
    * @details      メタデータビットデータをフラッシュします。
    */
    Result Flush() NN_NOEXCEPT;

    /**
    * @brief        キャッシュを無効化します。
    *
    * @param[in]    bitIndex        開始インデックス
    * @param[in]    bitCount        キャッシュを無効化するビットの数
    *
    * @return       関数の処理結果を返します。
    *
    * @pre          マウントしている。
    *
    * @details      フラッシュせずにキャッシュを無効化します。
    */
    Result Invalidate(
               uint32_t bitIndex,
               size_t bitCount
           ) NN_NOEXCEPT;

private:
    //! フォーマット時のキャッシュサイズ
    static const size_t FormatCacheSize = 64;

    //! イテレート時のキャッシュサイズ
    static const size_t CacheSize = 32;

private:
    /**
    * @brief        ストレージからデータを読み込みします。
    *
    * @param[out]   outBuf          読み込んだデータを格納するバッファ
    * @param[out]   outReadSize     読み込んだバイト数
    * @param[in]    maxBytes        読み込み先のバッファサイズ。
    * @param[in]    index           読み込み開始インデックス(bit)
    * @param[in]    pStorage        ストレージ
    * @param[in]    readOffset      ストレージの読み込みオフセット
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess   正常に処理が終了しました。
    * @retval       上記以外        ストレージからの読み込みに失敗しました。
    *
    * @pre          マウントしている。
    * @pre          outBuf != nullptr
    * @pre          outReadSize != nullptr
    * @pre          pStorage != nullptr
    * @pre          maxBytes % 4 == 0
    *
    * @details      ストレージからデータを読み込みします。
    *               読み込みは 32 ビット単位で行われます。
    *               読込先のバッファサイズは 4 の倍数にしておく必要があります。
    */
    Result ReadBlock(
               uint8_t* outBuf,
               uint32_t* outReadSize,
               size_t maxBytes,
               uint32_t index,
               fs::SubStorage* pStorage
           ) const NN_NOEXCEPT;

    /**
    * @brief        連続しているビット数を計測します。
    *
    * @param[out]   outTotalCount       連続しているビット数
    * @param[out]   outIter             イテレータ
    * @param[out]   outBuf              読み込んだデータを格納しているバッファ
    * @param[out]   outReadIndex        探索済みのデータサイズ
    * @param[out]   outRead             読み込んだデータサイズ
    * @param[in]    isHeadZero          連続している 0 を計測する場合は true、
    *                                   連続している 1 を計測する場合は false
    * @param[in]    pStorage            ストレージ
    * @param[in]    readOffset          ストレージの読み込みオフセット
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess       正常に処理が終了しました。
    * @retval       上記以外            ストレージからの読み込みに失敗しました。
    *
    * @pre          マウントしている。
    * @pre          outTotalCount != nullptr
    * @pre          outIter != nullptr
    * @pre          outBuf != nullptr
    * @pre          outReadIndex != nullptr
    * @pre          outRead != nullptr
    * @pre          pStorage != nullptr
    *
    * @details      連続しているビット数を計測します。
    */
    Result FindBitCount(
               size_t* outTotalCount,
               Iterator* outIter,
               uint8_t* outBuf,
               uint32_t* outReadIndex,
               uint32_t* outRead,
               bool isHeadZero,
               fs::SubStorage* pStorage
           ) const NN_NOEXCEPT;

    /**
    * @brief        二重化ビットマップに対してイテレーションを行います。
    *
    * @param[out]   outCountZero        0 が連続している回数
    * @param[out]   outCountOne         1 が連続している回数
    * @param[in]    pIter               イテレータ
    * @param[in]    index               インデックス
    * @param[in]    pStorage            ストレージ
    * @param[in]    readOffset          オフセット
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess      正常に処理が終了しました。
    * @retval       上記以外           ストレージからの読み込みに失敗しました。
    *
    * @pre          マウントしている。
    * @pre          outZeroCount != nullptr
    * @pre          outOneCount != nullptr
    * @pre          pIter != nullptr
    * @pre          pStorage != nullptr
    * @pre          pIter->index <= (総ビット数)
    * @post         イテレーションが終了したら outCountZero = 0、outCountOne = 0。
    *
    * @details      二重化ビットマップに対してイテレーションを行います。
    */
    Result IterateNextImpl(
               size_t* outCountZero,
               size_t* outCountOne,
               Iterator* pIter,
               fs::SubStorage* pStorage
           ) const NN_NOEXCEPT;

    /**
    * @brief        更新側メタデータビットマップから、32 ビット分のビットマップを取得します。
    *
    * @param[out]   outBitmap           ビットマップ
    * @param[in]    index               インデックス
    * @param[in]    pStorage            ストレージ
    * @param[in]    readOffset          読み込みオフセット
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess       正常に取得できました。
    * @retval       上記以外            ストレージからの読み込みに失敗しました。
    *
    * @pre          マウントしている。
    * @pre          outBitmap != nullptr
    * @pre          pStorage != nullptr
    *
    * @details      更新側メタデータビットマップから、32 ビット分のビットマップを取得します。
    */
    Result ReadBitmap32Impl(
               Bit32* outBitmap,
               uint32_t index,
               fs::SubStorage* pStorage
           ) const NN_NOEXCEPT;

private:
    uint32_t m_BitCount;                    //! 総ビット数
    fs::SubStorage m_Table;                 //! メタデータストレージ
    fs::SubStorage m_OriginalTable;         //! オリジナルメタデータストレージ

private:
    //! テスト用に private を公開します。
    friend class DuplexBitmapTest;
};

}}}
