﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cstring>

#include <nn/fssystem/dbm/fs_DbmParameters.h>
#include <nn/fssystem/dbm/fs_HierarchicalFileTableTemplate.h>
#include <nn/fssystem/dbm/fs_FileSystemTemplate.h>

class DirectoryObjectTestSuite;

namespace nn { namespace fssystem { namespace dbm {

template <typename TDirectoryName, typename TFileName>
class FileSystemObjectTemplate;

/**
* @brief    ディレクトリオブジェクトです。
*
* @tparam   DirectoryName   ディレクトリ名を判定するクラス
* @tparam   FileName        ファイル名を判定するクラス
*
* @details
*   nn::fssystem::dbm::FileSystemBase::OpenDirectory で初期化します。Finalize は不要です。
*
*   ディレクトリやファイルの列挙には、以下の関数を用います。
*   - FindOpen
*   - FindNextDirectory
*   - FindNextFile
*
*   ディレクトリとファイルの列挙はそれぞれ別々に行います。
*
*   制限事項
*
*   ディレクトリの列挙中に以下の操作をした場合、動作は不定になります。
*
*   - 列挙対象のディレクトリに対する、子ディレクトリの作成
*   - 列挙対象のディレクトリに対する、ファイルの作成
*   - 列挙済みではない子ディレクトリの削除
*   - 列挙済みではないファイルの削除
*
* 上記の操作後、再びFindOpenしなおす必要があります。
*/
template <typename TDirectoryName, typename TFileName>
class DirectoryObjectTemplate
{
    NN_DISALLOW_COPY(DirectoryObjectTemplate);

public:
    /**
    * @brief        コンストラクタです。
    *
    * @details      コンストラクタです。
    */
    DirectoryObjectTemplate() NN_NOEXCEPT;

    /**
    * @brief        ディレクトリに格納されているファイル数を取得します。
    *
    * @param[out]   outFileCounts                   ファイル数
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess                   正常にファイルが取得できました。
    * @retval       ResultNotInitialized            初期化されていません。
    * @retval       ResultIncompatiblePath          内部データに問題があります。
    * @retval       ResultDirectoryNotFound         内部データに問題があります。
    * @retval       ResultInvalidOffset             内部データに問題があります。
    * @retval       ResultDatabaseCorrupted         内部データに問題があります。
    * @retval       上記以外                        ストレージからのデータ読み込みに失敗しました。
    *
    * @pre          マウントしている。
    *
    * @details      ディレクトリに格納されているファイル数を取得します。
    */
    Result CountFile(uint32_t* outFileCounts) const NN_NOEXCEPT;

    /**
    * @brief        ディレクトリに格納されているディレクトリ数を取得します。
    *
    * @param[out]   outDirectoryCounts              ディレクトリ数
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess                   正常にディレクトリ数が取得できました。
    * @retval       ResultNotInitialized            初期化されていません。
    * @retval       ResultIncompatiblePath          内部データに問題があります。
    * @retval       ResultDirectoryNotFound         内部データに問題があります。
    * @retval       ResultInvalidOffset             内部データに問題があります。
    * @retval       ResultDatabaseCorrupted         内部データに問題があります。
    * @retval       上記以外                        ストレージからのデータ読み込みに失敗しました。
    *
    * @pre          マウントしている。
    *
    * @details      ディレクトリに格納されているディレクトリ数を取得します。
    */
    Result CountDirectory(uint32_t* outDirectoryCounts) const NN_NOEXCEPT;

    /**
    * @brief        子ファイル、ディレクトリに対するイテレーションを開始します。
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess                   正常に処理が終了しました。
    * @retval       ResultNotInitialized            初期化されていません。
    * @retval       ResultIncompatiblePath          内部データに問題があります。
    * @retval       ResultDirectoryNotFound         内部データに問題があります。
    * @retval       ResultInvalidOffset             内部データに問題があります。
    * @retval       ResultDatabaseCorrupted         内部データに問題があります。
    * @retval       上記以外                        ストレージからのデータ読み込みに失敗しました。
    *
    * @pre         マウントしている。
    *
    * @details     子ファイル、ディレクトリに対するイテレーションを開始します。
    */
    Result FindOpen() NN_NOEXCEPT;

    /**
    * @brief        子ディレクトリに対するイテレーションを行います。
    *
    * @param[out]   outDirectoryName        ディレクトリ名
    * @param[out]   outFinished             イテレーションが終了したか
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess                   正常に処理が終了しました。
    * @retval       ResultNotInitialized            初期化されていません。
    * @retval       ResultInvalidOffset             内部データに問題があります。
    * @retval       ResultDatabaseCorrupted         内部データに問題があります。
    * @retval       上記以外                        ストレージからのデータ読み込みに失敗しました。
    *
    * @pre         マウントしている。
    *
    * @details      子ディレクトリに対するイテレーションを行います。
    */
    Result FindNextDirectory(
               TDirectoryName* outDirectoryName,
               bool* outFinished
           ) NN_NOEXCEPT;

    /**
    * @brief        子ファイルに対するイテレーションを行います。
    *
    * @param[out]   outFileSize ファイルサイズ
    * @param[out]   outFileName ファイル名
    * @param[out]   outFinished イテレーションが終了したか
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess                   正常にファイルが取得できました。
    * @retval       ResultNotInitialized            初期化されていません。
    * @retval       ResultInvalidOffset             内部データに問題があります。
    * @retval       ResultDatabaseCorrupted         内部データに問題があります。
    * @retval       上記以外                        ストレージからのデータ読み込みに失敗しました。
    *
    * @pre          マウントしている。
    *
    * @details      子ファイルに対するイテレーションを行います。
    */
    Result FindNextFile(
               int64_t* outFileSize,
               TFileName* outFileName,
               bool* outFinished
           ) NN_NOEXCEPT;

    /**
    * @brief        エントリー更新の通知をします。
    *
    * @param[out]   outFinished             イテレーションが終了したか。
    * @param[in]    index                   エントリー ID
    * @param[in]    isFile                  ファイルなら true, ディレクトリなら false
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess                   正常にエントリーが取得できました。
    * @retval       ResultNotInitialized            初期化されていません。
    * @retval       ResultInvalidOffset             iter が範囲外です。
    * @retval       ResultInvalidOffset             内部で不明な問題が発生しました。
    * @retval       ResultDatabaseCorrupted         内部データに問題があります。
    * @retval       上記以外                        ストレージからのデータ読み込みに失敗しました。
    *
    * @pre          マウントしている。
    *
    * @details      エントリー更新の通知をします。
    */
    Result Notify(
               bool* outFinished,
               int64_t id,
               bool isFile
           ) NN_NOEXCEPT;

private:
    typedef FileSystemObjectTemplate<TDirectoryName, TFileName> FileSystemObjectTempl;

private:
    /**
    * @brief        ディレクトリオブジェクトを初期化します。
    *
    * @param[in]    pFileSystem         ファイルシステム
    * @param[in]    directoryKey        ディレクトリキー
    *
    * @pre          pFileSystem != nullptr
    *
    * @details      ディレクトリオブジェクトを初期化します。
    */
    void Initialize(
             FileSystemObjectTempl* pFileSystem,
             typename FileSystemObjectTempl::DirectoryKey& directoryKey
         ) NN_NOEXCEPT;

    /**
    * @brief        ディレクトリ情報を取得します。
    *
    * @return       ディレクトリ情報。
    *
    * @details      ディレクトリ情報を取得します。
    */
    inline detail::DirectoryInfo& GetDirectoryInfo() NN_NOEXCEPT
    {
        return m_DirectoryInfo;
    }

private:
    FileSystemObjectTempl* m_pFileSystem;                           //! ファイルシステム
    detail::DirectoryInfo m_DirectoryInfo;                          //! ディレクトリ情報
    typename FileSystemObjectTempl::DirectoryKey m_DirectoryKey;    //! ディレクトリキー
    typename FileSystemObjectTempl::FindIndex m_Iterator;           //! イテレータ

private:
    //! ファイルシステムが初期化処理を呼び出せるようにします。
    friend class FileSystemObjectTemplate<TDirectoryName, TFileName>;

private:
    //! テスト用に private を公開します。
    friend class DirectoryObjectTestSuite;
};

}}}
