﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <memory>

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>

#include <nn/fs/detail/fs_Newable.h>
#include <nn/fs/fs_SaveDataPrivate.h>
#include <nn/fs/fs_GameCard.h>
#include <nn/fs/fs_SdCardPrivate.h>
#include <nn/fat/fat_FatFileSystem.h>
#include <nn/fssystem/fs_SaveDataFileSystem.h>
#include <nn/fssystem/save/fs_IntegritySaveDataFileSystem.h>
#include <nn/fssystem/save/fs_JournalIntegritySaveDataFileSystem.h>

namespace nn { namespace fs {
    enum class BisPartitionId;
    class IStorage;
    struct FileSystemProxyErrorInfo;
}}

namespace nn { namespace fs { namespace fsa {
    class IFileSystem; // TODO: 別インターフェイスに置き換え
}}}

namespace nn { namespace fssystem {
    class NcaReader;
    class NcaFsHeaderReader;
    class ISaveDataExtraDataAccessor;
    class ISaveDataFileSystemCacheManager;
}}

namespace nn { namespace fssystem { namespace save {
    struct IntegritySaveDataParameters;
    struct JournalIntegritySaveDataParameters;
}}}

namespace nn { namespace fssrv { namespace detail {
    class SaveDataExtender;
    class FatFileSystemCacheManager;
}}}

namespace nn { namespace fssrv { namespace fscreator {

    class IRomFileSystemCreator
    {
    public:
        virtual ~IRomFileSystemCreator() NN_NOEXCEPT {}
        virtual Result Create(std::shared_ptr<nn::fs::fsa::IFileSystem>* outValue, std::shared_ptr<fs::IStorage> storage) NN_NOEXCEPT = 0;
    };

    class IPartitionFileSystemCreator
    {
    public:
        virtual ~IPartitionFileSystemCreator() NN_NOEXCEPT {}
        virtual Result Create(std::shared_ptr<nn::fs::fsa::IFileSystem>* outValue, std::shared_ptr<fs::IStorage> storage) NN_NOEXCEPT = 0;
    };

    class IStorageOnNcaCreator
    {
    public:
        virtual ~IStorageOnNcaCreator() NN_NOEXCEPT {}
        virtual Result Create(std::shared_ptr<fs::IStorage>* outValue, fssystem::NcaFsHeaderReader* outFsHeaderReader, std::shared_ptr<fssystem::NcaReader> ncaReader, int index, bool verifyHeaderSign2) NN_NOEXCEPT = 0;
        virtual Result CreateWithPatch(std::shared_ptr<fs::IStorage>* outValue, fssystem::NcaFsHeaderReader* outFsHeaderReader, std::shared_ptr<fssystem::NcaReader> originalNcaReader, std::shared_ptr<fssystem::NcaReader> currentNcaReader, int index, bool verifyHeaderSign2) NN_NOEXCEPT = 0;
        virtual Result CreateNcaReader(std::shared_ptr<fssystem::NcaReader>* outValue, std::shared_ptr<fs::IStorage> storage) NN_NOEXCEPT = 0;
        virtual Result VerifyAcid(fs::fsa::IFileSystem* pFileSystem, fssystem::NcaReader* ncaReader) NN_NOEXCEPT = 0;
    };

    class IFatFileSystemCreator
    {
    public:
        virtual ~IFatFileSystemCreator() NN_NOEXCEPT {}
        virtual Result Create(std::shared_ptr<nn::fs::fsa::IFileSystem>* outValue, std::shared_ptr<fs::IStorage> storage, nn::fat::FatAttribute* pFatAttribute, std::unique_ptr<nn::fat::FatErrorInfoSetter>&& pFatErrorInfoSetter, nn::Result resultInvalidFatFormat, nn::Result resultUsableSpaceNotEnough) NN_NOEXCEPT = 0;
        virtual Result Format(std::shared_ptr<fs::IStorage> storage, nn::fat::FatAttribute* pFatAttribute, nn::fat::FatFormatParam* pFatFormatParam, std::unique_ptr<nn::fat::FatErrorInfoSetter>&& pFatErrorInfoSetter, nn::Result resultInvalidFatFormat, nn::Result resultUsableSpaceNotEnough) NN_NOEXCEPT = 0;
    };

    class IHostFileSystemCreator
    {
    public:
        virtual ~IHostFileSystemCreator() NN_NOEXCEPT {}
        virtual Result Create(std::shared_ptr<nn::fs::fsa::IFileSystem>* outValue) NN_NOEXCEPT = 0;
        virtual Result Create(std::shared_ptr<nn::fs::fsa::IFileSystem>* outValue, const char* path) NN_NOEXCEPT = 0;
    };

    class ITargetManagerFileSystemCreator // TODO: IHostFileSystemCreator に一本化
    {
    public:
        virtual ~ITargetManagerFileSystemCreator() NN_NOEXCEPT {}
        virtual Result Create(std::shared_ptr<nn::fs::fsa::IFileSystem>* outValue) NN_NOEXCEPT = 0;
    };

    class ISubDirectoryFileSystemCreator
    {
    public:
        virtual ~ISubDirectoryFileSystemCreator() NN_NOEXCEPT {}
        virtual Result Create(std::shared_ptr<nn::fs::fsa::IFileSystem>* outValue, std::shared_ptr<nn::fs::fsa::IFileSystem> fileSystem, const char* path) NN_NOEXCEPT = 0;
        virtual Result Create(std::shared_ptr<nn::fs::fsa::IFileSystem>* outValue, std::shared_ptr<nn::fs::fsa::IFileSystem> fileSystem, const char* path, bool isUncPreserved) NN_NOEXCEPT = 0;
    };

    class IBuiltInStorageCreator
    {
    public:
        virtual ~IBuiltInStorageCreator() NN_NOEXCEPT {}
        virtual Result Create(std::shared_ptr<fs::IStorage>* outValue, nn::fs::BisPartitionId id) NN_NOEXCEPT = 0;
        virtual Result InvalidateCache() NN_NOEXCEPT = 0;
        virtual Result LoadAndVerifySignedSystemPartition(std::unique_ptr<nn::fs::IStorage>* outValue, std::unique_ptr<nn::fs::IStorage>&& baseStorage) NN_NOEXCEPT = 0;
    };

    class IBuiltInStorageFileSystemCreator
    {
    public:
        virtual ~IBuiltInStorageFileSystemCreator() NN_NOEXCEPT {}
        virtual Result Create(std::shared_ptr<fs::fsa::IFileSystem>* pOutFileSystem, const char* rootPath, nn::fs::BisPartitionId id) NN_NOEXCEPT = 0;
        virtual bool IsSignedSystemPartitionOnSdCardValid() NN_NOEXCEPT = 0;
        virtual void SetSdCardPortReady() NN_NOEXCEPT = 0;
        virtual void Initialize(fs::FileSystemProxyErrorInfo* pErrorInfo, os::Mutex* pErrorInfoMutex) NN_NOEXCEPT = 0;
        virtual Result SetBisRoot(nn::fs::BisPartitionId id, const char* rootPath) NN_NOEXCEPT = 0;
    };

    class ISdCardProxyFileSystemCreator
    {
    public:
        virtual ~ISdCardProxyFileSystemCreator() NN_NOEXCEPT {}
        virtual Result Create(std::shared_ptr<fs::fsa::IFileSystem>* pOutFileSystem) NN_NOEXCEPT = 0;
        virtual Result Format(bool isUncache) NN_NOEXCEPT = 0;
        virtual Result Format() NN_NOEXCEPT = 0;
        virtual void Initialize(fs::FileSystemProxyErrorInfo* pErrorInfo, os::Mutex* pErrorInfoMutex) NN_NOEXCEPT = 0;
    };

    class ISdStorageCreator
    {
    public:
        virtual ~ISdStorageCreator() NN_NOEXCEPT {}
        virtual Result Create(std::shared_ptr<fs::IStorage>* outValue) NN_NOEXCEPT = 0;
    };

    class ISaveDataFileSystemCreator
    {
    public:
        virtual ~ISaveDataFileSystemCreator() NN_NOEXCEPT {}
        virtual Result CreateRaw(std::shared_ptr<nn::fs::fsa::IFile>* outValue, std::shared_ptr<nn::fs::fsa::IFileSystem> fileSystem, nn::fs::SaveDataId saveDataId, fs::OpenMode mode) NN_NOEXCEPT = 0;
        virtual Result Create(std::shared_ptr<nn::fs::fsa::IFileSystem>* outValue, std::shared_ptr<nn::fssystem::ISaveDataExtraDataAccessor>* outExtraDataAccessor, fssystem::ISaveDataFileSystemCacheManager* pCacheManager, std::shared_ptr<nn::fs::fsa::IFileSystem> fileSystem, nn::fs::SaveDataId saveDataId, bool isAllowedDirectorySaveData, bool isDeviceUniqueMac, nn::fs::SaveDataType saveDataType, fssystem::SaveDataCommitTimeStampGetter timeStampGetter) NN_NOEXCEPT = 0;
        virtual Result CreateExtraDataAccessor(std::shared_ptr<fssystem::ISaveDataExtraDataAccessor>* outValue, std::shared_ptr<nn::fs::IStorage> storage, bool isDeviceUniqueMac, bool isTemporaryFs) NN_NOEXCEPT = 0;
        virtual Result CreateInternalStorage(std::shared_ptr<nn::fs::fsa::IFileSystem>* outValue, std::shared_ptr<nn::fssystem::ISaveDataExtraDataAccessor>* outExtraDataAccessor, std::shared_ptr<nn::fssystem::SaveDataFileSystem>&& saveDataFileSystem, std::shared_ptr<nn::fs::fsa::IFileSystem> fileSystem, nn::fs::SaveDataId saveDataId, bool isDeviceUniqueMac, bool isTemporaryFs, fssystem::SaveDataCommitTimeStampGetter timeStampGetter) NN_NOEXCEPT = 0;
        virtual Result RecoverMasterHeader(std::shared_ptr<nn::fs::fsa::IFileSystem> fileSystem, nn::fs::SaveDataId saveDataId, fssystem::IBufferManager* pBufferManager, bool isDeviceUniqueMac) NN_NOEXCEPT = 0;
        virtual Result UpdateMac(std::shared_ptr<nn::fs::fsa::IFileSystem> fileSystem, nn::fs::SaveDataId saveDataId, bool isDeviceUniqueMac) NN_NOEXCEPT = 0;

        virtual Result Format(
            fs::SubStorage baseStorage,
            size_t sizeBlock,
            int countExpandMax,
            const fssystem::save::HierarchicalDuplexStorageControlArea::InputParam& paramDuplex,
            const fssystem::save::HierarchicalIntegrityVerificationStorageControlArea::InputParam& paramIntegrity,
            uint32_t countDataBlock,
            uint32_t countReservedBlock,
            fssystem::IBufferManager* pBufferManager,
            bool isDeviceUniqueMac,
            const nn::fs::SaveDataHashSalt& hashSalt
        ) NN_NOEXCEPT = 0;

        virtual Result FormatAsIntegritySaveData(
            fs::SubStorage baseStorage,
            size_t sizeBlock,
            const fssystem::save::HierarchicalIntegrityVerificationStorageControlArea::InputParam& paramIntegrity,
            uint32_t countDataBlock,
            fssystem::IBufferManager* pBufferManager,
            bool isDeviceUniqueMac
        ) NN_NOEXCEPT = 0;

        virtual Result ExtractSaveDataParameters(
            fssystem::save::JournalIntegritySaveDataParameters* outParameters,
            fs::IStorage* pSaveDataStorage,
            bool isDeviceUniqueMac
        ) NN_NOEXCEPT = 0;

        virtual Result ExtendSaveData(
            detail::SaveDataExtender* pExtender,
            fs::SubStorage saveDataStorage,
            fs::SubStorage logStorage,
            bool isDeviceUniqueMac
        ) NN_NOEXCEPT = 0;

        virtual Result ReadExtraData(
            nn::fs::SaveDataExtraData* outData,
            fs::SubStorage baseStorage,
            fssystem::IBufferManager* pBufferManager,
            bool isDeviceUniqueMac,
            bool isTemporaryFs
        ) NN_NOEXCEPT = 0;

        virtual void SetMacGenerationSeed(const void* pSeed, size_t seedSize) NN_NOEXCEPT = 0;
    };

    class IApplicationTemporaryFileSystemCreator
    {
    public:
        virtual ~IApplicationTemporaryFileSystemCreator() NN_NOEXCEPT {}
        virtual Result Create(std::shared_ptr<nn::fs::fsa::IFileSystem>* outValue, std::shared_ptr<nn::fssystem::ISaveDataExtraDataAccessor>* outExtraDataAccessor, std::shared_ptr<nn::fs::fsa::IFileSystem> fileSystem, nn::fs::SaveDataId saveDataId, bool isDeviceUniqueMac) NN_NOEXCEPT = 0;
        virtual Result CreateExtraDataAccessor(std::shared_ptr<fssystem::ISaveDataExtraDataAccessor>* outValue, std::shared_ptr<nn::fs::IStorage> storage, bool isDeviceUniqueMac) NN_NOEXCEPT = 0;
        virtual Result Format(
            fs::SubStorage baseStorage,
            size_t sizeBlock,
            const fssystem::save::HierarchicalIntegrityVerificationStorageControlArea::InputParam& paramIntegrity,
            uint32_t countDataBlock,
            fssystem::IBufferManager* pBufferManager,
            bool isDeviceUniqueMac
        ) NN_NOEXCEPT = 0;

        virtual Result ReadExtraData(
            fssystem::save::IntegritySaveDataFileSystem::ExtraData* outData,
            fs::SubStorage baseStorage,
            fssystem::IBufferManager* pBufferManager,
            bool isDeviceUniqueMac
        ) NN_NOEXCEPT = 0;

        virtual void SetMacGenerationSeed(const void* pSeed, size_t seedSize) NN_NOEXCEPT = 0;
    };

    class IGameCardStorageCreator
    {
    public:
        virtual ~IGameCardStorageCreator() NN_NOEXCEPT {}
        virtual Result CreateReadOnly      (nn::fs::GameCardHandle handle, std::shared_ptr<fs::IStorage>* outValue) NN_NOEXCEPT = 0;
        virtual Result CreateSecureReadOnly(nn::fs::GameCardHandle handle, std::shared_ptr<fs::IStorage>* outValue) NN_NOEXCEPT = 0;
        virtual Result CreateWriteOnly     (nn::fs::GameCardHandle handle, std::shared_ptr<fs::IStorage>* outValue) NN_NOEXCEPT = 0;
    };

    class IGameCardFileSystemCreator
    {
    public:
        virtual ~IGameCardFileSystemCreator() NN_NOEXCEPT {}
        virtual Result Create(std::shared_ptr<nn::fs::fsa::IFileSystem>* outValue, nn::fs::GameCardHandle handle, nn::fs::GameCardPartition partition) NN_NOEXCEPT = 0;
    };

    class IEncryptedFileSystemCreator
    {
    public:
        enum class KeyId
        {
            SdCardSystemSave,
            SdCardContentStorage,

            Count,
        };

        static const int KeyIdCount = static_cast<int>(KeyId::Count);

    public:
        virtual ~IEncryptedFileSystemCreator() NN_NOEXCEPT {}
        virtual Result Create(std::shared_ptr<nn::fs::fsa::IFileSystem>* outValue, std::shared_ptr<nn::fs::fsa::IFileSystem> pFileSystem, KeyId id, const nn::fs::EncryptionSeed& seed) NN_NOEXCEPT = 0;
    };

    class IMemoryStorageCreator
    {
    public:
        enum class MemoryStorageId
        {
            UserPartition,
            SignedSystemPartitionRaw,
            SystemPartition,

            Count,
        };

    public:
        virtual ~IMemoryStorageCreator() NN_NOEXCEPT {}
        virtual Result Create(std::shared_ptr<fs::IStorage>* outStorage, char** outRawBuffer, MemoryStorageId id) NN_NOEXCEPT = 0;
        virtual Result RegisterMemory(MemoryStorageId id, char* buffer, size_t bufferSize) NN_NOEXCEPT = 0;
    };

    struct FileSystemCreatorInterfaces
    {
        IRomFileSystemCreator* romFsCreator;
        IPartitionFileSystemCreator* partitionFsCreator;
        IStorageOnNcaCreator* storageOnNcaCreator;
        IFatFileSystemCreator* fatFsCreator;
        IHostFileSystemCreator* hostFsCreator;
        ITargetManagerFileSystemCreator* targetFsCreator;
        ISubDirectoryFileSystemCreator* subDirFsCreator;
        IBuiltInStorageCreator* builtInStorageCreator;
        ISdStorageCreator* sdStorageCreator;
        ISaveDataFileSystemCreator* saveDataFsCreator;
        IGameCardStorageCreator* gameCardStorageCreator;
        IGameCardFileSystemCreator* gameCardFileSystemCreator;
        IEncryptedFileSystemCreator* encryptedFileSystemCreator;
        IMemoryStorageCreator* memoryStorageCreator;
        IBuiltInStorageFileSystemCreator* bisfsCreator;
        ISdCardProxyFileSystemCreator* sdProxyFsCreator;
    };
    NN_STATIC_ASSERT(std::is_pod<FileSystemCreatorInterfaces>::value);

}}}
