﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <memory>
#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/fs/fs_Result.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/fs/detail/fs_Newable.h>
#include <nn/fs/fs_IStorage.h>

namespace nn { namespace fs {


class SubStorage : public fs::IStorage, public nn::fs::detail::Newable
{
public:

    /**
    * @brief        デフォルトコンストラクタ
    *
    * @details      基となるストレージが設定されるまで Read、Write、Flush、GetSize はできない
    */
    SubStorage() NN_NOEXCEPT
    : m_pBaseStorage(nullptr),
      m_Offset(0),
      m_Size(0),
      m_IsResizable(false)
    {
    }

    /**
    * @brief        コピーコンストラクタ
    *
    * @param[in]    subStorage      基となるサブストレージ
    *
    * @details      コピーコンストラクタ
    */
    SubStorage(const SubStorage& subStorage) NN_NOEXCEPT
    : m_pBaseStorage(subStorage.m_pBaseStorage),
      m_Offset(subStorage.m_Offset),
      m_Size(subStorage.m_Size),
      m_IsResizable(subStorage.m_IsResizable)
    {
    }

    /**
    * @brief        IStorage を基としたコンストラクタ
    *
    * @param[in]    pBaseStorage    基となるストレージ
    * @param[in]    offset          サブストレージの先頭へのオフセット
    * @param[in]    size            サブストレージのサイズ
    *
    * @pre          pBaseStorage != nullptr
    * @pre          0 <= offset
    * @pre          0 <= size
    *
    * @details      IStorage を基としたコンストラクタ
    */
    SubStorage(IStorage* pBaseStorage, int64_t offset, int64_t size) NN_NOEXCEPT
    : m_pBaseStorage(pBaseStorage),
      m_Offset(offset),
      m_Size(size),
      m_IsResizable(false)
    {
        NN_SDK_REQUIRES_NOT_NULL(pBaseStorage);
        NN_SDK_REQUIRES_LESS_EQUAL(0, offset);
        NN_SDK_REQUIRES_LESS_EQUAL(0, size);
    }

    /**
    * @brief        サブストレージを基としたコンストラクタ
    *
    * @param[in]    pSubStorage     基となるサブストレージ
    * @param[in]    offset          基となるサブストレージのオフセットから
    *                               新しく作成するサブストレージの先頭へのオフセット
    * @param[in]    size            新しく作成するサブストレージのサイズ
    *
    * @pre          pSubStorage != nullptr
    * @pre          0 <= offset
    * @pre          0 <= size
    * @pre          pSubStorage が基となるストレージを持っている
    * @pre          新しく作成するサブストレージの範囲が pSubStorage の範囲内である
    *
    * @details      サブストレージを基としたコンストラクタ
    */
    SubStorage(SubStorage* pSubStorage, int64_t offset, int64_t size) NN_NOEXCEPT
    : m_pBaseStorage(pSubStorage->m_pBaseStorage),
      m_Offset(offset + pSubStorage->m_Offset),
      m_Size(size),
      m_IsResizable(false)
    {
        NN_SDK_REQUIRES_LESS_EQUAL(0, offset);
        NN_SDK_REQUIRES_LESS_EQUAL(0, size);
        NN_SDK_REQUIRES(pSubStorage->IsValid());
        NN_SDK_REQUIRES_GREATER_EQUAL(pSubStorage->m_Size, offset + size);
    }



    /**
    * @brief        shared_ptr を基としたコンストラクタ
    */
    SubStorage(std::shared_ptr<IStorage> pBaseStorage, int64_t offset, int64_t size) NN_NOEXCEPT
    : m_pBaseStorage(pBaseStorage.get()),
      m_Offset(offset),
      m_Size(size),
      m_IsResizable(false),
      m_pSharedBaseStorage(pBaseStorage)
    {
        NN_SDK_REQUIRES_NOT_NULL(pBaseStorage);
        NN_SDK_REQUIRES_LESS_EQUAL(0, offset);
        NN_SDK_REQUIRES_LESS_EQUAL(0, size);
    }

    /**
    * @brief        代入演算子
    *
    * @param[in]    subStorage      基となるサブストレージ
    *
    * @return       代入されたサブストレージ
    *
    * @details      代入演算子
    */
    SubStorage& operator=(const SubStorage& subStorage) NN_NOEXCEPT
    {
        if( this != &subStorage )
        {
            m_pBaseStorage = subStorage.m_pBaseStorage;
            m_Offset = subStorage.m_Offset;
            m_Size = subStorage.m_Size;
            m_IsResizable = subStorage.m_IsResizable;
        }
        return *this;
    }

    virtual Result Read(int64_t offset, void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_RESULT_THROW_UNLESS(IsValid(), nn::fs::ResultNotInitialized());
        if( size == 0 )
        {
            NN_RESULT_SUCCESS;
        }
        if( buffer == nullptr )
        {
            return nn::fs::ResultNullptrArgument();
        }
        if( ! IStorage::CheckAccessRange(offset, size, m_Size) )
        {
            return nn::fs::ResultOutOfRange();
        }

        return m_pBaseStorage->Read(m_Offset + offset, buffer, size);
    }

    virtual Result Write(int64_t offset, const void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_RESULT_THROW_UNLESS(IsValid(), nn::fs::ResultNotInitialized());
        if( size == 0 )
        {
            NN_RESULT_SUCCESS;
        }
        if( buffer == nullptr )
        {
            return nn::fs::ResultNullptrArgument();
        }
        if( ! IStorage::CheckAccessRange(offset, size, m_Size) )
        {
            return nn::fs::ResultOutOfRange();
        }

        return m_pBaseStorage->Write(m_Offset + offset, buffer, size);
    }

    virtual Result Flush() NN_NOEXCEPT NN_OVERRIDE
    {
        NN_RESULT_THROW_UNLESS(IsValid(), nn::fs::ResultNotInitialized());
        return m_pBaseStorage->Flush();
    }

    virtual Result GetSize(int64_t* outValue) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_RESULT_THROW_UNLESS(IsValid(), nn::fs::ResultNotInitialized());
        *outValue = m_Size;
        NN_RESULT_SUCCESS;
    }

    virtual Result SetSize(int64_t value) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_RESULT_THROW_UNLESS(IsValid(), nn::fs::ResultNotInitialized());

        if (!m_IsResizable)
        {
            return fs::ResultUnsupportedOperation();
        }

        NN_RESULT_THROW_UNLESS(CheckOffsetAndSize(m_Offset, value), nn::fs::ResultInvalidSize());

        int64_t size;
        NN_RESULT_DO(m_pBaseStorage->GetSize(&size));

        if (size == m_Offset + m_Size)
        {
            NN_RESULT_DO(m_pBaseStorage->SetSize(m_Offset + value));
            m_Size = value;
        }
        else
        {
            return fs::ResultUnsupportedOperation();
        }

        NN_RESULT_SUCCESS;
    }

    virtual Result OperateRange(
        void* outBuffer,
        size_t outBufferSize,
        OperationId operationId,
        int64_t offset,
        int64_t size,
        const void* inBuffer,
        size_t inBufferSize) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_RESULT_THROW_UNLESS(IsValid(), nn::fs::ResultNotInitialized());
        if( size == 0 )
        {
            NN_RESULT_SUCCESS;
        }
        if( ! IStorage::CheckOffsetAndSize(offset, size) )
        {
            return nn::fs::ResultOutOfRange();
        }

        return m_pBaseStorage->OperateRange(
            outBuffer,
            outBufferSize,
            operationId,
            m_Offset + offset,
            size,
            inBuffer,
            inBufferSize);
    }

    using IStorage::OperateRange;

    void SetResizable(bool isResizable) NN_NOEXCEPT
    {
        m_IsResizable = isResizable;
    }

private:
    bool IsValid() const NN_NOEXCEPT
    {
        return m_pBaseStorage != nullptr;
    }

private:
    fs::IStorage* m_pBaseStorage;
    int64_t m_Offset;
    int64_t m_Size;
    bool m_IsResizable;

    std::shared_ptr<IStorage> m_pSharedBaseStorage;
};

}}
