﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
*   @file
*   @brief  Rights ID 関連 API の宣言
*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/ncm/ncm_ProgramId.h>
#include <nn/ncm/ncm_StorageId.h>
#include <nn/spl/spl_Types.h>

namespace nn { namespace fs {

//! @name Rights ID 関連 API
//! @{

union RightsId
{
    uint8_t data[16];
    Bit64   data8[2]; // アライン制御のため
};
NN_STATIC_ASSERT(std::is_pod<RightsId>::value);
NN_STATIC_ASSERT(sizeof(RightsId) == 16);

//--------------------------------------------------------------------------
/**
*   @brief      指定したコンテンツから Rights ID を取得します。
*
*   @param[out] pOutValue  Rights ID の格納先
*   @param[in]  programId  コンテンツのプログラム ID
*   @param[in]  storageId  コンテンツのストレージ ID
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess               成功しました。
*
*/
Result GetRightsId(RightsId* pOutValue, nn::ncm::ProgramId programId, nn::ncm::StorageId storageId) NN_NOEXCEPT;

//--------------------------------------------------------------------------
/**
*   @brief      指定したコンテンツから Rights ID を取得します。
*
*   @param[out] pOutValue  Rights ID の格納先
*   @param[in]  path       コンテンツのパス
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess               成功しました。
*
*   @pre
*       - pOutValue が有効なメモリを指している。
*       - path が @link entry_path 有効なパス @endlink である。
*
*/
Result GetRightsId(RightsId* pOutValue, const char* path) NN_NOEXCEPT;

//--------------------------------------------------------------------------
/**
*   @brief      指定したコンテンツから Rights ID と鍵世代を取得します。
*
*   @param[out] pOutRightsId      Rights ID の格納先
*   @param[out] pOutKeyGeneration 鍵世代の格納先
*   @param[in]  path              コンテンツのパス
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess               成功しました。
*
*/
Result GetRightsId(RightsId* pOutRightsId, uint8_t* pOutKeyGeneration, const char* path) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
*   @brief      指定した Rights ID に外部鍵を紐づけます。
*
*   @param[in]  rightsId   Rights ID
*   @param[in]  accessKey  rightsId に紐づける外部鍵
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess               成功しました。
*
*/
Result RegisterExternalKey(const RightsId& rightsId, const nn::spl::AccessKey& accessKey) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
*   @brief      Rights ID と外部鍵の紐づけを全て削除します。
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess               成功しました。
*
*/
Result UnregisterAllExternalKey() NN_NOEXCEPT;


//! @}

}} // namespace nn::fs
