﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
*   @file
*   @brief  ProgramRegistry 関連のファイルシステム API の宣言
*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>

namespace nn { namespace fs {

//! @name ProgramRegistry 関連 API
//! @{

//--------------------------------------------------------------------------
/**
*   @brief      ファイルシステムサーバにプロセスに紐づくプログラム情報を登録します。
*
*   @param[in]  processId            プロセス ID
*   @param[in]  programId            プログラム ID
*   @param[in]  storageId            ストレージ ID
*   @param[in]  fsAccessControlData  FS アクセス制御データ
*   @param[in]  dataSize             FS アクセス制御データのサイズ
*   @param[in]  fsAccessControlDesc  FS アクセス制御ディスクリプタ
*   @param[in]  descSize             FS アクセス制御ディスクリプタのサイズ
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess                   成功しました。
*   @retval     ResultAllocationMemoryFailed    メモリが不足しています。
*/
Result RegisterProgram(uint64_t processId, uint64_t programId, uint8_t storageId, const void* fsAccessControlData, size_t dataSize, const void* fsAccessControlDesc, size_t descSize) NN_NOEXCEPT;

//--------------------------------------------------------------------------
/**
*   @brief      ファイルシステムサーバのプロセスに紐づくプログラム情報を削除します。
*
*   @param[in]  processId            プロセス ID
*
*   @return     なし
*/
void UnregisterProgram(uint64_t processId) NN_NOEXCEPT;

//--------------------------------------------------------------------------
/**
*   @brief      ファイルシステムサーバのプログラム署名検証の有効化・無効化を行います。
*
*   @param[in]  isEnabled            true で有効化、false で無効化
*
*   @return     なし
*/
void SetEnabledProgramVerification(bool isEnabled) NN_NOEXCEPT;

//! @}

}} // namespace nn::fs
