﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>

namespace nn { namespace fs {

/**
*   @brief      MMC の Speed Mode の種類を示す列挙型です。
*/
enum MmcSpeedMode {
    MmcSpeedMode_Identification,    //!< Identification
    MmcSpeedMode_LegacySpeed,       //!< Legacy Speed
    MmcSpeedMode_HighSpeed,         //!< High Speed
    MmcSpeedMode_Hs200,             //!< HS200
    MmcSpeedMode_Hs400,             //!< HS400
    MmcSpeedMode_Unknown            //!< 不明な Speed Mode です。
};

/**
*   @brief      MMC パーティションの識別子です。
*/
enum class MmcPartition
{
    UserData,
    BootPartition1,
    BootPartition2
};

/**
 * @brief   MMC の CID のサイズを表す定数です。
 */
const size_t MmcCidSize = 16;


/**
 * @brief   MMC の Extended CSD のサイズを表す定数です。
 */
const size_t MmcExtendedCsdSize = 512;

/**
 * @brief   MMC の Extended CSD の RE_EOL_INFO のオフセットを表す定数です。
 */
const int MmcExtendedCsdOffsetReEolInfo = 267;

/**
 * @brief   MMC の Extended CSD の DEVICE_LIFE_TIME_EST_TYP_A のオフセットを表す定数です。
 */
const int MmcExtendedCsdOffsetDeviceLifeTimeEstTypA = 268;

/**
 * @brief   MMC の Extended CSD の DEVICE_LIFE_TIME_EST_TYP_B のオフセットを表す定数です。
 */
const int MmcExtendedCsdOffsetDeviceLifeTimeEstTypB = 269;

/**
*   @brief      MMC の Speed Mode を取得します。
*
*   @param[out] pOutValue   取得した speed mode の格納先
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess                   成功しました。
*/
Result GetMmcSpeedMode(MmcSpeedMode* pOutValue) NN_NOEXCEPT;

/**
*   @brief      MMC の CID を取得します。
*
*   @param[out] pOutBuffer  取得した CID の格納先
*   @param[in]  size        取得した CID を格納する領域のサイズ
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess                   成功しました。
*
*   @pre        size >= MmcCidSize
*/
Result GetMmcCid(void* outBuffer, size_t size) NN_NOEXCEPT;

/**
*   @brief      MMC の CID から MID を取得します。
*
*   @param[in]  pMmcCid     nn::fs::GetMmcCid 関数で取得した CID
*
*   @return     MID の値を返します。
*/
inline uint8_t GetMidFromMmcCid(const uint8_t* pMmcCid) NN_NOEXCEPT
{
    return pMmcCid[14];
}

/**
*   @brief      MMC の CID から PNM を取得します。
*
*   @param[in]  pMmcCid     nn::fs::GetMmcCid 関数で取得した CID
*
*   @return     PNM の値を返します。
*/
inline uint64_t GetPnmFromMmcCid(const uint8_t* pMmcCid) NN_NOEXCEPT
{
    const int MmcCidPnmIndex = 6;
    const int MmcCidPnmBytes = 6;
    int i;
    uint64_t pnm = 0;
    for (i = 0; i < MmcCidPnmBytes; i++)
    {
        uint64_t val =  pMmcCid[MmcCidPnmIndex + i];
        pnm |= (val << (8 * i));
    }
    return pnm;
}

/**
*   @brief      MMC の CID から PRV を取得します。
*
*   @param[in]  pMmcCid     nn::fs::GetMmcCid 関数で取得した CID
*
*   @return     PRV の値を返します。
*/
inline uint8_t GetPrvFromMmcCid(const uint8_t* pMmcCid) NN_NOEXCEPT
{
    return pMmcCid[5];
}

/**
*   @brief      MMC パーティションを全域 erase します。
*
*   @param[in]  id        対象とする MMC パーティションの識別子
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess                   成功しました。
*/
Result EraseMmc(MmcPartition id) NN_NOEXCEPT;

/**
*   @brief      MMC パーティションのサイズを取得します。
*
*   @param[out] outValue  MMC パーティションサイズの格納先
*   @param[in]  id        対象とする MMC パーティションの識別子
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess                   成功しました。
*/
Result GetMmcPartitionSize(int64_t* outValue, MmcPartition id) NN_NOEXCEPT;

/**
*   @brief      MMC Patrol 回数を取得します。
*
*   @param[out] outValue  MMC Patrol 回数の格納先
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess                   成功しました。
*   @retval     ResultPatrolReaderHasNotGottenPatrolCount   まだ取得できません。
*/
Result GetMmcPatrolCount(uint32_t* outValue) NN_NOEXCEPT;

/**
*   @brief      MMC の Extended CSD を取得します。
*
*   @param[out] pOutBuffer  取得した Extended CSD の格納先
*   @param[in]  size        取得した Extended CSD を格納する領域のサイズ
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess                   成功しました。
*
*   @pre        size >= MmcExtendedCsdSize
*/
Result GetMmcExtendedCsd(void* outBuffer, size_t size) NN_NOEXCEPT;

/**
*   @brief      MMC Patrol 処理を一時停止します。
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess                   成功しました。
*/
Result SuspendMmcPatrol() NN_NOEXCEPT;

/**
*   @brief      MMC Patrol 処理を再開します。
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess                   成功しました。
*/
Result ResumeMmcPatrol() NN_NOEXCEPT;

}} // namespace nn::fs
