﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cstring>

#include <nn/nn_Common.h>
#include <nn/fs/fs_Result.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/fs/detail/fs_Newable.h>
#include <nn/fs/fs_IStorage.h>
#include <nn/fs/fs_QueryRange.h>

namespace nn { namespace fs {

class MemoryStorage : public IStorage, public nn::fs::detail::Newable
{
    NN_DISALLOW_COPY(MemoryStorage);

private:
    char* const m_Buffer;
    const int64_t m_Size;

public:
    MemoryStorage(void* buffer, int64_t size) NN_NOEXCEPT
        : m_Buffer(static_cast<char*>(buffer))
        , m_Size(size)
    {
    }

    virtual Result Read(int64_t offset, void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE
    {
        if( size == 0 )
        {
            NN_RESULT_SUCCESS;
        }
        if( buffer == nullptr )
        {
            return nn::fs::ResultNullptrArgument();
        }
        if( ! IStorage::CheckAccessRange(offset, size, m_Size) )
        {
            return nn::fs::ResultOutOfRange();
        }

        std::memcpy(buffer, m_Buffer + offset, size);

        NN_RESULT_SUCCESS;
    }

    virtual Result Write(int64_t offset, const void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE
    {
        if( size == 0 )
        {
            NN_RESULT_SUCCESS;
        }
        if( buffer == nullptr )
        {
            return nn::fs::ResultNullptrArgument();
        }
        if( ! IStorage::CheckAccessRange(offset, size, m_Size) )
        {
            return nn::fs::ResultOutOfRange();
        }

        std::memcpy(m_Buffer + offset, buffer, size);

        NN_RESULT_SUCCESS;
    }

    virtual Result Flush() NN_NOEXCEPT NN_OVERRIDE
    {
        NN_RESULT_SUCCESS;
    }

    virtual Result GetSize(int64_t* outValue) NN_NOEXCEPT NN_OVERRIDE
    {
        *outValue = m_Size;
        NN_RESULT_SUCCESS;
    }

    virtual Result OperateRange(
        void* outBuffer,
        size_t outBufferSize,
        OperationId operationId,
        int64_t offset,
        int64_t size,
        const void* inBuffer,
        size_t inBufferSize) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(offset);
        NN_UNUSED(size);
        NN_UNUSED(inBuffer);
        NN_UNUSED(inBufferSize);

        switch( operationId )
        {
        case OperationId::Invalidate:
            NN_RESULT_SUCCESS;

        case OperationId::QueryRange:
            NN_RESULT_THROW_UNLESS(outBuffer != nullptr, nn::fs::ResultNullptrArgument());
            NN_RESULT_THROW_UNLESS(outBufferSize == sizeof(QueryRangeInfo), nn::fs::ResultInvalidSize());

            reinterpret_cast<QueryRangeInfo*>(outBuffer)->Clear();
            NN_RESULT_SUCCESS;

        default:
            return nn::fs::ResultUnsupportedOperation();
        }
    }

};

}}
