﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <type_traits>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_StaticAssert.h>
#include <nn/fat/fat_FatFileSystem.h>

namespace nn { namespace fs {

/**
*   @brief      ストレージのエラー情報を表す構造体です。
*/
struct StorageErrorInfo {
    uint32_t numActivationFailures;         //!< ストレージの初期化に失敗した回数
    uint32_t numActivationErrorCorrections; //!< ストレージの初期化でリトライによりエラーが解消した回数
    uint32_t numReadWriteFailures;          //!< ストレージのリード・ライトに失敗した回数
    uint32_t numReadWriteErrorCorrections;  //!< ストレージのリード・ライトでリトライによりエラーが解消した回数
};
NN_STATIC_ASSERT(std::is_pod<StorageErrorInfo>::value);

/**
*   @brief      FileSystemProxy のエラー情報を表す構造体です。
*/
struct FileSystemProxyErrorInfo {
    uint32_t romFsRemountForDataCorruptionCount;                 //!< RomFs で data corruption が発生しリマウントで解消した回数（5 NUP 以前は RomFs で data corruption が発生した後リマウントした回数）
    uint32_t romFsUnrecoverableDataCorruptionByRemountCount;     //!< RomFs で data corruption が発生しリトライで解消しなかった回数
    nn::fat::FatError fatFsError;                                //!< FatFs の 一番最後に発生したエラー情報
    uint32_t romFsRecoveredByInvalidateCacheCount;               //!< RomFs でエラーが発生しキャッシュ無効化後のリトライで解消した回数
    uint32_t saveDataIndexCount;                                 //!< SaveData の Index 登録数
    uint8_t reserved[128 - 16 - sizeof(nn::fat::FatError)];
};
NN_STATIC_ASSERT(std::is_pod<FileSystemProxyErrorInfo>::value);
NN_STATIC_ASSERT(sizeof(FileSystemProxyErrorInfo) == 128);

/**
*   @brief      MMC のエラー情報を取得し、クリアします。
*
*   @param[out] pOutStorageErrorInfo    取得したエラー情報の格納先
*   @param[out] pOutLogSize             取得したエラーログのサイズの格納先
*   @param[out] pOutLogBuffer           取得したエラーログの格納先
*   @param[in]  logBufferSize           取得したエラーログの格納先のサイズ
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess               成功しました。
*
*   @pre
*       - pOutStorageErrorInfo が有効なメモリを指している。
*       - pOutLogSize が有効なメモリを指している。
*       - pOutLogBuffer が有効なメモリを指している。
*/
Result GetAndClearMmcErrorInfo(StorageErrorInfo* pOutStorageErrorInfo, size_t* pOutLogSize, char* pOutLogBuffer, size_t logBufferSize) NN_NOEXCEPT;

/**
*   @brief      SD カードのエラー情報を取得し、クリアします。
*
*   @param[out] pOutStorageErrorInfo    取得したエラー情報の格納先
*   @param[out] pOutLogSize             取得したエラーログのサイズの格納先
*   @param[out] pOutLogBuffer           取得したエラーログの格納先
*   @param[in]  logBufferSize           取得したエラーログの格納先のサイズ
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess               成功しました。
*
*   @pre
*       - pOutStorageErrorInfo が有効なメモリを指している。
*       - pOutLogSize が有効なメモリを指している。
*       - pOutLogBuffer が有効なメモリを指している。
*/
Result GetAndClearSdCardErrorInfo(StorageErrorInfo* pOutStorageErrorInfo, size_t* pOutLogSize, char* pOutLogBuffer, size_t logBufferSize) NN_NOEXCEPT;


/**
*   @brief      FileSystemProxy のエラー情報を取得します。
*
*   @param[out] pOutValue    取得したエラー情報の格納先
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess               成功しました。
*
*   @pre
*       - pOutValue が有効なメモリを指している。
*/
Result GetAndClearFileSystemProxyErrorInfo(FileSystemProxyErrorInfo* pOutValue) NN_NOEXCEPT;

}} // namespace nn::fs
