﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>

#include <nn/fs/fs_DbmRomTypes.h>
#include <nn/fs/fs_DbmRomPathTool.h>
#include <nn/fs/fs_DbmKeyValueRomStorageTemplate.h>
#include <nn/fs/fs_IStorage.h>

namespace nn { namespace fs {

/*!
    @brief ROM ファイルシステム専用ファイル、ディレクトリエントリー管理テーブルクラスです。


    以下のような操作を行った場合はエラーを返します。

    ・パスに存在しないディレクトリが含まれている場合は ResultDbmDirectoryNotFound を返します。
    ・パスに含まれているディレクトリ名が長すぎる場合は ResultDbmDirectoryNameTooLong を返します。
    ・パスに含まれているファイル名が長すぎる場合は ResultDbmFileNameTooLong を返します。
    ・パスに含まれている区切り文字が複数回連続している場合には
      ResultDbmInvalidPathFormat を返します。
    ・パスの 1 文字目が "/" でない場合は ResultDbmInvalidPathFormat を返します。
    ・ファイルに対してディレクトリ操作を行おうとした場合は ResultDbmInvalidOperation を返します。
    ・ディレクトリに対してファイル操作を行おうとした場合は ResultDbmInvalidOperation を返します。

    作成処理用エラー
    ・すでに存在しているディレクトリ、ファイルを作成しようとした場合は
      ResultDbmAlreadyExists を返します。
*/
class HierarchicalRomFileTable
{
public:
    //! オフセット
    typedef uint32_t StoragePosition;

    //! ファイル探索位置指示構造体です。
    struct FindPosition
    {
        StoragePosition nextPositionDirectory;     //!< 探索中のディレクトリインデックス
        StoragePosition nextPositionFile;          //!< 探索中のファイルインデックス
    };
    NN_STATIC_ASSERT(std::is_pod<FindPosition>::value);

    typedef RomDirectoryInfo DirectoryInfo;
    typedef RomFileInfo FileInfo;

    //<! 位置データをファイル ID に変換します。
    static inline RomFileId PositionToFileId(StoragePosition pos) NN_NOEXCEPT
    {
        return static_cast<RomFileId>(pos);
    }

public:

    /*!
        @brief ディレクトリエントリー用ストレージに必要なサイズを求めます。

               1 エントリーに対して大きめなサイズを要求します。
               初回ファイルツリー作成(メタデータ構築)時に使用します。

        @param[in] countDirectoryEntry ディレクトリエントリー数

        @return ディレクトリエントリー用ストレージに必要なサイズ
    */
    static int64_t QueryDirectoryEntryStorageSize(uint32_t countDirectoryEntry) NN_NOEXCEPT;

    /*!
        @brief ファイルエントリー用ストレージに必要なサイズを求めます。

               1 エントリーに対して大きめなサイズを要求します。
               初回ファイルツリー作成(メタデータ構築)時に使用します。

        @param[in] countFileEntry ファイルエントリー数

        @return ファイルエントリー用ストレージに必要なサイズ
    */
    static int64_t QueryFileEntryStorageSize(uint32_t countFileEntry) NN_NOEXCEPT;


    /*!
        @brief バケット数から必要なストレージのサイズを求めます。

        ・ディレクトリバケット
           ・"バケット数"とは、ディレクトリ名の探索に使用するハッシュテーブルのサイズです。
           ・バケット数の大きさに比例し、ディレクトリの探索速度が向上します。
           ・総ディレクトリ数の数分の一を目安で与えます。
           ・バケット数はフォーマット以降拡張できません。

        @param[in] countDirectoryBucket ディレクトリバケット数

        @return ディレクトリエントリーバケット用ストレージに必要なサイズ
    */
    static int64_t QueryDirectoryEntryBucketStorageSize(int64_t countDirectoryBucket) NN_NOEXCEPT;

    /*!
        @brief バケット数から必要なストレージのサイズを求めます。

        ・ファイルバケット
           ・"バケット数"とは、ファイル名の探索に使用するハッシュテーブルのサイズです。
           ・バケット数の大きさに比例し、ファイルの探索速度が向上します。
           ・総ファイル数の数分の一を目安で与えます。
           ・バケット数はフォーマット以降拡張できません。

        @param[in] countFileBucket ファイルバケット数

        @return ファイルエントリーバケット用ストレージに必要なサイズを取得します。
    */
    static int64_t QueryFileEntryBucketStorageSize(int64_t countFileBucket) NN_NOEXCEPT;

    /*!
        @brief ファイル管理テーブルをフォーマットします。

               バケットストレージを初期化します。使用する前に一度呼び出します。

        @param[in] directoryBucket      ディレクトリエントリーバケット用ストレージ
        @param[in] fileBucket           ファイルエントリーバケット用ストレージ

        @retval ResultSuccess       成功しました。
        @retval ResultOutOfRange    ストレージの範囲外を書き込もうとしました。
        @retval それ以外            ストレージの書き込みで失敗しました。

        @return 関数の処理結果を返します。
    */
    static Result Format(
                      SubStorage directoryBucket,
                      SubStorage fileBucket
                  ) NN_NOEXCEPT;

    /*!
        @brief  コンストラクタです。
    */
    HierarchicalRomFileTable() NN_NOEXCEPT;

    /*!
        @brief  ストレージをマウントします。

        @param[in] directoryBucket ディレクトリバケット用ストレージ
        @param[in] directoryEntry ディレクトリエントリー用ストレージ
        @param[in] fileBucket ファイルエントリーバケット用ストレージ
        @param[in] fileEntry ファイルエントリー用ストレージ

        @return 関数の処理結果を返します。

        @retval ResultSuccess   成功しました。
    */
    Result Initialize(
               SubStorage directoryBucket,
               SubStorage directoryEntry,
               SubStorage fileBucket,
               SubStorage fileEntry
           ) NN_NOEXCEPT;

    /*!
        @brief  ストレージをアンマウントします。
    */
    void Finalize() NN_NOEXCEPT;

    /*!
        @brief ルートディレクトリエントリーを作成します。

        @return 関数の処理結果を返します。

        @retval ResultSuccess           成功しました。
        @retval ResultDbmAlreadyExists  すでにルートディレクトリが作成されています。
        @retval ResultDbmKeyFull        割り当て可能領域はありません。
        @retval ResultOutOfRange        ストレージの範囲外をアクセスしようとしました。
        @retval それ以外                ストレージの読み書きで失敗しました。
    */
    Result CreateRootDirectory() NN_NOEXCEPT;

    /*!
        @brief ディレクトリを作成します。

        @param[out] pOutValue ディレクトリ ID
        @param[in] pFullPath ディレクトリパス
        @param[in] dirInfo ディレクトリ情報

        @return 関数の処理結果を返します。

        @retval ResultSuccess               成功しました。
        @retval ResultDbmAlreadyExists      すでにディレクトリが存在します。
        @retval ResultDbmDirectoryEntryFull これ以上ディレクトリを作成できません。

        @pre    pOutValue が有効なメモリを指している。
        @pre    pFullPath != nullptr
    */
    Result CreateDirectory(
        RomDirectoryId* pOutValue, const RomPathChar* pFullPath, const DirectoryInfo& dirInfo
    ) NN_NOEXCEPT;

    /*!
        @brief ファイルを作成します。

        @param[out] pOutValue ディレクトリ ID
        @param[in] pFullPath ファイルパス
        @param[in] fileInfo ファイル情報

        @return 関数の処理結果を返します。

        @retval ResultSuccess                   成功しました。
        @retval ResultDbmAlreadyExists          すでにファイルが存在します。
        @retval ResultDbmFileEntryFull          これ以上ファイルを作成できません。
        @retval ResultDbmDirectoryNotFound      作成先ディレクトリが見つかりませんでした。
        @retval ResultDbmInvalidOperation       不正な操作です。

        @pre    pOutValue が有効なメモリを指している。
        @pre    pFullPath != nullptr
    */
    Result CreateFile(
        RomFileId* pOutValue, const RomPathChar* pFullPath, const FileInfo& fileInfo
    ) NN_NOEXCEPT;

    /*!
        @brief 指定したフルパスをディレクトリ ID に変換します。

        @param[out] pOutValue ディレクトリ ID
        @param[in] pFullPath ディレクトリパス

        @return 関数の処理結果を返します。

        @retval ResultSuccess                   成功しました。
        @retval ResultDbmDirectoryNameTooLong   ディレクトリ名が長過ぎます。
        @retval ResultDbmDirectoryNotFound      親ディレクトリが見つかりませんでした。
        @retval ResultDbmInvalidOperation       不正な操作です。

        @pre    pOutValue が有効なメモリを指している。
        @pre    pFullPath != nullptr
    */
    Result ConvertPathToDirectoryId(
        RomDirectoryId* pOutValue, const RomPathChar* pFullPath
    ) NN_NOEXCEPT;

    /*!
        @brief 指定したフルパスをファイル ID に変換します。

        @param[out] pOutValue ファイル ID
        @param[in] pFullPath ファイルパス

        @return 関数の処理結果を返します。

        @retval ResultSuccess                   成功しました。
        @retval ResultDbmDirectoryNameTooLong   ディレクトリ名が長過ぎます。
        @retval ResultDbmDirectoryNotFound      親ディレクトリが見つかりませんでした。
        @retval ResultDbmFileNotFound           ファイルが見つかりませんでした。
        @retval ResultDbmInvalidOperation       不正な操作です。

        @pre    pOutValue が有効なメモリを指している。
        @pre    pFullPath != nullptr
    */
    Result ConvertPathToFileId(RomFileId* pOutValue, const RomPathChar* pFullPath) NN_NOEXCEPT;

    /*!
        @brief 指定したディレクトリ情報を取得します。

        @param[out] pOutValue ディレクトリ情報
        @param[in] pFullPath ディレクトリパス

        @return 関数の処理結果を返します。

        @retval ResultSuccess                   成功しました。
        @retval ResultDbmDirectoryNameTooLong   ディレクトリ名が長過ぎます。
        @retval ResultDbmDirectoryNotFound      ディレクトリが見つかりませんでした。
        @retval ResultDbmInvalidOperation       不正な操作です。

        @pre    pOutValue が有効なメモリを指している。
        @pre    pFullPath != nullptr
    */
    Result GetDirectoryInformation(
        DirectoryInfo* pOutValue, const RomPathChar* pFullPath
    ) NN_NOEXCEPT;

    /*!
        @brief 指定したディレクトリ情報を取得します。

        @param[out] pOutValue ディレクトリ情報
        @param[in] directoryId ディレクトリ ID

        @return 関数の処理結果を返します。

        @retval ResultSuccess               成功しました。
        @retval ResultDbmDirectoryNotFound  ディレクトリが見つかりませんでした。
        @retval ResultDbmInvalidOperation   不正な操作です。

        @pre    pOutValue が有効なメモリを指している。
    */
    Result GetDirectoryInformation(
        DirectoryInfo* pOutValue, RomDirectoryId directoryId
    ) NN_NOEXCEPT;

    /*!
        @brief ファイルを開きます。

        @param[out] pOutValue ファイル情報
        @param[in] pFullPath ファイルパス

        @return 関数の処理結果を返します。

        @retval ResultSuccess                   成功しました。
        @retval ResultDbmDirectoryNameTooLong   ディレクトリ名が長過ぎます。
        @retval ResultDbmDirectoryNotFound      親ディレクトリが見つかりませんでした。
        @retval ResultDbmFileNotFound           ファイルが見つかりませんでした。
        @retval ResultDbmInvalidOperation       不正な操作です。

        @pre    pOutValue が有効なメモリを指している。
        @pre    pFullPath != nullptr
    */
    Result OpenFile(FileInfo* pOutValue, const RomPathChar* pFullPath) NN_NOEXCEPT;

    /*!
        @brief ファイルを開きます。

        @param[out] pOutValue ファイル情報
        @param[in] fileId ファイル ID

        @return 関数の処理結果を返します。

        @retval ResultSuccess               成功しました。
        @retval ResultDbmFileNotFound       ファイルが見つかりませんでした。
        @retval ResultDbmInvalidOperation   不正な操作です。

        @pre    pOutValue が有効なメモリを指している。
    */
    Result OpenFile(FileInfo* pOutValue, RomFileId fileId) NN_NOEXCEPT;

    /*!
        @brief 指定したディレクトリ以下のファイル列挙を開始します。

               イテレータを @ref FindNextDirectory や @ref FindNextFile に
               渡すことでディレクトリ名、ファイル名をイテレーションできます。
               また、イテレーションはファイル、ディレクトリ各々に対して行うことができます。

        @param[out] pOutValue イテレータ
        @param[in] pFullPath ディレクトリパス

        @return 関数の処理結果を返します。

        @retval ResultSuccess                   成功しました。
        @retval ResultDbmDirectoryNameTooLong   ディレクトリ名が長過ぎます。
        @retval ResultDbmDirectoryNotFound      ディレクトリが見つかりませんでした。
        @retval ResultDbmInvalidOperation       不正な操作です。

        @pre    pOutValue が有効なメモリを指している。
        @pre    pFullPath != nullptr
    */
    Result FindOpen(FindPosition* pOutValue, const RomPathChar* pFullPath) NN_NOEXCEPT;

    /*!
        @brief 指定したディレクトリ以下のファイル列挙を開始します。

               イテレータを @ref FindNextDirectory や @ref FindNextFile に
               渡すことでディレクトリ名、ファイル名をイテレーションできます。
               また、イテレーションはファイル、ディレクトリ各々に対して行うことができます。

        @param[out] pOutValue イテレータ
        @param[in] directoryId ディレクトリ ID

        @return 関数の処理結果を返します。

        @retval ResultSuccess               成功しました。
        @retval ResultDbmDirectoryNotFound  ディレクトリが見つかりませんでした。
        @retval ResultDbmInvalidOperation   不正な操作です。

        @pre    pOutValue が有効なメモリを指している。
    */
    Result FindOpen(FindPosition* pOutValue, RomDirectoryId directoryId) NN_NOEXCEPT;

    /*!
        @brief ディレクトリ以下の次のディレクトリを取得します。

               この関数の前に @ref FindOpen でイテレータを初期化する必要があります。

        @param[out]     pOutValue       ディレクトリ名を格納するバッファ
        @param[in,out]  pFindPosition   イテレータ
        @param[in]      length          pOutValue のバッファ文字数

        @return 関数の処理結果を返します。

        @retval ResultSuccess               成功しました。
        @retval ResultDbmFindFinished       イテレーションが完了しました。

        @pre    pOutValue が有効なメモリを指している。
        @pre    pFindPosition != nullptr
        @pre    length > RomPathTool::MaxPathLength
    */
    Result FindNextDirectory(
        RomPathChar* pOutValue, FindPosition* pFindPosition, size_t length
    ) NN_NOEXCEPT;

    /*!
        @brief ディレクトリ以下の次のファイルを取得します。

               この関数の前に @ref FindOpen でイテレータを初期化する必要があります。

        @param[out]     pOutValue       ベースファイル名
        @param[in,out]  pFindPosition   イテレータ
        @param[in]      length          pOutValue のバッファ文字数

        @return 関数の処理結果を返します。

        @retval ResultSuccess               成功しました。
        @retval ResultDbmFindFinished       イテレーションが完了しました。

        @pre    pOutValue が有効なメモリを指している。
        @pre    pFindPosition != nullptr
        @pre    length > RomPathTool::MaxPathLength
    */
    Result FindNextFile(RomPathChar* pOutValue, FindPosition* pFindPosition, size_t length) NN_NOEXCEPT;

    /*!
        @brief ファイルシステム構築に必要なバッファサイズを取得します。

        @param[out] pOutSizeDirectoryEntry ディレクトリエントリに必要なサイズ
        @param[out] pOutSizeFileEntry ファイルエントリに必要なサイズ

        @return 関数の処理結果を返します。

        @retval ResultSuccess   成功しました。

        @pre    pOutSizeDirectoryEntry が有効なメモリを指している。
        @pre    pOutSizeFileEntry が有効なメモリを指している。
    */
    Result QueryRomFileSystemSize(
               int64_t* pOutSizeDirectoryEntry,
               int64_t* pOutSizeFileEntry
           ) NN_NOEXCEPT;

    //! ディレクトリエントリー数を取得します。
    inline uint32_t GetDirectoryEntryCount() const NN_NOEXCEPT
    {
        return m_TableDirectory.GetEntryCount();
    }

    //! ファイルエントリー数を取得します。
    inline uint32_t GetFileEntryCount() const NN_NOEXCEPT
    {
        return m_TableFile.GetEntryCount();
    }

    /*!
        @brief ファイル構成をダンプします。

               デバッグ用途の関数です。内部でスタックを多く消費します。

        @return 関数の処理結果を返します。

        @retval ResultSuccess       成功しました。
        @retval ResultOutOfRange    ストレージの範囲外を読み込もうとしました。
        @retval それ以外            ストレージの読み込みで失敗しました。
    */
    Result DumpTree() NN_NOEXCEPT;

private:
    //! 無効な位置です。
    // (KeyValueStorageTemplate::KVSTORAGE_FREEENTRY と同じ値を使用)
    static const StoragePosition PositionNone = 0xFFFFFFFF;

    //! ルートディレクトリを示す位置です。
    static const StoragePosition PositionRootDirectory = 0;

    //! システムが暗黙的に使用するディレクトリエントリー(ルートディレクトリ分)
    static const uint32_t DirectoryCountSystemReserved = 1;

private:
    //! ディレクトリエントリーです。
    struct DirectoryRomEntry
    {
        StoragePosition  posNext;           //!< 兄弟へのインデックス
        StoragePosition  posDirectory;      //!< 子ディレクトリー内にあるディレクトリのインデックス
        StoragePosition  posFile;           //!< 子ディレクトリー内にあるファイルのインデックス
#ifdef NN_DBM_ENABLE_DIRECTORY_METADATA
        DirectoryInfo    info;              //!< ディレクトリ付加情報
#endif
    };
    NN_STATIC_ASSERT(std::is_pod<DirectoryRomEntry>::value);

    //! ファイルエントリーです。
    struct FileRomEntry
    {
        StoragePosition  posNext;           //!< 兄弟へのインデックス
        FileInfo         info;              //!< ファイル付加情報
    };
    NN_STATIC_ASSERT(std::is_pod<FileRomEntry>::value);

    //! エントリーマップテーブルの共通実装
    static const uint32_t MaxKeyLength = RomPathTool::MaxPathLength;
    template <
        typename TKey,
        typename TKey2,
        typename TValue
    >
    class EntryMapTable
        : public KeyValueRomStorageTemplate<TKey, TValue, MaxKeyLength>
    {
    public:
        typedef TKey Key;
        typedef TKey2 Key2;
        typedef TValue Value;
        typedef uint32_t Position;
        typedef KeyValueRomStorageTemplate<TKey, TValue, MaxKeyLength> BaseClass;

    public:
        //! KeyValueRomStorageTemplate::AddInternal を公開します。
        inline Result Add(Position* pOutPosition, const Key2& key, const Value& value) NN_NOEXCEPT
        {
            return BaseClass::AddInternal(
                       pOutPosition,
                       key.key,
                       key.Hash(),
                       key.name.path,
                       key.name.length * sizeof(RomPathChar),
                       value
                   );
        }

        //! KeyValueRomStorageTemplate::GetInternal を公開します。
        inline Result Get(Position* pOutPosition, Value* pValue, const Key2& key) NN_NOEXCEPT
        {
            return BaseClass::GetInternal(
                       pOutPosition,
                       pValue,
                       key.key,
                       key.Hash(),
                       key.name.path,
                       key.name.length * sizeof(RomPathChar)
                   );
        }

        //! KeyValueRomStorageTemplate::GetByPosition を公開します。
        inline Result GetByPosition(
            Key* pOutKey, Value* pOutValue, Position pos
        ) NN_NOEXCEPT
        {
            return BaseClass::GetByPosition(pOutKey, pOutValue, pos);
        }

        //! KeyValueRomStorageTemplate::GetByPosition を公開します。
        inline Result GetByPosition(
            Key* pOutKey, Value* pOutValue, void* pOutExtraKey, size_t* pOutExtraSize, Position pos
        ) NN_NOEXCEPT
        {
            return BaseClass::GetByPosition(pOutKey, pOutValue, pOutExtraKey, pOutExtraSize, pos);
        }

        //! KeyValueRomStorageTemplate::SetByPosition を公開します。
        inline Result SetByPosition(
            Position pos, const Value& value
        ) NN_NOEXCEPT
        {
            return BaseClass::SetByPosition(pos, value);
        }

    };

    //! エントリーを管理するハッシュテーブル用固定長キー
    struct RomEntryKey
    {
        StoragePosition parentDirectory;          //!< 親ディレクトリ

        //! キーの比較を行います。
        inline bool IsEqual(
            const RomEntryKey& rKey,
            const void* pExtraKey1,
            size_t extraSize1,
            const void* pExtraKey2,
            size_t extraSize2
        ) const NN_NOEXCEPT
        {
            if( this->parentDirectory != rKey.parentDirectory )
            {
                return false;
            }
            if( extraSize1 != extraSize2 )
            {
                return false;
            }
            return RomPathTool::IsEqualPath(
                reinterpret_cast<const RomPathChar*>(pExtraKey1),
                reinterpret_cast<const RomPathChar*>(pExtraKey2),
                extraSize1 / sizeof(RomPathChar)
            );
        }
    };
    NN_STATIC_ASSERT(std::is_pod<RomEntryKey>::value);

    //! エントリーを管理するハッシュテーブル用キー
    struct EntryKey
    {
        RomEntryKey key;                    //!< キ－
        RomPathTool::RomEntryName name;     //!< 名前

        //! ハッシュ値を取得します。
        inline uint32_t Hash() const NN_NOEXCEPT
        {
            uint32_t v = 123456789;
            v ^= this->key.parentDirectory;
            const RomPathChar* pName = this->name.path;
            const RomPathChar* pNameEnd = pName + this->name.length;
            while( pName < pNameEnd )
            {
                // *pName の値を符号なしに統一します。
                const auto unsignedNameChar
                    = static_cast<std::make_unsigned<RomPathChar>::type>(*pName);
                v = ((v >> 5) | (v << (32 - 5))) ^ unsignedNameChar;
                pName++;
            }
            return v;
        }
    };
    NN_STATIC_ASSERT(std::is_pod<EntryKey>::value);

    //! ディレクトリエントリーを管理するハッシュテーブル
    typedef EntryMapTable<RomEntryKey, EntryKey, DirectoryRomEntry> DirectoryEntryMapTable;

    //! ファイルエントリーを管理するハッシュテーブル
    typedef EntryMapTable<RomEntryKey, EntryKey, FileRomEntry> FileEntryMapTable;

private:
    //<! 位置データをディレクトリ ID に変換します。
    static inline RomDirectoryId PositionToDirectoryId(StoragePosition pos) NN_NOEXCEPT
    {
        return static_cast<RomDirectoryId>(pos);
    }

    //! ディレクトリ ID を位置データに変換します。
    static inline StoragePosition DirectoryIdToPosition(RomDirectoryId directoryId) NN_NOEXCEPT
    {
        return static_cast<StoragePosition>(directoryId);
    }

    //! ファイル ID を位置データに変換します。
    static inline StoragePosition FileIdToPosition(RomFileId fileId) NN_NOEXCEPT
    {
        return static_cast<StoragePosition>(fileId);
    }

private:

    /*!
        @brief 親の親ディレクトリを取得します。

        @param[out] pOutPosition 親の親ディレクトリエントリーのストレージ位置
        @param[out] pOutDirectoryKey 親の親ディレクトリキー
        @param[out] pOutDirectoryEntry 親の親ディレクトリエントリー
        @param[in] pos 親ディレクトリエントリーのストレージ位置
        @param[in] name 親ディレクトリ名
        @param[in] pFullPath フルパス

        @return 関数の処理結果を返します。

        @retval ResultSuccess               成功しました。
        @retval ResultDbmDirectoryNotFound  ディレクトリが見つかりませんでした。
        @retval ResultDbmInvalidOperation   不正な操作です。
        @retval ResultDbmInvalidPathFormat  ルートディレクトリの親を取得しようとしました。

        @pre    pOutPosition が有効なメモリを指している。
        @pre    pOutDirectoryKey が有効なメモリを指している。
        @pre    pOutDirectoryEntry が有効なメモリを指している。
        @pre    pFullPath != nullptr
    */
    Result GetGrandparent(
               StoragePosition* pOutPosition,
               EntryKey* pOutDirectoryKey,
               DirectoryRomEntry* pOutDirectoryEntry,
               StoragePosition pos,
               RomPathTool::RomEntryName name,
               const RomPathChar* pFullPath
          ) NN_NOEXCEPT;

    /*!
        @brief 指定した親ディレクトリを取得します。

        @param[out] pOutParentPosition 親ディレクトリエントリーのストレージ位置
        @param[out] pOutParentDirectoryKey 親ディレクトリキー
        @param[out] pOutParentDirectoryEntry 親ディレクトリエントリー
        @param[in] pParser パスパーサー
        @param[in] pFullPath フルパス

        @return 関数の処理結果を返します。

        @retval ResultSuccess                   成功しました。
        @retval ResultDbmDirectoryNameTooLong   ディレクトリ名が長過ぎます。
        @retval ResultDbmDirectoryNotFound      ディレクトリが見つかりませんでした。
        @retval ResultDbmInvalidOperation       ルートディレクトリの親ディレクトリは取得できません。

        @pre    pOutParentPosition が有効なメモリを指している。
        @pre    pOutParentDirectoryKey が有効なメモリを指している。
        @pre    pOutParentDirectoryEntry が有効なメモリを指している。
        @pre    pParser != nullptr
        @pre    pFullPath != nullptr
    */
    Result FindParentDirectoryRecursive(
               StoragePosition* pOutParentPosition,
               EntryKey* pOutParentDirectoryKey,
               DirectoryRomEntry* pOutParentDirectoryEntry,
               RomPathTool::PathParser* pParser,
               const RomPathChar* pFullPath
           ) NN_NOEXCEPT;

    /*!
        @brief 指定したパスからディレクトまたはファイルを取得します。

        @param[out] pOutKey 取得したディレクトリまたはファイルエントリーのキー
        @param[out] pOutParentDirectoryEntry 親ディレクトリエントリー
        @param[in] isFindDirectory ディレクトリを取得するなら true,
                                   ファイルを取得するなら false
        @param[in] pFullPath 分離するフルパス

        @return 関数の処理結果を返します。

        @retval ResultSuccess                   成功しました。
        @retval ResultDbmDirectoryNameTooLong   ディレクトリ名が長過ぎます。
        @retval ResultDbmDirectoryNotFound      ディレクトリが見つかりませんでした。
        @retval ResultDbmInvalidOperation       ルートディレクトリの親ディレクトリは取得できません。
        @retval ResultDbmInvalidOperation       不正な操作です。

        @pre    pOutKey が有効なメモリを指している。
        @pre    pOutParentDirectoryEntry が有効なメモリを指している。
        @pre    pFullPath != nullptr
    */
    Result FindPathRecursive(
               EntryKey* pOutKey,
               DirectoryRomEntry* pOutParentDirectoryEntry,
               bool isFindDirectory,
               const RomPathChar* pFullPath
           ) NN_NOEXCEPT;

    /*!
        @brief 指定したパスからディレクトリキーを取得します。

        @param[out] pOutDirectoryKey ディレクトリキー
        @param[out] pOutParentDirectoryEntry 親ディレクトリエントリー
        @param[in] pFullPath 分離するフルパス

        @return 関数の処理結果を返します。

        @retval ResultSuccess                   成功しました。
        @retval ResultDbmDirectoryNameTooLong   ディレクトリ名が長過ぎます。
        @retval ResultDbmDirectoryNotFound      ディレクトリが見つかりませんでした。
        @retval ResultDbmInvalidOperation       ルートディレクトリの親ディレクトリは取得できません。
        @retval ResultDbmInvalidOperation       不正な操作です。

        @pre    pOutParentDirectoryEntry が有効なメモリを指している。
        @pre    pOutDirectoryKey が有効なメモリを指している。
        @pre    pFullPath != nullptr
    */
    Result FindDirectoryRecursive(
               EntryKey* pOutDirectoryKey,
               DirectoryRomEntry* pOutParentDirectoryEntry,
               const RomPathChar* pFullPath
           ) NN_NOEXCEPT;

    /*!
        @brief 指定したパスからファイルキーを取得します。

        @param[out] pOutFileKey ファイルキー
        @param[out] pOutParentDirectoryEntry 親ディレクトリエントリー
        @param[in] pFullPath 分離するフルパス

        @return 関数の処理結果を返します。

        @retval ResultSuccess                   成功しました。
        @retval ResultDbmDirectoryNameTooLong   ディレクトリ名が長過ぎます。
        @retval ResultDbmDirectoryNotFound      ディレクトリが見つかりませんでした。
        @retval ResultDbmInvalidOperation       ルートディレクトリの親ディレクトリは取得できません。
        @retval ResultDbmInvalidOperation       不正な操作です。

        @pre    pOutParentDirectoryEntry が有効なメモリを指している。
        @pre    pOutFileKey が有効なメモリを指している。
        @pre    pFullPath != nullptr
    */
    Result FindFileRecursive(
               EntryKey* pOutFileKey,
               DirectoryRomEntry* pOutParentDirectoryEntry,
               const RomPathChar* pFullPath
           ) NN_NOEXCEPT;

    /*!
        @brief 指定したエントリーと同名のファイル、ディレクトリが存在するかどうかを取得します。

        @param[in] key キー
        @param[in] resultIfExist 同名のエントリーが存在したときの結果

        @retval ResultSuccess   同名のエントリーは存在しません。
        @retval resultIfExist   同名のエントリーが存在します。
        @retval それ以外        ストレージによるエラーです。

        @return 関数の処理結果を返します。
    */
    Result CheckSameEntryExists(
               const EntryKey& key,
               Result resultIfExist
           ) NN_NOEXCEPT;

    /*!
        @brief ディレクトリエントリーを取得します。

        @param[out] pOutPosition ディレクトリエントリーのストレージ位置
        @param[out] pOutEntry ディレクトリエントリー
        @param[in] key ディレクトリキー

        @return 関数の処理結果を返します。

        @retval ResultSuccess                   成功しました。
        @retval ResultDbmDirectoryNotFound      ディレクトリが見つかりませんでした。
        @retval ResultDbmInvalidOperation       不正な操作です。

        @pre    pOutPosition が有効なメモリを指している。
        @pre    pOutEntry が有効なメモリを指している。
    */
    Result GetDirectoryEntry(
        StoragePosition* pOutPosition, DirectoryRomEntry* pOutEntry, const EntryKey& key
    ) NN_NOEXCEPT;

    /*!
        @brief ディレクトリエントリーを取得します。

        @param[out] pOutValue ディレクトリエントリー
        @param[in] directoryId ディレクトリ ID

        @return 関数の処理結果を返します。

        @retval ResultSuccess               成功しました。
        @retval ResultDbmDirectoryNotFound  ディレクトリが見つかりませんでした。
        @retval ResultDbmInvalidOperation   不正な操作です。

        @pre    pOutValue が有効なメモリを指している。
    */
    Result GetDirectoryEntry(DirectoryRomEntry* pOutValue, RomDirectoryId directoryId) NN_NOEXCEPT;

    /*!
        @brief ファイルエントリーを取得します。

        @param[out] pOutPosition ファイルエントリーのストレージ位置
        @param[out] pOutEntry ファイルエントリー
        @param[in] key ファイルキー

        @return 関数の処理結果を返します。

        @retval ResultSuccess               成功しました。
        @retval ResultDbmFileNotFound       ファイルが見つかりませんでした。
        @retval ResultDbmInvalidOperation   不正な操作です。

        @pre    pOutPosition が有効なメモリを指している。
        @pre    pOutEntry が有効なメモリを指している。
    */
    Result GetFileEntry(
        StoragePosition* pOutPosition, FileRomEntry* pOutEntry, const EntryKey& key
    ) NN_NOEXCEPT;

    /*!
        @brief ファイルエントリーを取得します。

        @param[out] pOutValue ファイルエントリー
        @param[in] fileId ファイル ID

        @return 関数の処理結果を返します。

        @retval ResultSuccess               成功しました。
        @retval ResultDbmFileNotFound       ファイルが見つかりませんでした。
        @retval ResultDbmInvalidOperation   不正な操作です。

        @pre    pOutValue が有効なメモリを指している。
    */
    Result GetFileEntry(FileRomEntry* pOutValue, RomFileId fileId) NN_NOEXCEPT;

    /*!
        @brief 指定したディレクトリ情報を取得します。

        @param[out] pOutValue ディレクトリ情報
        @param[in] directoryKey ディレクトリキー

        @retval ResultSuccess               成功しました。
        @retval ResultDbmDirectoryNotFound  ディレクトリが見つかりませんでした。
        @retval ResultDbmInvalidOperation   不正な操作です。

        @return 関数の処理結果を返します。

        @pre    pOutValue が有効なメモリを指している。
    */
    Result GetDirectoryInformation(
        DirectoryInfo* pOutValue, const EntryKey& directoryKey
    ) NN_NOEXCEPT;

    /*!
        @brief ファイルを開きます。

        @param[out] pOutValue ファイル情報
        @param[in] fileKey ファイルキー

        @return 関数の処理結果を返します。

        @retval ResultSuccess               成功しました。
        @retval ResultDbmFileNotFound       ファイルが見つかりませんでした。
        @retval ResultDbmInvalidOperation   不正な操作です。

        @pre    pOutValue が有効なメモリを指している。
    */
    Result OpenFile(FileInfo* pOutValue, const EntryKey& fileKey) NN_NOEXCEPT;

    /*!
        @brief 指定したディレクトリ以下のファイル列挙を開始します。

               イテレータを @ref FindNextDirectory、@ref FindNextFile に
               渡すことでディレクトリ名、ファイル名をイテレーションできます。
               また、イテレーションはファイル、ディレクトリ各々に対して行うことができます。

        @param[out] pOutValue イテレータ
        @param[in] directoryKey ディレクトリキー

        @return 関数の処理結果を返します。

        @retval ResultSuccess               成功しました。
        @retval ResultDbmDirectoryNotFound  ディレクトリが見つかりませんでした。
        @retval ResultDbmInvalidOperation   不正な操作です。

        @pre    pOutValue が有効なメモリを指している。
    */
    Result FindOpen(FindPosition* pOutValue, const EntryKey& directoryKey) NN_NOEXCEPT;

    /*!
        @brief 指定したディレクトリ以下のファイルツリーをデバッグ表示します。

        @param[in] posDirectory ディレクトリ位置
        @param[in] depth ファイル階層の深さ

        @return 関数の処理結果を返します。

        @retval ResultSuccess       成功しました。
        @retval ResultOutOfRange    ストレージの範囲外を読み込もうとしました。
        @retval それ以外            ストレージの読み込みで失敗しました。
    */
    Result DumpDirectory(StoragePosition posDirectory, int depth) NN_NOEXCEPT;

    //! テストのため TestSuite に対して実装を公開します。
    friend class HierarchicalRomFileTableTestSuite;

private:
    DirectoryEntryMapTable m_TableDirectory;    //!< ディレクトリ管理テーブル
    FileEntryMapTable m_TableFile;              //!< ファイル管理テーブル
};

}}

