﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
*   @file
*   @brief  Built-In-Storage FS 関連のファイルシステム API の宣言
*/

#pragma once

#include <memory>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>

namespace nn { namespace fs {
    class IStorage;
}}

namespace nn { namespace fs {

//! @name Built-In-Storage FS 関連 API
//! @{

enum class BisPartitionId
{
    BootPartition1Root = 0,
    // reserved
    BootPartition2Root = 10,
    // reserved
    UserDataRoot = 20,
    BootConfigAndPackage2Part1,
    BootConfigAndPackage2Part2,
    BootConfigAndPackage2Part3,
    BootConfigAndPackage2Part4,
    BootConfigAndPackage2Part5,
    BootConfigAndPackage2Part6,
    CalibrationBinary,
    CalibrationFile,
    SafeMode,
    User,
    System,
    SystemProperEncryption,     //! 固有鍵がかかった system パーティションを常に指定するための id
    SystemProperPartition,      //! safe モード用 fs で本来の system パーティションを指定するための id

    SignedSystemPartitionOnSafeMode, //! for internal

    Invalid,
};

/**
*   @brief      MountBis でマウントしたファイルシステムを利用するためのマウント名を取得します。
*/
const char* GetBisMountName(BisPartitionId id);

/**
*   @brief      Built-In-Storage の指定した領域を扱うファイルシステムをマウントします。
*
*   @details    関数が成功した場合、GetBisMountName(id) で取得したマウント名でファイルシステムにアクセスできます。
*               rootPath にはマウントするファイルシステムに実在するパスを指定する必要があります。
*               rootPath に null を指定すると、ファイルシステムの本来のルートが利用されます。
*
*   @param[in]  id         パーティション識別子
*   @param[in]  rootPath   ルートとして利用するファイルシステムのディレクトリパス
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess               成功しました。
*   @retval     ResultPathNotFound          マウントするファイルシステムに rootPath で指定したパスが存在しません。
*   @retval     ResultPartitionNotFound     利用可能なパーティションが見つかりません。
*
*   @pre
*       - rootPath が @link entry_path 有効なパス @endlink である。
*/
Result MountBis(BisPartitionId id, const char* rootPath) NN_NOEXCEPT;

/**
*   @brief      Built-In-Storage の指定した領域を扱うファイルシステムをマウントします。
*
*   @param[in]  name       マウント名
*   @param[in]  id         パーティション識別子
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess               成功しました。
*   @retval     ResultPartitionNotFound     利用可能なパーティションが見つかりません。
*
*   @pre
*       - name が @link mount_name 有効なマウント名 @endlink である。
*/
Result MountBis(const char* name, BisPartitionId id) NN_NOEXCEPT;

/**
*   @brief      Built-In-Storage として利用するホスト PC 上のパスを設定します。
*
*   @param[in]  id         パーティション識別子
*   @param[in]  rootPath   ファイルシステムのルートとする、ホスト PC 上のディレクトリパス
*
*   @return     なし
*
*   @pre
*       - rootPath が @link entry_path 有効なパス @endlink である。
*
*   @details    指定した Built-In-Storage の領域にホスト PC 上のパスを設定します。
*               この API は Built-In-Storage が Windows 上のファイルシステムを指している場合にのみ有効であり、
*               将来的には削除される予定です。この API を実行後、 MountBis() を実行することで、設定したルートパスが
*               マウントされます。 MountBis() 実行後にこの API を実行しても設定は反映されません。
*               設定の反映のためには再度マウントする必要があります。
*/
void SetBisRootForHost(BisPartitionId id, const char* rootPath) NN_NOEXCEPT;

//--------------------------------------------------------------------------

/**
*   @brief      Built-In-Storage の指定した領域のインターフェイスを取得します。
*
*   @param[out] outValue   取得されたインターフェイス
*   @param[in]  id         パーティション識別子
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess               成功しました。
*   @retval     ResultPartitionNotFound     利用可能なパーティションが見つかりません。
*
*   @pre
*       - outValue が有効なメモリを指している。
*/
Result OpenBisPartition(std::unique_ptr<nn::fs::IStorage>* outValue, BisPartitionId id) NN_NOEXCEPT;

/**
*   @brief      Built-In-Storage のパーティション情報キャッシュを無効にします。
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess                   成功しました。
*/
Result InvalidateBisCache() NN_NOEXCEPT;

//! @}

}} // namespace nn::fs
