﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>

#include <nn/os/os_LightEvent.h>
#include <nn/os/os_Mutex.h>
#include <nn/os/os_ReaderWriterLock.h>

#include <nn/fs/fs_IStorage.h>
#include <nn/fs/detail/fs_BufferRegion.h>
#include <nn/fs/detail/fs_IFileDataCache.h>
#include <nn/fs/detail/fs_LruList.h>


namespace nn { namespace fs { namespace detail {

class LruFileDataCacheSystem
{
private:
    static const int PageSizeBit = 14;

public:
    static const size_t PageSize = 1 << PageSizeBit;

    // サイズを大きくする場合はアプリに副作用が起こらないかどうか注意すること
    static const size_t MaxReadAheadSize = 64 * 1024;

private:
    struct Key
    {
        IStorage* pStorage;
        int64_t offset;

        Key() NN_NOEXCEPT
            : pStorage(nullptr)
            , offset(-1)
        {
        }
        Key(IStorage* pStorage, int64_t offset) NN_NOEXCEPT
            : pStorage(pStorage)
            , offset(offset)
        {
        }
        int64_t GetKey() const NN_NOEXCEPT
        {
            // TORIAEZU: offset だけでキーを作る
            // FIXME: サイズの小さい IStorage がたくさん登録される状況だとキーの偏りが起こるので何とかしないといけない
            NN_STATIC_ASSERT(PageSize == (1 << PageSizeBit));
            NN_SDK_ASSERT(nn::util::is_aligned(offset, PageSize));
            return offset >> PageSizeBit;
        }
        bool operator==(const Key& rhs) const NN_NOEXCEPT
        {
            return pStorage == rhs.pStorage && offset == rhs.offset;
        }
        bool operator!=(const Key& rhs) const NN_NOEXCEPT
        {
            return !(*this == rhs);
        }
    };
    struct Page
    {
        /*
         *    m_pCache[] = | c0 | c1 | c2 | c3 |...
         *    m_pPages[] = | p0 | p1 | p2 | p3 |...
         *
         *    上のようなメモリ配置になっていて、例えばページ p3 が変数 address でキャッシュ c1 を指しているとする。この時、
         *
         *       - キャッシュ c1 と同じインデックスの位置にあるページ p1 から p3 へのポインタを reverse pointer と呼ぶことにする
         *         reverse pointer は c1 が分かっている時に c1 を指しているページ (p3) の割り出しに使用する
         *
         *       - 逆方向のポインタ (p3 から p1 へのポインタ) を forward pointer と呼ぶことにする
         *         これは変数としては保存しない。p3 の address が c1 を指しているので簡単に割り出せる
         */
        void* address;
        Page* reversePointer;
    };
    typedef LruList<Key, Page> LruListType;

    class PageRegion
        : public util::IntrusiveListBaseNode<PageRegion>
    {
    public:
        int index;
        int count;
        int GetEndIndex() const { return index + count; }
    };
    typedef util::IntrusiveList<PageRegion, util::IntrusiveListBaseNodeTraits<PageRegion>> PageRegionList;

    class LruFileDataCacheSystemAccessResult;

private:
    void* m_pCache;
    Page* m_pPages;
    LruList<Key, Page> m_LruList;

    int m_TotalPageCount;
    int m_FreePageCount;
    int m_ReservedPageCount;

    PageRegion* m_pPageRegions;
    PageRegionList m_UnusedPageRegionList;
    PageRegionList m_FreePageRegionList;
    PageRegionList m_ReservedPageRegionList;
    os::LightEvent m_CommitEvent;

    int64_t m_LastReadOffset;
    size_t m_ReadAheadSize;
    os::Mutex m_ReadAheadLock;

    os::Mutex m_LruListLock;
    os::ReaderWriterLock m_CacheOperationLock;

public:
    LruFileDataCacheSystem() NN_NOEXCEPT;
    ~LruFileDataCacheSystem() NN_NOEXCEPT;

    Result Initialize(void* pBuffer, size_t bufferSize) NN_NOEXCEPT;
    void Finalize() NN_NOEXCEPT;

    Result Read(
        IStorage* pStorage,
        int64_t offset,
        void* pBuffer,
        size_t size,
        int64_t offsetBase,
        int64_t offsetLimit,
        FileDataCacheAccessResult* pAccessResult) NN_NOEXCEPT;

    void Purge(IStorage* pStorage) NN_NOEXCEPT;

    size_t GetCacheSize() const NN_NOEXCEPT { return m_LruList.CountTotal() * PageSize; }

private:
    static Result CalculateAvailableCount(size_t* pOutPageCount, size_t* pOutBucketCount, size_t bufferSize) NN_NOEXCEPT;

    void DoCopy(
        IStorage* pStorage,
        void* pBuffer,
        const BufferRegion& copyRegion,
        const BufferRegion& originalRequest,
        LruFileDataCacheSystemAccessResult* pAccessResult) NN_NOEXCEPT;

    Result DoCache(
        IStorage* pStorage,
        void* pBuffer,
        const BufferRegion& cacheRegion,
        const BufferRegion& originalRequest,
        LruFileDataCacheSystemAccessResult* pAccessResult) NN_NOEXCEPT;

    Result DoCacheForce(
        IStorage* pStorage,
        void* pBuffer,
        const BufferRegion& cacheRegion,
        const BufferRegion& originalRequest,
        LruFileDataCacheSystemAccessResult* pAccessResult) NN_NOEXCEPT;

    void* ReserveFreePages(IStorage* pStorage, int64_t offset, size_t size) NN_NOEXCEPT;
    void* ReserveFreePagesBySinglePageReservation(IStorage* pStorage, int64_t offset, size_t size) NN_NOEXCEPT;
    void* ReserveFreePagesByGatherCompaction(IStorage* pStorage, int64_t offset, size_t size) NN_NOEXCEPT;
    void CommitCachePages(IStorage* pStorage, void* pBuffer, int64_t offset, size_t size) NN_NOEXCEPT;
    void ReturnBackReservedPages(void* pBuffer, size_t size) NN_NOEXCEPT;

    void GetMaxAllocatableConsecutivePageRegion(int* pOutIndex, int* pOutCount) const NN_NOEXCEPT;
    int GetMaxAllocatableConsecutivePageCount() const NN_NOEXCEPT;

    void AddPageRegionToSortedList(PageRegion& pageRegion, PageRegionList& pageRegionList) NN_NOEXCEPT;
    void AddFreePage(Page* pPage) NN_NOEXCEPT;

    PageRegion* DoCompaction() NN_NOEXCEPT;
    PageRegion* DoCompactionCore(PageRegion* pTargetRegion, int leftLimit, int rightLimit) NN_NOEXCEPT;

    void MovePage(Page* pDestination, Page* pSource) NN_NOEXCEPT;

    bool ShouldPerformCoalescedRead(const BufferRegion& cacheRegion1, const BufferRegion& cacheRegion2) NN_NOEXCEPT;

    void* GetCacheOfIndex(int index) const NN_NOEXCEPT;
    Page* GetPageOfIndex(int index) const NN_NOEXCEPT;
    int GetIndexOfCache(void* pCache) const NN_NOEXCEPT;
    int GetIndexOfPage(Page* pPage) const NN_NOEXCEPT;

    Page* GetForwardPointer(Page* pPage) const NN_NOEXCEPT;
};

}}}  // namespace nn::fs::detail
